;;; pg-tests.lisp -- incomplete test suite
;;;
;;; Author: Eric Marsden <emarsden@laas.fr>
;;
;;
;; These tests assume that a table named "test" is defined in the
;; system catalog, and that the user identified in
;; CALL-WITH-TEST-CONNECTION has the rights to access that table.

(defpackage :pg-tests
  (:use :cl
        :pg
        #+cmu :fwrappers))
(in-package :pg-tests)

(defmacro with-pg-connection/2 ((con &rest open-args) &body body)
  `(let ((,con (pg::pg-connect/v2 ,@open-args)))
     (unwind-protect
         (progn ,@body)
       (when ,con (pg-disconnect ,con)))))

;; !!! CHANGE THE VALUES HERE !!!
(defun call-with-test-connection (function)
  (with-pg-connection (conn "test" "pgdotlisp")
    (funcall function conn)))

(defmacro with-test-connection ((conn) &body body)
  `(call-with-test-connection
     (lambda (,conn) ,@body)))

(defun test-insert ()
  (format *debug-io* "Testing INSERT & SELECT on integers ...~%")
  (with-test-connection (conn)
    (let ((res nil)
          (count 0)
          (created nil))
      (unwind-protect
           (progn
             (pg-exec conn "CREATE TABLE count_test(key int, val int)")
             (loop :for i :from 1 :to 100
                   :for sql = (format nil "INSERT INTO count_test VALUES(~s, ~s)"
                                      i (* i i))
                   :do (pg-exec conn sql))
             (setq created t)
             (setq res (pg-exec conn "SELECT count(val) FROM count_test"))
             (assert (eql 100 (first (pg-result res :tuple 0))))
             (setq res (pg-exec conn "SELECT sum(key) FROM count_test"))
             (assert (eql 5050 (first (pg-result res :tuple 0))))
             ;; this iterator does the equivalent of the sum(key) SQL statement
             ;; above, but on the client side.
             (pg-for-each conn "SELECT key FROM count_test"
                          (lambda (tuple) (incf count (first tuple))))
             (assert (= 5050 count)))
        (when created
          (pg-exec conn "DROP TABLE count_test"))))))

(defun test-insert/float ()
  (format *debug-io* "Testing INSERT & SELECT on floats ...~%")
  (with-test-connection (conn)
    (let ((res nil)
          (sum 0.0)
          (created nil))
      (flet ((float-eql (a b)
               (< (/ (abs (- a b)) b) 1e-5)))
        (unwind-protect
             (progn
               (pg-exec conn "CREATE TABLE count_test_float(key int, val float)")
               (setq created t)
               (loop :for i :from 1 :to 1000
                     :for sql = (format nil "INSERT INTO count_test_float VALUES(~d, ~f)"
                                        i i)
                     :do (pg-exec conn sql))
               (setq res (pg-exec conn "SELECT count(val) FROM count_test_float"))
               (assert (eql 1000 (first (pg-result res :tuple 0))))
               (setq res (pg-exec conn "SELECT sum(key) FROM count_test_float"))
               (assert (float-eql 500500.0 (first (pg-result res :tuple 0))))
               ;; this iterator does the equivalent of the sum(key) SQL statement
               ;; above, but on the client side.
               (pg-for-each conn "SELECT val FROM count_test_float"
                            (lambda (tuple) (incf sum (first tuple))))
               (assert (float-eql 500500 sum)))
          (when created
            (pg-exec conn "DROP TABLE count_test_float")))))))

(defun test-insert/numeric ()
  (format *debug-io* "Testing INSERT & SELECT on NUMERIC ...~%")
  (with-test-connection (conn)
    (let ((res nil)
          (sum 0)
          (created nil))
      (unwind-protect
           (progn
             (pg-exec conn "CREATE TABLE count_test_numeric(key int, val numeric(10,2))")
             (setq created t)
             (loop :for i :from 1 :to 1000
                   :for sql = (format nil "INSERT INTO count_test_numeric VALUES(~d, ~f)"
                                      i i)
                   :do (pg-exec conn sql))
             (setq res (pg-exec conn "SELECT count(val) FROM count_test_numeric"))
             (assert (eql 1000 (first (pg-result res :tuple 0))))
             (setq res (pg-exec conn "SELECT sum(key) FROM count_test_numeric"))
             (assert (eql 500500 (first (pg-result res :tuple 0))))
             ;; this iterator does the equivalent of the sum(key) SQL statement
             ;; above, but on the client side.
             (pg-for-each conn "SELECT val FROM count_test_numeric"
                          (lambda (tuple) (incf sum (first tuple))))
             (assert (eql 500500 sum)))
        (when created
          (pg-exec conn "DROP TABLE count_test_numeric"))))))

(defun test-date ()
  (format *debug-io* "Testing DATE and TIMESTAMP parsing ...~%")
  (with-test-connection (conn)
    (let ((created nil))
      (unwind-protect
           (progn
             (pg-exec conn "CREATE TABLE pgltest (a timestamp, b abstime, c time, d date)")
             (setq created t)
             (pg-exec conn "INSERT INTO pgltest VALUES "
                      "(current_timestamp, 'now', 'now', 'now')")
             (let* ((res (pg-exec conn "SELECT * FROM pgltest"))
                    (parsed (first (pg-result res :tuples))))
               (format t "attributes ~a~%" (pg-result res :attributes))
               (format t "Timestamp = ~s~%abstime = ~s~%time = ~s (CL universal-time = ~d)~%date = ~s~%"
                       (first parsed)
                       (second parsed)
                       (third parsed)
                       (get-universal-time)
                       (fourth parsed))))
        (when created
          (pg-exec conn "DROP TABLE pgltest"))))))

(defun test-booleans ()
  (format *debug-io* "Testing support for BOOLEAN type ...~%")
  (with-test-connection (conn)
    (let ((created nil))
      (unwind-protect
           (progn
             (pg-exec conn "CREATE TABLE pgbooltest (a BOOLEAN, b INT4)")
             (setq created t)
             (pg-exec conn "INSERT INTO pgbooltest VALUES ('t', 42)")
             (dotimes (i 100)
               (pg-exec conn (format nil "INSERT INTO pgbooltest VALUES ('f', ~D)" i)))
             (let ((sum 0))
               (pg-for-each conn "SELECT * FROM pgbooltest"
                            (lambda (tuple) (when (first tuple) (incf sum (second tuple)))))
               (assert (eql 42 sum))))
        (when created
          (pg-exec conn "DROP TABLE pgbooltest"))))))

(defun test-integrity ()
  (format *debug-io* "Testing integrity constaint signaling ...~%")
  (with-test-connection (conn)
    (let ((created nil))
      (unwind-protect
           (progn
             (pg-exec conn "CREATE TABLE pgintegritycheck (a INTEGER UNIQUE)")
             (setq created t)
             (dotimes (i 100)
               (pg-exec conn (format nil "INSERT INTO pgintegritycheck VALUES (~D)" i)))
             (handler-case (pg-exec conn "INSERT INTO pgintegritycheck VALUES (1)")
               (pg:backend-error (exc)
                 (format *debug-io* "OK: integrity constraint handled: ~A~%" exc))
               (error (exc)
                 (format *debug-io* "FAIL: unhandled integrity constraint: ~A~%" exc))))
        (when created
          (pg-exec conn "DROP TABLE pgintegritycheck"))))))

(defun test-introspection ()
  (format *debug-io* "Testing support for introspection ...~%")
  (with-test-connection (conn)
    (dotimes (i 500)
      (pg-tables conn))))


;; Fibonnaci numbers with memoization via a database table
(defun fib (n)
  (declare (type integer n))
  (if (< n 2) 1 (+ (fib (- n 1)) (fib (- n 2)))))

;; (compile 'fib)

#+cmu
(define-fwrapper memoize-fib (n)
  (let* ((conn (fwrapper-user-data fwrapper))
         (res (pg-exec conn (format nil "SELECT fibn FROM fib WHERE n = ~d" n)))
         (tuples (pg-result res :tuples)))
    (cond ((zerop (length tuples))
           (let ((fibn (call-next-function)))
             (pg-exec conn (format nil "INSERT INTO fib VALUES (~D, ~D)" n fibn))
             fibn))
          ((eql 1 (length tuples))
           (caar tuples))
          (t
           (error "integrity error in fibn table")))))

(defun test-fib ()
  (format *debug-io* "Testing fibonnaci number generation ...~%3")
  (with-test-connection (conn)
    (let ((created nil)
          (non-memoized 0)
          (memoized 0))
      (unwind-protect
           (progn
             (pg-exec conn "CREATE TABLE fib (n INTEGER, fibn INT8)")
             (setq created t)
             (funwrap 'fib)
             (time (setq non-memoized (fib 40)))
             #+cmu (fwrap 'fib #'memoize-fib :user-data conn)
             (update-fwrappers 'fib)    ; remove stale conn user-data object
             (time (setq memoized (fib 40)))
             (format t "~S" (pg-exec conn "SELECT COUNT(n) FROM fib"))
             (assert (eql non-memoized memoized)))
        (when created
          (pg-exec conn "DROP TABLE fib"))))))


(defun test-lo ()
  (format *debug-io* "Testing large object support ...~%")
  (with-test-connection (conn)
   (with-pg-transaction conn
    (let* ((oid (pglo-create conn))
           (fd (pglo-open conn oid)))
      (sleep 1)
      (pglo-tell conn fd)
      (sleep 1)
      (pglo-unlink conn oid)))))

;; test of large-object interface. We are careful to use vectors of
;; bytes instead of strings, because with the v3 protocol strings
;; undergo \\xxx encoding (for instance #\newline is transformed to \\012). 
(defun test-lo-read ()
  (format *debug-io* "Testing read of large object ...~%")
  (with-test-connection (conn)
   (with-pg-transaction conn
    (let* ((oid (pglo-create conn "rw"))
           (fd (pglo-open conn oid "rw")))
      (pglo-write conn fd (map '(vector (unsigned-byte 8)) #'char-code (format nil "Hi there mate~%What's up?~%")))
      (pglo-lseek conn fd 3 0)           ; SEEK_SET = 0
      (assert (eql 3 (pglo-tell conn fd)))
      ;; this should print "there mate"
      (format *debug-io* "Read ~s from lo~%" (map 'string #'code-char (pglo-read conn fd 10)))
      (format *debug-io* "Rest is ~s~%" (map 'string #'code-char (pglo-read conn fd 1024)))
      (pglo-close conn fd)
      #+nil (pglo-unlink conn oid)))))

#+cmu
(defun test-lo-import ()
  (format *debug-io* "Testing import of large object ...~%")
  (with-test-connection (conn)
   (with-pg-transaction conn
    (let ((oid (pglo-import conn "/etc/group")))
      (pglo-export conn oid "/tmp/group")
      (cond ((zerop
              (ext:process-exit-code
               (ext:run-program "diff" (list "/tmp/group" "/etc/group"))))
             (format *debug-io* "pglo-import test succeeded~%")
             (unix:unix-unlink "/tmp/group"))
            (t
             (format *debug-io* "pglo-import test failed: check differences
between files /etc/group and /tmp/group")))
      (pglo-unlink conn oid)))))

(defun test-simple ()
  (let ((*pg-disable-type-coercion* t))
    (with-test-connection (conn)
     (format t "backend ~a~%" (pg-backend-version conn)))))

(defun test-notifications ()
  (with-test-connection (conn)
    (let (res)
      (setq res (pg-exec conn "LISTEN pg_test_listen"))
      (format t "LISTEN -> ~S~%" (pg-result res :status))
      (assert (null (pg::pgcon-notices conn)))
      (pg-exec conn "SELECT * FROM pg_type")
      (assert (null (pg::pgcon-notices conn)))
      (setq res (pg-exec conn "NOTIFY pg_test_listen"))
      (format t "NOTIFY -> ~S~%" (pg-result res :status))
      (format t "In TEST-NOTIFICATIONS notices are ~S~%"
              (pg::pgcon-notices conn)))))


;; FIXME could add interaction between producer and consumers via NOTIFY

#+(and cmu mp)
(defun test-multiprocess ()
  (format *debug-io* "Testing multiprocess database access~%")
  (when (eq mp::*current-process* mp::*initial-process*)
    (mp::startup-idle-and-top-level-loops))
  (with-test-connection (conn)
    (pg-exec conn "CREATE TABLE pgmt (a TEXT, b INTEGER, C FLOAT)"))
  (flet ((producer ()
           (with-test-connection (conn)
             (dotimes (i 5000)
               (pg-exec conn (format nil "INSERT INTO pgmt VALUES (~S, ~D, ~F)" i i i))
               (when (zerop (mod i 100))
                 (pg-exec conn "COMMIT WORK")))))
         (consumer ()
           (with-test-connection (conn)
             (dotimes (i 10)
               (sleep 1)
               (let ((res (pg-exec conn "SELECT count(*) FROM pgmt")))
                 (format *debug-io* "  Consumer sees ~D rows~%"
                         (first (pg-result res :tuple 0))))))))
    (let ((p1 (mp:make-process #'producer :name "PG data producer"))
          (p2 (mp:make-process #'producer :name "PG data producer"))
          (p3 (mp:make-process #'producer :name "PG data producer"))
          (co (mp:make-process #'consumer :name "PG data consumer")))
      (loop :while (some 'mp:process-alive-p (list p1 p2 p3 co))
            :do (sleep 5) (mp:show-processes t))))
  (with-test-connection (conn)
    (pg-exec conn "DROP TABLE pgmt")))


(defun test-pbe ()
  (with-test-connection (conn)
    (when (pg-supports-pbe conn)
      (format *debug-io* "~&Testing pbe...")
      (let ((res nil)
            (count 0)
            (created nil))
        (unwind-protect
             (progn
               (pg-exec conn "CREATE TABLE count_test(key int, val int)")
               (setq created t)
               (format *debug-io* "~&table created")
               (pg-prepare conn "ct_insert"
                           "INSERT INTO count_test VALUES ($1, $2)"
                           '("int4" "int4"))
               (loop :for i :from 1 :to 100
                     :do
                     (pg-bind conn
                              "ct_portal" "ct_insert"
                              `((:int32 ,i)
                                (:int32 ,(* i i))))
                     (pg-execute conn "ct_portal")
                     (pg-close-portal conn "ct_portal"))
               (format *debug-io* "~&data inserted")
               (setq res (pg-exec conn "SELECT count(val) FROM count_test"))
               (assert (eql 100 (first (pg-result res :tuple 0))))
               (setq res (pg-exec conn "SELECT sum(key) FROM count_test"))
               (assert (eql 5050 (first (pg-result res :tuple 0))))
               ;; this iterator does the equivalent of the sum(key) SQL statement
               ;; above, but on the client side.
               (pg-for-each conn "SELECT key FROM count_test"
                            (lambda (tuple) (incf count (first tuple))))
               (assert (= 5050 count)))
          (when created
            (pg-exec conn "DROP TABLE count_test")))))))

(defun test-copy-in-out ()
  (with-test-connection (conn)
    (ignore-errors
      (pg-exec conn "DROP TABLE foo"))
    (pg-exec conn "CREATE TABLE foo (a int, b int)")
    (pg-exec conn "INSERT INTO foo VALUES (1, 2)")
    (pg-exec conn "INSERT INTO foo VALUES (2, 4)")
    
    (with-open-file (stream "/tmp/foo-out"
			    :direction :output
			    :element-type '(unsigned-byte 8)
			    :if-does-not-exist :create
			    :if-exists :overwrite)
      (setf (pgcon-sql-stream conn) stream)
      (pg-exec conn "COPY foo TO stdout"))

    (pg-exec conn "DELETE FROM foo")
    (with-open-file (stream "/tmp/foo-out"
			    :direction :input
			    :element-type '(unsigned-byte 8)
			    :if-does-not-exist :error
			    :if-exists :overwrite)
      (setf (pgcon-sql-stream conn) stream)
      (pg-exec conn "COPY foo FROM stdout"))

    (let ((res (pg-exec conn "SELECT b FROM foo WHERE a = 1")))
      (assert (eql 2 (first (pg-result res :tuple 0)))))
    (let ((res (pg-exec conn "SELECT b FROM foo WHERE a = 2")))
      (assert (eql 4 (first (pg-result res :tuple 0)))))

    (pg-exec conn "DROP TABLE foo")))

(defun test ()
  (let (#+nil (*pg-client-encoding* "UNICODE"))
    (with-test-connection (conn)
      (format t "Running pg.lisp tests against backend ~a~%" (pg-backend-version conn))
      ;; client encoding supported since PostgreSQL v7.1
      (format t "Client encoding is ~A~%" (pg-client-encoding conn))
      (format t "Date style is ~A~%" (pg-date-style conn))
      (let ((r2 (pg-exec conn "CREATE TABLE pgltest (a int, b float, c money)"))
            (r3 (pg-exec conn "INSERT INTO pgltest VALUES (3, -1234.5e67, '$123.45')"))
            (r4 (pg-exec conn "DROP TABLE pgltest")))
        (format t "~%==============================================~%")
        (format t "status of CREATE is ~s~%" (pg-result r2 :status))
        (format t "status of INSERT is ~s~%" (pg-result r3 :status))
        (format t "oid of INSERT is ~s~%" (pg-result r3 :oid))
        (format t "status of DROP is ~s~%" (pg-result r4 :status))
        (format t "==============================================~%")))
    (test-simple)
    (test-insert)
    (test-insert/float)
    (test-insert/numeric)
    (test-date)
    (test-booleans)
    (test-integrity)
    (test-notifications)
    (test-lo)
    (test-lo-read)
    #+cmu (test-lo-import)
    (test-pbe)))


;; EOF
