;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: imho -*-
;;; $Id: methods.lisp,v 1.26 2001/11/12 20:07:08 jesse Exp $
;;;
;;; Copyright (c) 1999, 2000, 2001 onShore Development, Inc.
;;;
;;; See the file 'COPYING' in this directory for terms.
;;;
;;; - method definition
;;; - method lookup and dispatch
;;; - argument internalizing / externalizing

(in-package :imho)

;; ------------------------------------------------------------
;; Add methods to the system

(eval-when (:load-toplevel :compile-toplevel)
(defstruct wmethod
  name
  type
  args
  body)


(defclass target ()
  ((yield-target
    :accessor yield-target
    :initform nil))) ; greedy by default

;; target class: mixin
;; elements inheriting from target should also inherit
;; from something else which specializes element-target (like html-element)
(defmethod element-target ((self target))
  "return self unless explicitly forbidden"
  (if (yield-target self)
      (call-next-method)
      self))

(defmethod element-method (sth)
  (declare (ignore sth))
  nil)

;; targeted class: mixin
;; elements inheriting from targeted should also inherit
;; from something else which specializes element-target
;; and element-method (like html-element)
(defclass targeted ()
  ((element-method
    :accessor element-method
    :initarg :method
    :initform nil)
   (frame-args
    :reader element-frame-args
    :initarg :frame-args
    :initform nil)
   (wm-target
    :accessor wm-target
    :initarg :target
    :initform nil)
   (wm-args
    :accessor wm-args
    :initarg :args
    :initform nil)))

(defmethod set-target ((element targeted) target method)
  (setf (slot-value element 'wm-target)
        target
        (slot-value element 'element-method)
        method))

(defmethod element-target ((targeted targeted))
  "look elsewhere unless specifically given a target"
  (or (wm-target targeted)
      (call-next-method)))

(defmethod element-args ((targeted targeted))
  (wm-args targeted))

(defgeneric intern-ref (t t))

(defmethod intern-ref (object arg)
  (declare (ignore arg))
  (error "No default internalizer for objects of type '~s'" object))

(defun process-wm-args (method-args request-args)
  (flet ((process (method-arg ext-arg)
	   (case method-arg
	     ('string
	      ext-arg)
	     ('integer
	      (parse-integer ext-arg))
	     ('symbol
	      (intern (string-upcase ext-arg)))
	     ('keyword
	      (intern (string-upcase ext-arg) :keyword))
	     (t
	      (intern-ref method-arg ext-arg)))))
    (mapcar #'process method-args request-args)))

(defvar *methods* (make-hash-table :test #'equal))

(defun list-wms ()
  "list all defined web methods"
  (maphash (lambda (k v)
             (declare (ignore k))
             (describe v))
           *methods*))

(defun lookup-wm (name &optional default)
  "look up a WM by name"
  (let ((method-name (etypecase name
                       (symbol (string-downcase (symbol-name name)))
                       (string (string-downcase name)))))
  (or (gethash method-name *methods*)
      (and default (gethash default *methods*)))))

(defmacro invoke-wm (name &rest args)
  `(let ((method (lookup-wm ,name)))
     (apply (wmethod-body method) (or ',args (list t)))))

;; like invoke-wm but takes a lisp list of args
(defmacro apply-wm (name &optional args)
  `(let ((method (lookup-wm ,name)))
     (apply (wmethod-body method) (or ,args (list t)))))

(defmacro undefine-wm (method)
  `(setf (gethash (string-downcase (symbol-name ,method)) *methods*) nil))

(defmacro refer-wm (method &rest args)
  `(let ((method (lookup-wm (string-downcase (symbol-name ',method))))
           (argv (list ,@args)))
       (if (not method)
           (error "No reference for method ~s" ',method))
       (let ((name (wmethod-name method)))
         (if argv
             (progn
               (setf name (concatenate 'string name "?"))
               (mapc (lambda (arg)
                       (setf name (concatenate 'string name (extern-ref arg)))
                       )
                     argv)))
         name)))

(defun dyn-refer-wm (method-name &rest parameters)
  "construct a reference to the WM named by METHOD-NAME"
  (when (consp method-name)
    (setq method-name (car method-name)))
  (let ((method (lookup-wm (string-downcase (symbol-name method-name)))))
    (if (not method)
        (error "No reference for method ~s" method-name))
    (let ((name (wmethod-name method)))
      (if parameters
	  (setf name
		(format nil "~A?~{~A~^& ~}"
			name (mapcar #'extern-ref parameters))))
      name)))

;; A string externalizes as itself, mutatis mutandis URI escaping.

(defmethod extern-ref ((string string))
  string)

(defmethod extern-ref ((int integer))
  (format nil "~d" int))

#|
(defmacro define-wm (method-name method-ll &body method-body)
  `(labels ((wm-args (arglist)
	     (values (mapcar #'car arglist)
		     (mapcar #'cadr arglist)))
	    ;; Should we install new objects into the session here?
	    (wm-lambda (args &rest body)
	     (compile nil (coerce `(lambda ,args (declare (ignorable ,(car args))) ,@body) 'function))))
    (destructuring-bind (type &rest args)
	',method-ll
      (let ((name (string-downcase (symbol-name ',method-name))))
	(multiple-value-bind (arg-vars arg-types)
	    (wm-args args)
	  (let ((body-func (wm-lambda (cons (car type) arg-vars) '(progn ,@method-body))))
	    (setf (gethash name *methods*)
		  (make-wmethod :name name :type type
				:args arg-types :body body-func))))))))
|#


;;; Craig put this here, it's the next-gen webmethod once
;;; I get some stuff ironed out.

(defun %method-ll-from-wmethod-ll (ll)
  (cond
   ((= 1 (length ll))
    ll)
   (t
    (cons (car ll) (mapcar #'car (cdr ll))))))

(defun %wmethod-internal-name-from-symbol (method-name)
   (intern (string-downcase
		 (concatenate 'string
			      "wmethod-internal-"
			      (package-name
			       (symbol-package method-name))
			      "-"
			      (symbol-name method-name)))
	   "IMHO"))
			     
(defmacro define-wm (method-name method-ll &body method-body)
  (let ((mname (%wmethod-internal-name-from-symbol method-name))
	(mll (%method-ll-from-wmethod-ll method-ll))
	(namesym (gensym "name"))
	(typesym (gensym "type"))
	(argssym (gensym "args"))
	(wm-argssym (gensym "wm-args"))
	(arglistsym (gensym "arglist"))
	(wmethod-docstring
	 (format nil "Webmethod ~A body method, don't peek." `,method-name)))
  `(labels ((,wm-argssym (,arglistsym)
			 (mapcar #'cadr ,arglistsym)))
     (destructuring-bind (,typesym &rest ,argssym)
	 ',method-ll
       (let* ((,namesym (string-downcase (symbol-name ',method-name))))
	 (defmethod ,mname ,mll
	   ,wmethod-docstring
	   ,@method-body)
         ;; I killed this line because I like redefining wm's on the fly.
         ;; put it back in if you should.   [2000/08/03 :lh]
	 ;; (unless (gethash ,namesym *methods*)
	   (setf (gethash ,namesym *methods*)
		 (make-wmethod :name ,namesym :type ,typesym
			       :args (,wm-argssym ,argssym)
			       :body ',mname)))))))

)



