;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: imho -*-
;; $Id: cmucl-patches.lisp,v 1.2 2001/01/23 17:05:07 jesse Exp $

;;; **********************************************************************
;;; This code was written as part of the CMU Common Lisp project at
;;; Carnegie Mellon University, and has been placed in the public domain.

(in-package :lisp)

;; Values for errno notes in Linux write(2):
;;
;; EBADF  fd is not a valid file descriptor or  is  not  open
;;        for writing.
;;
;; EINVAL fd is attached to an object which is unsuitable for
;;        writing.
;;
;; EFAULT buf is outside your accessible address space.
;;
;; EPIPE  fd is connected to a pipe or socket  whose  reading
;;        end  is closed.  When this happens the writing pro
;;        cess will receive a SIGPIPE signal; if it  catches,
;;        blocks or ignores this the error EPIPE is returned.
;;
;; EAGAIN Non-blocking I/O has been selected using O_NONBLOCK
;;        and  there  was  no room in the pipe or socket con
;;        nected to fd to write the data immediately.
;;
;; EINTR  The call was interrupted by  a  signal  before  any
;;        data was written.
;;
;; ENOSPC The  device  containing  the file referred to by fd
;;        has no room for the data.
;;
;; EIO    A low-level I/O error occurred while modifying  the
;;        inode.


;;; DO-OUTPUT-LATER -- internal
;;;
;;;   Called by the server when we can write to the given file descriptor.
;;; Attemt to write the data again. If it worked, remove the data from the
;;; output-later list. If it didn't work, something is wrong.
;;;

(defun do-output-later (stream)
  (let* ((stuff (pop (fd-stream-output-later stream)))
	 (base (car stuff))
	 (start (cadr stuff))
	 (end (caddr stuff))
	 (reuse-sap (cadddr stuff))
	 (length (- end start)))
    (declare (type index start end length))
    (multiple-value-bind
          (count errno)
	(unix:unix-write (fd-stream-fd stream)
			 base
			 start
			 length)
      (cond ((not count)
                 (case errno
                   ((#.unix:ewouldblock
                     #.unix:eintr)
                    (format t "~&;; ***** Output lost: unblocked, but failed. Oh well.~%"))
                   (#.unix:epipe  (format t "~&;; ***** EPIPE: Output lost.~%"))
                   (#.unix:ebadf  (format t "~&;; ***** EBADF: Output lost.~%"))
                   (#.unix:einval (format t "~&;; ***** EBADF: Output lost.~%"))
                   (#.unix:efault (format t "~&;; ***** EBADF: Output lost.~%"))
                   (#.unix:enospc (format t "~&;; ***** EBADF: Output lost.~%"))
                   (#.unix:eio    (format t "~&;; ***** EBADF: Output lost.~%"))
                   (t             (format t "~&;; ***** ERROR: Output lost: ~d (~a)~%"
                                          errno (unix:get-unix-error-msg errno)))))
	    ((eql count length) ; Hot damn, it workded.
	     (when reuse-sap
	       (push base *available-buffers*)))
	    ((not (null count)) ; Sorta worked.
	     (push (list base
			 (the index (+ start count))
			 end)
		   (fd-stream-output-later stream))))))
  (unless (fd-stream-output-later stream)
    (system:remove-fd-handler (fd-stream-handler stream))
    (setf (fd-stream-handler stream) nil)))


;;; DO-OUTPUT -- internal
;;;
;;;   Output the given noise. Check to see if there are any pending writes. If
;;; so, just queue this one. Otherwise, try to write it. If this would block,
;;; queue it.
;;;
;;; added checking for sigpipe during write: just throw away the
;;; output and return. <jesse@onshore.com>

(defun do-output (stream base start end reuse-sap)
  (declare (type fd-stream stream)
	   (type (or system-area-pointer (simple-array * (*))) base)
	   (type index start end))
  (if (not (null (fd-stream-output-later stream))) ; something buffered.
      (progn
	(output-later stream base start end reuse-sap)
	;; ### check to see if any of this noise can be output
	)
      (let ((length (- end start)))
	(multiple-value-bind (count errno)
	    (unix:unix-write (fd-stream-fd stream) base start length)
	  (cond ((not count)
                 (case errno
                   ((#.unix:ewouldblock
                     #.unix:eintr)
                    (output-later stream base start end reuse-sap))
                   (#.unix:epipe  (format t "~&;; ***** EPIPE: Output lost.~%"))
                   (#.unix:ebadf  (format t "~&;; ***** EBADF: Output lost.~%"))
                   (#.unix:einval (format t "~&;; ***** EBADF: Output lost.~%"))
                   (#.unix:efault (format t "~&;; ***** EBADF: Output lost.~%"))
                   (#.unix:enospc (format t "~&;; ***** EBADF: Output lost.~%"))
                   (#.unix:eio    (format t "~&;; ***** EBADF: Output lost.~%"))
                   (t             (format t "~&;; ***** ERROR: Output lost: ~d (~a)~%"
                                          errno (unix:get-unix-error-msg errno)))))
		((not (eql count length))
		 (output-later stream base (the index (+ start count))
			       end reuse-sap)))))))

