#include "cp_types.h"
#include "cp_proto.h"

/* Routines for drawing spherical objects */

/* Drawing flag scheme: flags indicate drawing instructions
   for objects -- circles/lines/faces. Bits set as follows:
      1  ->  draw object?
      2  ->  fill? (4 and 16 imply 2, also)
      4  ->  off = foreground, on = background 
             (applies to interior, not border, overriden by bit 16)
      8  ->  border color? (off = foreground, on = recorded color)
      16 ->  interior color? (default set by bit 4, on --> recorded color)
      32 ->  display label?

Eg.  flag=3: filled object, in foreground
     flag=9: open object, border in (recorded) color
             (for 'edge', this gives colored edge)
     flag=19: filled in color, border in foreground
     flag=27: filled, border and interior in color
     flag=15: filled with background, border in color
     flag=32: label only (face or circle)

Normally, flag for each type of object; often, when passed on to
subroutine, you may need to pass color code too:
Eg. (cflag, ecol, ccol) for circle flag, border color code, int color 
code. */

int dum_int;

int s_triangle(struct s_data *q,complex p1,complex p2,complex p3,
	       int fflag,int ecol,int fcol,int show)
{
  struct Pathlist *list;

  /* determine if convex from orientation */
	
  if ((list=sph_tri_list(p1,p2,p3,num_plot))==NULL) return 0;
  s_draw_convex(q,list,fflag,ecol,fcol,show);
  return 1;
} /* s_triangle */

int s_polygon(struct p_data *p,struct Vertlist *vertlist,
	      int fflag,int ecol,int fcol,int show)
     /* draws hyp polygon through verts. Return 0 on error. 
Code has limit on length.*/
{
  struct Pathlist *list;

  /* may give problems when not convex or goes behind sphere */

  if ((list=sph_poly_list(p,vertlist))==NULL) return 0;
  s_draw_convex(p->screen,list,fflag,ecol,fcol,show);
  return 1;
} /* s_polygon */

int s_draw_convex(struct s_data *q,struct Pathlist *list,int fflag,
		  int ecol,int col,int show)
     /* draws convex sph tri or circle. fixup? problems managing list: 
should create/destroy list in approp routine, but that seems to crash ?? */
{
  int n=0;
  complex normpt,pt;
  struct Pathlist *trace;
  XPoint *pts;

  if (list==NULL || (list=fix_convex_list(list))==NULL) return 0;
  trace=list;
  while (trace!=NULL)
    {
      n++;
      trace=trace->next;
    }
  pts=(XPoint *)malloc((n+2)*sizeof(XPoint));
  trace=list;
  n=0;
  while (trace!=NULL)
    {
      pt.re=trace->x;pt.im=trace->y;
      r_to_pix(pt,&normpt,q->pix_box,q->box);
      pts[n].x=(short)normpt.re; 
      pts[n].y=(short)normpt.im;
      n++;
      trace=trace->next;
    }
  pts[n].x=pts[0].x;
  pts[n].y=pts[0].y;
  if (fflag & 2)
    {
      if (!(fflag & 16))
	{
	  col=FG_COLOR;
	  if (fflag & 4) col=BG_COLOR;
	}
      FillPolygon(q->xpm,pts,n+1,colors[col]);
    }
  if (!(fflag & 8)) ecol=FG_COLOR;
  DrawLines(q->xpm,pts,n+1,colors[ecol]);
  if (show) refresh_canvas(q);
  path_free(&list);
  free(pts);
  return 1;
} /* s_draw_convex */

struct Pathlist *sph_poly_list(struct p_data *p,struct Vertlist *v_list)
     /* clockw' list of geodesics in polygon. */
{
  int v,w,front;
  complex zv,zw;

  struct Pathlist *firstlist=NULL,*nlist,*trace=NULL;
  struct Vertlist *vtrace;
  struct R_data *pR_ptr;

  pR_ptr=p->packR_ptr;
  if (!(vtrace=v_list) || !(v_list->next)) return NULL;
  w=v_list->v;
  zw=ss_view(p->screen,pR_ptr[w].center,1,&front);
  vtrace=vtrace->next;
  while (vtrace)
    {
      v=w;
      zv=zw;
      while (vtrace && vtrace->v == v) vtrace=vtrace->next;
      /* skip repeats */
      if (vtrace)
	{
	  w=vtrace->v;
	  zw=ss_view(p->screen,pR_ptr[w].center,1,&front);
	  if ((nlist=full_s_geodesic(zv,zw,num_plot))==NULL) 
	    {path_free(&firstlist);return NULL;}
	  if (!firstlist) firstlist=trace=nlist;
	  else trace->next=nlist;
	  while (trace->next) 
	    trace=trace->next;
	  vtrace=vtrace->next;
	}
    }
  return firstlist;
} /* sph_poly_list */

int sgeo(struct s_data *q,complex a,complex b,int col,int show)
     /* draws spherical geodesic on screen q. */
{
  int length=1;
  struct Pathlist *trace,*plist=NULL;
  XPoint *Xptr=NULL;

  if ((plist=s_geodesic(ss_view(q,a,1,&dum_int),
			ss_view(q,b,1,&dum_int),num_plot))==NULL) return 0;
  trace=plist;
  while ((trace->next)!=NULL)
    {trace=trace->next;length++;}
  if (length<2) return 0;
  Xptr=path_XPoints(q,plist,&length);
  if (col!=FG_COLOR)
    DrawLines(q->xpm,Xptr,length,colors[col]);
  else DrawLines(q->xpm,Xptr,length,fgcolor);
  free(Xptr);path_free(&plist);
  if (show) refresh_canvas(q);
  return 1;
} /* sgeo */

int s_circle(struct s_data *q,complex ctr,double r,int cflag,
	     int ccol,int col,int show)
{
  int n=0,i;
  double step,ss,a11,a12,a21,a22,zr,zi,wr,wi,arad;
  complex pt,normpt,antip;
  struct Pathlist *list,*ulist,*trace;
  XPoint *pts;

  if (isnan((double)r)
      || isnan((double)ctr.re) || isnan((double)ctr.im)
      || r>=M_PI || r<=0.0)
    return 0; /* invalid data */
  if (!(cflag & 2) || r<=M_PI_2) /* convex circle */
    {
      list=s_circle_list(ctr,r,num_plot);
      n=s_draw_convex(q,list,cflag,ccol,col,show);
      return n;
    }
  /* non-convex filled circle, draw filled unit circle and filled cap;
     use EvenOddRule. */
  antip.re=M_PI+ctr.re;
  antip.im=M_PI-ctr.im;
  arad=M_PI-r;
  list=fix_convex_list(s_circle_list(antip,arad,num_plot));
  /* form unit circle list */
  ulist=(struct Pathlist *)calloc(1,sizeof(struct Pathlist));
  ulist->x=1.0;ulist->y=0.0;
  trace=ulist;
  step=2.0*M_PI/(double)num_plot;
  ss=step*step/4;
  a11=(1-ss)/(1+ss);a12= (-step)/(1+ss);a21= (-a12);a22=a11; 
  zr=1.0;zi=0.0;
  for (i=1;i<=num_plot;i++)
    {
      trace=trace->next=
	(struct Pathlist *)calloc(1,sizeof(struct Pathlist));
      wr=zr;
      wi=zi;
      zr=a11*wr+a12*wi;
      zi=a21*wr+a22*wi;
      trace->x=zr;trace->y=zi;
    }
  trace->next=list; /* link in cap, if any shows */
  /* now to display combined list */
  trace=ulist;
  while (trace!=NULL)
    {
      n++;
      trace=trace->next;
    }
  pts=(XPoint *)malloc((n+2)*sizeof(XPoint));
  trace=ulist;
  n=0;
  while (trace!=NULL)
    {
      pt.re=trace->x;pt.im=trace->y;
      r_to_pix(pt,&normpt,q->pix_box,q->box);
      pts[n].x=(short)normpt.re; 
      pts[n].y=(short)normpt.im;
      n++;
      trace=trace->next;
    }
  pts[n].x=pts[0].x;
  pts[n].y=pts[0].y;
  FillPolygon(q->xpm,pts,n+1,colors[col]);
  /* redraw bdry of cap only */
  DrawLines(q->xpm,pts+num_plot+1,n-num_plot-1,colors[ccol]);
  if (show) refresh_canvas(q);
  free(pts);
  path_free(&ulist);
  return n;
} /* s_circle */

int equator(struct s_data *q)
/* put equator on screen q if spherical */
{
  int dum_int;
  complex ctr;

  ctr.re=0.0;ctr.im=M_PI;
  return s_circle(q,ss_view(q,ctr,1,&dum_int),M_PI_2,0,FG_COLOR,
		  BG_COLOR,1);
} /* equator */

