/* hyp_math.c   generic hyperbolic geometry routines */

/* gcc hyp_math.c -o hyp_math.o -c -g -lm */

#include "cp_types.h"

/* tmp for debugging, move h_compcenter to separate file */

/*
s_radius is variable of form s=exp(-h) where h is hyperbolic radius.
Value s<=0 indicates infinite hyp radius; however, the stored value is
(generally) the negative of a eucl radius to be used for plotting.

An inversive distance or overlap of theta is stored as cos(theta)
for overlap or cosh(theta) for inversive distance; thus overlaps lie
in [0,1] and inv dist in [1,infty).
*/

int e_to_h_data(complex e_center,double e_rad,
		complex *h_center,double *s_rad)
     /* converts circle data from eucl to hyp. */
{
  double aec,ahc,b,c2,r2,dist;

  aec=cAbs(e_center);
  dist=aec+e_rad;
  if (dist>(1.000000000001)) return 0; /* not in closed disc */
  if ((.999999999999)<dist) /* horocycle */
    {
      *s_rad=(-e_rad);
      h_center->re=e_center.re/aec;
      h_center->im=e_center.im/aec;
      return 1;
    }
  c2=aec*aec;
  r2=e_rad*e_rad;
  if (aec<.0000000000001) /* circle at origin */
    {
      h_center->re=0;
      h_center->im=0;
    }
  else
    {
      b=sqrt((1+2*aec+c2-r2)/(1-2*aec+c2-r2));
      ahc=(b-1)/(b+1);
      b=ahc/aec;
      h_center->re=b*e_center.re;
      h_center->im=b*e_center.im;
    }
  *s_rad=sqrt((1-2*e_rad+r2-c2)/(1+2*e_rad+r2-c2));
  return 1;
} /* e_to_h_data */

int h_to_e_data(complex h_center,double s_rad,
		complex *e_center,double *e_rad)
     /* converts circle data from hyp to eucl. */
{
  double a,b,d,k,aec,ahc,g;

  if (s_rad<=0) /* infinite hyp radius */
    {
      a=1+s_rad;
      e_center->re=h_center.re*a;
      e_center->im=h_center.im*a;
      *e_rad=(-s_rad); /* assumes -s_rad is meaningful */
      return 1;
    }
  ahc=cAbs(h_center);
  if (ahc > .999999999999) /* almost horocycle? avoid error. */
    ahc = .999999999999;
  g=((1+ahc)/(1-ahc));
  a=g/s_rad;
  d=(a-1)/(a+1);
  b=g*s_rad;
  k=(b-1)/(b+1);
  *e_rad = (d-k)*0.5;
  aec=(d+k)*0.5;
  if (ahc<.0000000000001)
    {
      e_center->re=0;
      e_center->im=0;
    }
  else
    {
      b=aec/ahc;
      e_center->re = b*h_center.re;
      e_center->im=b*h_center.im;
    }
  return 1;
} /* h_to_e_data */

int circle_3(complex z1,complex z2,complex z3,complex *c,double *rad)
     /* find eucl center and rad for circle through 3 given points. 
Returns 1 in case of success. */
{
  double det,a1,a2,b1,b2,c1,c2,dum;

  a1=z2.re-z1.re;
  a2=z3.re-z2.re;
  b1=z2.im-z1.im;
  b2=z3.im-z2.im;
  det=2*(a1*b2-b1*a2);
  if (fabs(det)<.0000000000001) return 0;
  dum=cAbs(z2);
  dum=dum*dum;
  c1=cAbs(z1);
  c1=(-c1*c1);
  c1+=dum;
  c2=cAbs(z3);
  c2=c2*c2;
  c2-=dum;
  c->re=(b2*c1-b1*c2)/det;
  c->im=(a1*c2-a2*c1)/det;
  *rad=cAbs(csub(*c,z1));
  return 1;
} /* circle_3 */

double h_comp_cos(double s1,double s2,double s3)
     /* given ordered triple of s_radii, compute the cosine of the 
angle at first circle in triangle formed by mutually tangent circles. */
{
  double s1s,s2s,s3s,ans;

  if (s1<=0) return (1.0);
  s1s=s1*s1; 
  if ((s2<=0) && (s3<=0)) return (1-2*s1s);
  s3s=s3*s3;
  if (s2<=0) return ((1+s1s*(s3s-2))/(1-s1s*s3s));
  s2s=s2*s2;
  if (s3<=0) return ((1+s1s*(s2s-2))/(1-s1s*s2s));
  ans=((1+s1s*s2s)*(1+s1s*s3s)-2*s1s*(1+s2s*s3s))/
    ((1-s1s*s2s)*(1-s1s*s3s));
  if (ans>1) return 1;
  if (ans<-1) return -1;
  return (ans);
} /* h_comp_cos */

double h_area(double s1,double s2,double s3)
     /* given triple of s_radii, compute hyp area of triangle */
{return (M_PI-cos(h_comp_cos(s1,s2,s3))-cos(h_comp_cos(s2,s1,s3))
	 -cos(h_comp_cos(s3,s1,s2)) );}

double h_cos_overlap(double s1,double s2,double s3,
		     double t1,double t2,double t3,int *flag)
     /* given three s_rad, cos's of opp overlap angles, return cos of 
angle at s1. If radii and overlaps incompatible, return value 
(for 0 or Pi, depending) and set flag. Note: tj <= 1 for overlap, 
tj > 1 for inv distance. Note: hyp distance between hyp centers 
of circles j/k with inv dist d is 
arccosh(cosh(rj)*cosh(rk)+sinh(rj)*sinh(rk)*d). 
But possibility of horocycles causes us to convert to eucl computation. */
{
	double h1,h2,h3,e1,e2,e3,L,len,ans;

	if (s1<=0) return (1.0);
	e1=(1.0-s1)/(1.0+s1);
	h1=-log(s1);
	if (s2<=0) L=1.0;
	else 
	 {
		h2=-log(s2);
		len=exp(-h2-acosh(cosh(h1)*cosh(h2)+sinh(h1)*sinh(h2)*t3));
		L=(1.0-len)/(1.0+len);
	 }
	e2=(L*L-e1*e1)/(2.0*e1*t3+2.0*L);
	if (s3<=0) L=1.0;
	else 
	 {
		h3=-log(s3);
		len=exp(-h3-acosh(cosh(h1)*cosh(h3)+sinh(h1)*sinh(h3)*t2));
		L=(1.0-len)/(1.0+len);
	 }
	e3=(L*L-e1*e1)/(2.0*e1*t2+2.0*L);
		/* have euclidean radii now */
	ans=e_cos_overlap(e1,e2,e3,t1,t2,t3,flag); 
		/* find cos from eucl routine */
	return (ans);
} /* h_cos_overlap */

double h_cos_overlap_special(double s1,double s2,double s3,
			     double t1,double t2,double t3,int *flag,
			     double *incompat_err)
/* given three s_rad, cos's of opp overlap angles, return cos 
of angle at s1. If radii and overlaps incompatible, return value 
(for 0 or Pi, depending) and set flag. Note: tj <= 1 for overlap, 
tj > 1 for inv distance. Note: hyp distance between hyp centers of 
circles j/k with inv dist d is 
arccosh(cosh(rj)*cosh(rk)+sinh(rj)*sinh(rk)*d). 
But possibility of horocycles causes us to convert to eucl computation. */
{
	double h1,h2,h3,e1,e2,e3,L,len,ans;

	if (s1<=0) return (1.0);
	e1=(1.0-s1)/(1.0+s1);
	h1=-log(s1);
	if (s2<=0) L=1.0;
	else 
	 {
		h2=-log(s2);
		len=exp(-h2-acosh(cosh(h1)*cosh(h2)+sinh(h1)*sinh(h2)*t3));
		L=(1.0-len)/(1.0+len);
	 }
	e2=(L*L-e1*e1)/(2.0*e1*t3+2.0*L);
	if (s3<=0) L=1.0;
	else 
	 {
		h3=-log(s3);
		len=exp(-h3-acosh(cosh(h1)*cosh(h3)+sinh(h1)*sinh(h3)*t2));
		L=(1.0-len)/(1.0+len);
	 }
	e3=(L*L-e1*e1)/(2.0*e1*t2+2.0*L);
		/* have euclidean radii now */
	ans=e_cos_overlap_special(e1,e2,e3,t1,t2,t3,flag,incompat_err); 
		/* find cos from eucl routine */
	return (ans);
} /* h_cos_overlap_special */

int h_horo_center(complex z1,complex z2,complex *z3,
		  double s1,double s2,double *s3,
		  double o1,double o2,double o3)
/* compute center/s-radius for z3 when z1,z2 are horocycles; 
overlaps as usual. */
{
        int err_flag=0;
	double a_sq,srad2,R,d,r,h,factor,rad,erad,pp;
	complex p1,p2,p3,ectr,One,negOne,c;
	complex ecent,w1,w2,w3;
	Mobius M1,M2;
	extern Mobius trans_abAB();

	if ((z1.re*z1.re+z1.im*z1.im)<(.999999999999) 
	    || (z2.re*z2.re+z2.im*z2.im)<(.999999999999) || *s3>1.0) return 0;
	     /* too far from unit circle to be accurate. */
	if (*s3<=0) srad2=0.0;
	else srad2=(*s3)*(*s3);
	
/* first, compute eucl center/rad in normalized location:
first horo at -1, eucl rad R, second horo at 1, eucl rad 1/2, 
target circle in upper half plane; r/ectr are
eucl rad/cent of circle three in normalized position. */

	R=2/(o3+3.0);
	d=sqrt(0.25 + R*R+R*o3);
	r=(2.0*(d+R*o3)+1.0)*(1.0-srad2)/
		(8.0*d*(1.0+srad2)+((2.0-4.0*d)*o1 - 4.0*R*o2)*(srad2-1.0));
	ectr.re=0.5*(1-d)-(1.0/(2.0*d))*(0.25+r*o1-R*R-2.0*R*r*o2);
	a_sq=0.25+r*(r+o1);
	ectr.im=sqrt(a_sq-(ectr.re-0.5)*(ectr.re-0.5));

/* second, find normalizing mobius trans: M1 moves tangencies to
-1/1, resp, M2 fixes -1/1 and makes circle 2 got through origin. */

		/* need 3 pts on c2 */
	erad=-s2;
	ecent.re=(1.0-erad)*z2.re;ecent.im=(1.0-erad)*z2.im;
	p1.re=ecent.re;p1.im=ecent.im+erad;
	p2.im=ecent.im;p2.re=ecent.re+erad;
	p3.im=ecent.im;p3.re=ecent.re-erad;

	One.re=1.0;One.im=0.0;
	negOne.re=-1.0;negOne.im=0.0;
	M1=trans_abAB(z1,z2,negOne,One,&err_flag);
	/* if there's some error, err_flag will be set (though
	   we don't currently test it) and M1 should be identity */

	w1=mobius(M1,p1,1);
	w2=mobius(M1,p2,1);
	w3=mobius(M1,p3,1);
	circle_3(w1,w2,w3,&c,&rad); /* find new eucl rad */
	pp=(1.0-2*rad); /* place where M1(c2) hits real axis */
		/* M2(z)->(z-pp)/(-z*pp+1) moves pp to origin. */
	M2.a.re=1.0;
	M2.b.re=-pp;
	M2.c.re=-pp;
	M2.d.re=1.0;
	M2.a.im=M2.b.im=M2.c.im=M2.d.im=0.0;
	M2.flip=0;

	w1=mobius(M2,w1,1);
	w2=mobius(M2,w2,1);
	w3=mobius(M2,w3,1);
	circle_3(w1,w2,w3,&c,&rad);

		/* pick three pts on normalized circle three */
	h=cAbs(ectr);
	if (h>.0000000000001)
	 {
		factor=(h+r)/h;
		p1=ectr;p1.re *= factor;p1.im *= factor;
		factor=(h-r)/h;
		p2=ectr;p2.re *= factor;p2.im *= factor;
		p3=ectr;p3.re += r;
	 }
	else 
	 {
			p1=ectr;p1.re -= r;
			p2=ectr;p2.re += r;
			p3=ectr;p3.im -= r;
	 }
		/* apply M2 and then M1 inverses */
	p1=mobius(M2,p1,-1);
	p2=mobius(M2,p2,-1);
	p3=mobius(M2,p3,-1);
	p1=mobius(M1,p1,-1);
	p2=mobius(M1,p2,-1);
	p3=mobius(M1,p3,-1);

 		/* images under mobius */
	circle_3(p1,p2,p3,&c,&rad); /* find new eucl cent/rad */
	if (*s3<=.0000000000001) /* target=horocycle? p1 should 
				    be on unit circle */
	 {
		*z3=p1;
		*s3=-rad;
		return 1;
	 }
	e_to_h_data(c, rad, z3, s3);
	return 1;	
} /* h_horo_center */

int h_geodesic(complex a,complex b,hyp_geodesic *hg)
/* Given pts a b in closed disc, fills in info for approp hyp_geodesic
structure hg. */
{
	complex x,z,z1,z2,c;
	double aba,abb,d,dd,arg1,arg2,rad;

	hg->line_flag=0;
	aba=cAbs(a);
	abb=cAbs(b);
	x=csub(a,b);
	d=cAbs(x);
	if ( (aba<.0000000000001) || (abb<.0000000000001) 
	     || (d<.00002) || (d>1.98) 
		|| ((fabs(a.im)<.0000000000001) 
		    && (fabs(b.im)<.0000000000001)) ) 
		/* one or other at origin, two points close, points almost
		on opposite sides of origin, or both close to real axis.
		Use line. */
		{hg->z1=a;hg->z2=b;hg->line_flag=1;return 1;}
	if (aba>(.999999999999)) /* a on unit circle */
	 {
		if (abb>(.999999999999)) /* both on unit circle */
		 {
			dd=4-d*d;
			x=cadd(a,b);
			c.re=2*(x.re)/dd;
			c.im=2*(x.im)/dd;
			rad=d/sqrt(dd);
			if (rad>20.0) /* large, do line */
			 {hg->z1=a;hg->z2=b;hg->line_flag=1;return 1;}
			z=cdiv(csub(c,a),csub(b,a));
			if (z.im>0)
			 {z1=a;z2=b;}
			else {z1=b;z2=a;}
			arg1=Arg(csub(z1,c));
			arg2=Arg(csub(z2,c));
			if (arg2<arg1) arg2 += 2.0*M_PI;
			hg->z1=z1;hg->z2=z2;hg->c=c;hg->line_flag=0;
			hg->arg1=arg1;hg->arg2=arg2;hg->rad=rad;
			return 1;
		 }
		x=a;a=b;b=x;aba=abb; /* interchange so a not on unit circle */
	 }
	x.re=a.re/(aba*aba);
	x.im=a.im/(aba*aba);
	if (!circle_3(a,b,x,&c,&rad) || rad>20.0) /* large, do line */
	 {hg->z1=a;hg->z2=b;hg->line_flag=1;return 1;}
	z=cdiv(csub(c,a),csub(b,a));
	if (z.im>0)
		{z1=a;z2=b;}
	else {z1=b;z2=a;}
	arg1=Arg(csub(z1,c));
	arg2=Arg(csub(z2,c));
	if (arg2<arg1) arg2 += 2.0*M_PI;
	hg->z1=z1;hg->z2=z2;hg->c=c;hg->line_flag=0;
	hg->arg1=arg1;hg->arg2=arg2;hg->rad=rad;
	if ((arg2-arg1)<0.0175) hg->line_flag=1; /* very small angle */
	return 1;
} /* h_geodesic */

double h_edge_length(double s1,double s2,double ovlp)
/* compute hyp distance between hyp centers of circles given 
s_radii >0 and cosine of overlap angle */
{
	double ss1,ss2,dist;

	ss1=s1*s2;ss2=s2*s2;
	dist=acosh( (1/(4.0*s1*s2))*((1+ss1)*(1+ss2)+(1-ss1)*(1-ss2)*ovlp) );
	return dist; 
} /* h_edge_length */

double h_horo_rad(double s1,double ovlp)
/* return eucl rad of horocycle which overlaps circle at origin 
with s_radius >0 and cosine of overlap angle. */
{
	double r,x;

	r=(1-s1)/(1+s1);
	x=(1-r*r)/(2.0+2*r*ovlp);
	return x;
} /* h_horo_rad */

double h_dist(complex z,complex w,int uhp_flag)
     /* Compute hyperbolic distance in disc or upper halp plane (if
uhp_flag is set). Return 0 if z,w within small tolerance.  Return 
negative of euclidean distance if one or both of z,w are outside or 
within small tolerance of ideal bdry. Formula in disc is 
log((a+d)/(a-d)), where d = | z - w | 
and a = | 1 - zwb |, with zwb = z*cconj(w).  For uhp, move to disc first. */
{
  double a,d;
  complex zwb1,cpi;

  cpi.re=0.0;cpi.im=1.0;
  if ((d=cAbs(csub(z,w)))<.0000000000001) return 0.0;
  if ((uhp_flag && (w.im<.0000000000001 || z.im<.0000000000001))
      || (!uhp_flag 
	  && (cAbs(z) > .999999999999 || (cAbs(w) > .999999999999)))) 
    return -d;
  if (uhp_flag)
    {
      z=cdiv(csub(z,cpi),cadd(z,cpi));
      w=cdiv(csub(w,cpi),cadd(w,cpi));
    }
  zwb1=cmult(z,cconj(w));
  zwb1.re=1-zwb1.re;
  zwb1.im *= -1;
  a=cAbs(zwb1);
  return (log((a+d)/(a-d)));
} /* h_dist */
