/*  This file is part of "choosewm"
 *  Copyright (C) 2005 Bernhard R. Link
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include <config.h>

#include <errno.h>
#include <limits.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <unistd.h>
#include <stdbool.h>
#include <assert.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "global.h"

#define TRUE true
#define FALSE false

#ifndef SYSTEMWMFILE
#define SYSTEMWMFILE "/var/lib/choosewm/windowmanagers"
#endif
#ifndef SYSTEMCONFIG
#define SYSTEMCONFIG "/etc/X11/choosewm/config"
#endif
#ifndef XWINDOWMANAGERLINK
#define XWINDOWMANAGERLINK "/etc/alternatives/x-window-manager"
#endif

struct windowmanager  *windowmanagers;
bool dontask;
bool haderrors;
bool rescuemode;

static char *defaultwindowmanagerpath = NULL;
static char *userconffile;
static char *lastdecisionfile;
static char *dontaskfile;

struct inalias {
	struct inalias *next;
	char *toreplace;
	struct windowmanager *with;
	bool forceask;
} *aliases;


/* this is supposed to be done first, so nothing bad can happen when
 * we exit. */
char *xstrdup(/*@null@*/const char *s) {
	char *r;

	if( s == NULL )
		return NULL;
	r = strdup(s);
	if( r == NULL ) {
		fputs("Out of Memory!\n",stderr);
		exit(EXIT_FAILURE);
	}
	return r;

}
static void *xmalloc(size_t len) {
	char *result;

	result = malloc(len);
	if( result == NULL ) {
		fputs("Out of Memory!\n",stderr);
		exit(EXIT_FAILURE);
	}
	return result;
}

#define xisspace(x) (isspace(x)!=0)

static char *xreadlink(const char *path) {
	char *result,*h;
	int len,size;

	size = 1000;
	result = xmalloc(size);
	while( (len = readlink(path,result,size)) >= size ) {
		free(result);
		if( size >= INT_MAX/2 ) {
			fprintf(stderr,"Error looking where '%s' points too: Seems extremly big\n",path);
			exit(EXIT_FAILURE);
		}
		size *= 2;
		result = xmalloc(size);
	}
	if( len < 0 ) {
		fprintf(stderr,"Error looking where '%s' points to: %m\n",path);
		exit(EXIT_FAILURE);
	}
	if( len == 0 )
		len = 1;
	result[len] = '\0';
	h = realloc(result,len+1);
	if( h == NULL )
		return result;
	else
		return h;
}

static void removewm(const char *filename, int linenr, const char *path) {
	struct windowmanager **wm_p,*wm;

	wm_p = &windowmanagers;
	while( (wm=*wm_p) != NULL ) {
		int c = strcmp(wm->path,path);

		if( c < 0 ) {
			wm_p = &(wm->next);
			continue;
		}
		if( c == 0 ) {
			*wm_p = wm->next;
			free(wm->name);
			free(wm->path);
			free(wm->saveas);
			free(wm);
			return;
		}
		break;
	}
	if( !rescuemode )
	fprintf(stderr,"%s:%d: Ignoring removal of '%s' as not defined before.\n",filename,linenr,path);
}

static struct windowmanager *addwm(const char *name, const char *path, /*@null@*/const char *saveas) {
	struct windowmanager **wm_p,*wm;

	wm_p = &windowmanagers;
	while( (wm=*wm_p) != NULL ) {
		int c = strcmp(wm->path,path);

		if( c < 0 ) {
			wm_p = &(wm->next);
			continue;
		}
		if( c == 0 ) {
			free(wm->name);
			free(wm->saveas);
			/* I only hope g_strdup also terminates
			 * if there is no memory */
			wm->name = xstrdup(name);
			wm->saveas = xstrdup(saveas);
			return wm;
		}
		break;
	}
	wm = malloc(sizeof(*wm));
	if( wm == NULL ) {
		fputs("Out of Memory!\n",stderr);
		exit(EXIT_FAILURE);
	}
	wm->next = *wm_p;
	*wm_p = wm;
	wm->name = xstrdup(name);
	wm->path = xstrdup(path);
	wm->saveas = xstrdup(saveas);
	wm->isdefault = FALSE;
	return wm;
}

static char *extractvalue(char *buffer) {
	char *seperator;

	seperator = strchr(buffer,'=');
	if( seperator == NULL ) {
		return NULL;
	}
	*seperator = '\0';
	seperator++;
	return seperator;
}

static void parseaddition(const char *filename,int linenr,char *buffer) {
	char *value = extractvalue(buffer);
	if( value == NULL ) {
		fprintf(stderr,"%s:%d: Missing = sign!\n",filename,linenr);
		/* breaking as this might mean that following
		 * items are corrupt. */
		exit(EXIT_FAILURE);
	}
	addwm(buffer,value,NULL);
}

static bool isexecutable(const char *path) {
	struct stat s;
	int r;

	r = stat(path, &s);
	if( r != 0 )
		return false;
	/* should be a file (or symlink to file, as we use stat) */
	if( !(S_ISREG(s.st_mode)) )
		return false;

	if( (s.st_mode & (S_IXUSR|S_IXGRP|S_IXOTH)) == 0 )
		return false;
	/* file exists and there is someone that is alloed to execute: */
	return true;
}

static bool wmexists(const char *filename) {
	const char *path, *pb, *pe;
	size_t l;
	char *fullfilename;
	bool found;

	if( filename == NULL || filename[0] == '\0' )
		return false;
	if( filename[0] == '/' )
		return isexecutable(filename);
	l = strlen(filename);
	path = getenv("PATH");
	if( path == NULL )
		return false;
	pb = path;
	while( *pb == ':')
		pb++;
	while( *pb != '\0' ) {
		pe = strchr(pb, ':');
		if( pe == NULL )
			pe = pb + strlen(pb);
		if( pe == pb )
			continue;
		fullfilename = xmalloc(l+(pe-pb)+2);
		memcpy(fullfilename, pb, (size_t)(pe-pb));
		fullfilename[pe-pb] = '/';
		memcpy(fullfilename + (pe-pb) + 1, filename, l+1);
		found = isexecutable(fullfilename);
		free(fullfilename);
		if( found )
			return true;
		pb = pe;
		while( *pb == ':')
			pb++;
	}
	return false;
}

static struct windowmanager *findwm(const char *path) {
	struct windowmanager *wm;

	if( path == NULL )
		return NULL;
	wm = windowmanagers;
	while( wm != NULL ) {
		int c = strcmp(wm->path,path);

		if( c < 0 ) {
			wm = wm->next;
			continue;
		}
		if( c == 0 )
			return wm;
		break;
	}
	if( rescuemode && wmexists(path) ) {
		const char *wmbasename = strrchr(path, '/');

		if( wmbasename == NULL || wmbasename[1] == '\0' )
			wmbasename = path;
		else
			wmbasename++;

		return addwm(wmbasename, path, NULL);
	}
	return NULL;
}


static void parseinalias(const char *filename,int linenr,char *buffer) {
	char *value = extractvalue(buffer);
	struct windowmanager *wm;

	if( value == NULL ) {
		fprintf(stderr,"%s:%d: Missing = sign!\n",filename,linenr);
		/* breaking as this might mean that following
		 * items are corrupt. */
		exit(EXIT_FAILURE);
	}

	wm = findwm(value);
	if( wm == NULL ) {
		fprintf(stderr,"%s:%d: Could not find '%s', not yet defined?\n",filename,linenr,value);
	} else {
		struct inalias *alias;

		alias = xmalloc(sizeof(struct inalias));
		alias->next = aliases;
		aliases = alias;
		alias->toreplace = xstrdup(buffer);
		alias->with = wm;
		alias->forceask = FALSE;
	}
}

static void parseaskalias(const char *filename,int linenr,char *buffer) {
	char *value = extractvalue(buffer);
	struct windowmanager *wm;

	if( value == NULL ) {
		fprintf(stderr,"%s:%d: Missing = sign!\n",filename,linenr);
		/* breaking as this might mean that following
		 * items are corrupt. */
		exit(EXIT_FAILURE);
	}

	wm = findwm(value);
	if( wm == NULL ) {
		fprintf(stderr,"%s:%d: Could not find '%s', not yet definined?\n",filename,linenr,value);
	} else {
		struct inalias *alias;

		alias = xmalloc(sizeof(struct inalias));
		alias->next = aliases;
		aliases = alias;
		alias->toreplace = xstrdup(buffer);
		alias->with = wm;
		alias->forceask = TRUE;
	}
}

static void parseoutalias(const char *filename,int linenr,char *buffer) {
	char *value = extractvalue(buffer);
	struct windowmanager *wm;

	if( value == NULL ) {
		fprintf(stderr,"%s:%d: Missing = sign!\n",filename,linenr);
		/* breaking as this might mean that following
		 * items are corrupt. */
		exit(EXIT_FAILURE);
	}

	wm = findwm(buffer);
	if( wm == NULL ) {
		fprintf(stderr,"%s:%d: Could not find '%s', not yet definined?\n",filename,linenr,buffer);
	} else {
		free(wm->saveas);
		wm->saveas = xstrdup(value);
	}
}

static void parseconfigline(const char *filename,int linenr,char *buffer,
		const char **stringkeys,char **stringvalues) {

	char *start;
	int i;

	switch( buffer[0] ) {
	  case '+':
		  start = buffer+1;
		  while( *start != '\0' && xisspace(*start) ) {
			  start++;
		  }
		  parseaddition(filename,linenr,start);
		  return;
	  case '-':
		  start = buffer+1;
		  while( *start != '\0' && xisspace(*start) ) {
			  start++;
		  }
		  removewm(filename,linenr,start);
		  return;
#define stroption(x,y) 	if( strncasecmp(buffer,x,strlen(x)) == 0 ) { \
				start=buffer+strlen(x); \
			  	while( *start != '\0' && xisspace(*start) ) { \
					 start++; \
			  	} \
				free(y); \
				y = xstrdup(start); \
				return; \
			}
#define booloption(x,y) 	if( strncasecmp(buffer,x,strlen(x)) == 0 ) { \
				start=buffer+strlen(x); \
			  	while( *start != '\0' && xisspace(*start) ) { \
					 start++; \
			  	} \
				if( *start == '0' ) y = FALSE; \
				else if( *start =='1' ) y = TRUE; \
				else { fprintf(stderr,"%s:%d: Unparseable as 0/1 value: '%s'\n",filename,linenr,start); exit(EXIT_FAILURE); } \
				return; \
			}
#define calloption(x,func) if( strncasecmp(buffer,x,strlen(x)) == 0 ) { \
				start=buffer+strlen(x); \
			  	while( *start != '\0' && xisspace(*start) ) { \
					 start++; \
			  	} \
				func(filename,linenr,start); \
				return; \
			}

	  case 'd':
		  stroption("default:",defaultwindowmanagerpath);
		  stroption("dontaskfile:",dontaskfile);
		  break;
	  case 'u':
		  stroption("userconfig:",userconffile);
		  break;
	  case 'l':
		  stroption("lastdecisionfile:",lastdecisionfile);
		  break;
	  case 'a':
		  calloption("add:",parseaddition);
		  calloption("askalias:",parseaskalias);
		  break;
	  case 'i':
		  calloption("inalias:",parseinalias);
		  break;
	  case 'o':
		  calloption("outalias:",parseoutalias);
		  break;
	  case 'r':
		  calloption("remove:",removewm);
		  break;
	  case 's':
		  start = buffer+1;
		  for( i = 0 ; stringkeys[i] != NULL ; i++ ) {
			  size_t l = strlen(stringkeys[i]);
			  if( strncasecmp(start,stringkeys[i],l) == 0
				&& start[l] == ':' ) {
				  start += l+1;
				  while( *start != '\0' && xisspace(*start)){
					  start++;
				  }
				  stringvalues[i] = xstrdup(start);
				  return;
			  }
		  }
		  break;
#undef stroption
#undef booloption
#undef calloption

	}
	fprintf(stderr,"%s:%d: Unexpected command: '%s'\n",filename,linenr,buffer);
	exit(EXIT_FAILURE);
}

static void readconfigfile(const char *filename,const char **stringkeys,char **stringvalues) {
	FILE *f;

	f = fopen(filename,"r");
	if( f != NULL ) {
		char buffer[1000];
		int linenr = 0;

		while( fgets(buffer,999,f) != NULL ) {
			size_t len = strlen(buffer);

			linenr++;

			if( buffer[0] == '#' )
				continue;

			while( len > 0 && xisspace(buffer[len-1])){
				buffer[--len] = '\0';
			}
			if( len == 0 )
				continue;

			parseconfigline(filename,linenr,buffer,stringkeys,stringvalues);
		}
		fclose(f);
	}
}

static bool trywm(const char *path) {
	struct windowmanager *wm;
	const char *lastpart;

	if( path == NULL )
		return FALSE;

	for( wm = windowmanagers; wm != NULL ; wm = wm->next ) {
		if(strcmp(wm->path,path) == 0) {
			wm->isdefault = TRUE;
			return TRUE;
		}
	}
	lastpart = strrchr(path,'/');
	if( lastpart == NULL )
		lastpart = path;
	else
		lastpart++;

	for( wm = windowmanagers; wm != NULL ; wm = wm->next ) {
		const char *wmlastpart;

		wmlastpart = strrchr(wm->path,'/');
		if( wmlastpart == NULL )
			wmlastpart = wm->path;
		else
			wmlastpart++;

		if(strcmp(wmlastpart,lastpart) == 0) {
			wm->isdefault = TRUE;
			return TRUE;
		}
	}
	/* wm was not specified, check if it is there, anyway */
	if( wmexists(path) ) {
		const char *wmbasename;

		wmbasename = strrchr(path,'/');
		if( wmbasename == NULL || wmbasename[1] == '\0' )
			wmbasename = path;
		else
			wmbasename++;

		wm = addwm(wmbasename, path, NULL);
		if( wm != NULL ) {
			wm->isdefault = TRUE;
			fprintf(stderr, "Added '%s' to list of window managers, as it is the configured default.\n", path);
			return true;
		}
	}
	return FALSE;
}

static void finddefaultwm(const char *lastwm) {
	char *xwindowmanagerlink;
	const struct inalias *a;

	if( lastwm != NULL)
	for( a = aliases ; a != NULL ; a = a->next ) {
		if( strcmp(a->toreplace,lastwm) == 0 ) {
			a->with->isdefault = TRUE;
			if( a->forceask )
				dontask = FALSE;
			return;
		}
	}
	if( trywm(lastwm) )
		return;
	haderrors = TRUE;
	if( defaultwindowmanagerpath != NULL )
		findwm(defaultwindowmanagerpath);
	if( trywm(defaultwindowmanagerpath) )
		return;
	if( trywm(getenv("DEFAULTWINDOWMANAGER")) )
		return;
	if( trywm(getenv("DEFAULTWINDOWMANAGER")) )
		return;
	xwindowmanagerlink = xreadlink(XWINDOWMANAGERLINK);
	if( trywm(xwindowmanagerlink) ) {
		free(xwindowmanagerlink);
		return;
	}
	free(xwindowmanagerlink);

	fputs("No default window manager found!\n",stderr);
}

/* called in rescue mode to get some more selections */
static void addfallbackwms(void) {
	char *xwindowmanagerlink;

	/* just look for them, in rescue mode they will be added
	 * if not yet found: */
	findwm(defaultwindowmanagerpath);
	findwm(getenv("DEFAULTWINDOWMANAGER"));
	xwindowmanagerlink = xreadlink(XWINDOWMANAGERLINK);
	findwm(xwindowmanagerlink);
	free(xwindowmanagerlink);
}


void readconfigs(const char **stringkeys,char **stringvalues) {
	FILE *f;
	const char *homeenv = getenv("HOME");
	char *filename,*lastwm;

	if( homeenv == NULL ) {
		fputs("No HOME environment variable defined!\n",stderr);
		exit(EXIT_FAILURE);
	}
	haderrors = FALSE;
	rescuemode = FALSE;

	userconffile = strdup(".choosewm/config");
	lastdecisionfile = strdup(".choosewm/lastwm");
	dontaskfile = strdup(".choosewm/dontask");
	windowmanagers = NULL;
	aliases = NULL;

	f = fopen(SYSTEMWMFILE,"r");
	if( f != NULL ) {
		char buffer[1000];
		int linenr = 0;

		while( fgets(buffer,999,f) != NULL ) {
			size_t len = strlen(buffer);

			linenr++;

			if( buffer[0] == '#' )
				continue;

			while( len > 0 && xisspace(buffer[len-1])){
				buffer[--len] = '\0';
			}
			if( len == 0 )
				continue;

			parseaddition(SYSTEMWMFILE,linenr,buffer);

		}
		fclose(f);
	}
	if( windowmanagers == NULL ) {
		fprintf(stderr, "No window managers found in global '%s'!\n"
				"Entering rescue mode.\n", SYSTEMWMFILE);
		rescuemode = true;
	}
	readconfigfile(SYSTEMCONFIG,stringkeys,stringvalues);
	if( userconffile != NULL && strcmp(userconffile,"-") != 0 ) {
		char *userconfig;
		if( asprintf(&userconfig,"%s/%s",homeenv,userconffile )<0
				|| userconfig == NULL) {
			fputs("Out of Memory!\n",stderr);
			exit(EXIT_FAILURE);
		}
		readconfigfile(userconfig,stringkeys,stringvalues);
	}
	if( windowmanagers == NULL && !rescuemode ) {
		fputs("No window managers found!\n",stderr);
		exit(EXIT_FAILURE);
	}
	if( rescuemode )
		addfallbackwms();
	if( asprintf(&filename,"%s/%s",homeenv,dontaskfile)< 0 || filename == NULL ) {
		fputs("Out of Memory!\n",stderr);
		exit(EXIT_FAILURE);
	}
	if( (f = fopen(filename,"r")) != NULL ) {
		dontask = TRUE;
		fclose(f);
	} else {
		dontask = FALSE;
	}
	free(filename);

	if( asprintf(&filename,"%s/%s",homeenv,lastdecisionfile)< 0 || filename == NULL ) {
		fputs("Out of Memory!\n",stderr);
		exit(EXIT_FAILURE);
	}

	lastwm = NULL;
	if( (f = fopen(filename,"r")) != NULL ) {
		char lastdecision[4001];
		errno = 0;
		if( fgets(lastdecision,4000,f) != NULL ) {
			size_t l = strlen(lastdecision);
			if( l > 0 && lastdecision[l-1] !='\n' ) {
				fprintf(stderr,"%s:0: Unexpected long line!\n",filename);
				exit(EXIT_FAILURE);
			}
			while( l > 0 && xisspace(lastdecision[l-1]) ) {
				lastdecision[--l] = '\0';
			}
			if( l > 0 ) {
				const char *start = lastdecision;

				while( *start != '\0' && xisspace(*start) )
					start++;

				lastwm = xstrdup(start);
			}

		} else {
			int e = errno;
			if( e == 0 )
				fprintf(stderr,"Could not get information from %s. File empty? Falling back to default.\n",filename);
			else {
				fprintf(stderr,"Error reading %s: %s. Falling back to default.\n",
						filename, strerror(e));
			}
			haderrors = TRUE;
		}
		fclose(f);
	}
	free(filename);

	finddefaultwm(lastwm);
	free(lastwm);
}

static void makeparentdirs(char *filename) {
	char *p;

	p = filename;
	while( (p=strchr(p,'/')) != NULL ) {
		*p = '\0';
		mkdir(filename,S_IRWXU /*0777*/);
		*p = '/';
		p++;
	}
}

static void save(struct windowmanager *wm, bool dontaskagain) {
	char *filename, *finalfilename;
	const char *homeenv = getenv("HOME");
	// TODO: return error code

	assert(homeenv != NULL);

	if( asprintf(&filename,"%s/%s",homeenv,dontaskfile)< 0 || filename == NULL ) {
		fputs("Out of Memory!\n",stderr);
	} else if( dontaskagain ) {
		FILE *f;

		makeparentdirs(filename);
		f = fopen(filename,"w");
		if( f != NULL ) {
			fputs("Delete this file to be asked again for a wm.\n",f);
			fclose(f);
		} else {
			fprintf(stderr,"Error creating '%s': %m\n",filename);
		}
		free(filename);
	} else {
		unlink(filename);
		free(filename);
	}
	if( asprintf(&filename,"%s/%s.tmp",homeenv,lastdecisionfile)< 0
	    || filename == NULL ) {
		fputs("Out of Memory!\n",stderr);
	} else if( (finalfilename = strndup(filename, strlen(filename)-4)) == NULL ) {
		free(filename);
		fputs("Out of Memory!\n",stderr);
	} else {
		FILE *f;

		makeparentdirs(filename);
		f = fopen(filename,"w");
		if( f != NULL ) {
			if( wm->saveas )
				fputs(wm->saveas,f);
			else
				fputs(wm->path,f);
			fputs("\n",f);
			if( ferror(f) != 0 ) {
				fprintf(stderr,"Error writing to '%s': %m\n",filename);
				fclose(f);
				unlink(filename);
			} else if( fclose(f) != 0 ) {
				fprintf(stderr,"Error writing to '%s': %m\n",filename);
				unlink(filename);
			} else {
				rename(filename, finalfilename);
			}
		} else {
			fprintf(stderr,"Error creating '%s': %m\n",filename);
		}
		free(filename);
		free(finalfilename);
	}
}

void printwm(struct windowmanager *wm, bool dontaskagain) {
	save(wm,dontaskagain);
	puts(wm->path);
	exit(EXIT_SUCCESS);
}

void savewm(struct windowmanager *wm,bool dontaskagain) {
	save(wm,dontaskagain);
	exit(EXIT_SUCCESS);
}

void startwm(struct windowmanager *wm, bool dontaskagain) {
	save(wm,dontaskagain);
	execlp(wm->path,wm->path,NULL);
	fprintf(stderr,"Error executing '%s': %m\n",wm->path);
	exit(EXIT_FAILURE);
}
