/* catdvi - get text from DVI files
   Copyright (C) 2001 Bjoern Brill <brill@fs.math.uni-frankfurt.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdlib.h>
#include "sparse.h"
#include "util.h"


#define SPARP_STAGE_MASK ((1 << SPARP_BITS_PER_STAGE) - 1)


/* Readability note: below, the notion of "index shift" of a tree node
 * is used. Ignoring the technical meaning, one could define
 * stage(node) := height(tree) - distance(root, node)
 * (i.e. the "distance to the top") and
 * index shift(node) := SPARP_BITS_PER_STAGE * stage(node) .
 *
 * In some places, a _negative_ index shift (impossible according to above
 * definition) is used to indicate that the tree is empty.
 */
 

/* Allocate & initialize a new tree node. shift is the index shift of the
 * new node and tells us wheter it should become a leaf or not.
 */
sparp_rec_t * sparp_new_block(sparp_t * this, signed int shift);

/* Free a tree node and, recursively, all its iterated children. shift
 * is the index shift of the freed node and tells us how deep we have
 * to recurse.
 */
void sparp_free_block(sparp_t * this, sparp_rec_t * block, signed int shift);

/* Create a new tree root and make the old one the 0-th child of the new,
 * thus increasing the height of the tree by 1.
 */
void sparp_add_stage(sparp_t * this);


/***********************************************************************/


void sparp_init(sparp_t * this)
{
    /* we start with a single entry (index 0) which is NULL */
    this->head.v = NULL;
    this->max_bits = 0;
    this->max_shift = this->max_bits - SPARP_BITS_PER_STAGE;
    this->max_index = 0;
}

void sparp_done(sparp_t * this)
{
    if(this->max_shift >= 0) {
    	sparp_free_block(this, this->head.p, this->max_shift);
    }
    sparp_init(this);
    	/* doesn't alloc anything and leaves a consistent state */
}


void * sparp_read(sparp_t * this,  sparp_index_t i)
{
    signed int sh;
    sparp_rec_t r;
    
    if(i > this->max_index) return NULL;
    
    r = this->head;
    for(sh = this->max_shift; sh >= 0; sh -= SPARP_BITS_PER_STAGE) {
    	if(r.p == NULL) return NULL;
	r = r.p[(i >> sh) & SPARP_STAGE_MASK];
    }
    
    return r.v;
}


void ** sparp_addressof_entry(sparp_t * this, sparp_index_t i)
{
    signed int sh;
    sparp_rec_t r;
    sparp_rec_t * p;
    
    while(i > this->max_index) sparp_add_stage(this);
    
    p = &(this->head);
    for(sh = this->max_shift; sh >= 0; sh -= SPARP_BITS_PER_STAGE) {
    	r = *p;
	if(r.p == NULL) r.p = p->p = sparp_new_block(this, sh);
	p = r.p + ((i >> sh) & SPARP_STAGE_MASK);
    }
    
    return &(p->v);
}


void sparp_write(sparp_t * this, sparp_index_t i, void * v)
{
    *sparp_addressof_entry(this, i) = v;
}


sparp_rec_t * sparp_new_block(sparp_t * this, signed int shift)
{
    sparp_rec_t * p, * q;
    
    p = malloc((1 << SPARP_BITS_PER_STAGE) * sizeof(sparp_rec_t));
    if(p == NULL) enomem();

    /* Fill the allocated block with NULLs. Unfortunately, it is possible that
     * NULL pointers are represented by something different from a zero bit
     * pattern and it is also possible that pointers to different types have
     * different memory layout. Otherwise a calloc() would have done it all
     * at once, and more efficient.
     */
    if(shift > 0) {
    	for(q = p; q < p + (1 << SPARP_BITS_PER_STAGE); ++q) q->p = NULL;
    }
    else {
    	for(q = p; q < p + (1 << SPARP_BITS_PER_STAGE); ++q) q->v = NULL;
    }
    
    return p;
}


void sparp_free_block(sparp_t * this, sparp_rec_t * block, signed int shift)
{
    int i;
    
    if(shift > 0) {
    	for(i = 0; i <= SPARP_STAGE_MASK; ++i) {
	    if(block[i].p != NULL) sparp_free_block(
	    	this,
		block[i].p,
		shift - SPARP_BITS_PER_STAGE
	    );
	}
    }
    free(block);
}

void sparp_add_stage(sparp_t * this)
{
    sparp_rec_t * p;
    
    p = sparp_new_block(this, this->max_shift + SPARP_BITS_PER_STAGE);

    this->max_bits += SPARP_BITS_PER_STAGE;
    this->max_shift += SPARP_BITS_PER_STAGE;
    this->max_index = (this->max_index << SPARP_BITS_PER_STAGE) |
    	SPARP_STAGE_MASK;

    p[0] = this->head;
    this->head.p = p;
}
