/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/


#ifndef MESHCOMPONENT_H
#define MESHCOMPONENT_H


#include "Component.h"
#include "Geometry.h"

class vtkPointSet;

namespace camitk
{
  /** 
   * @ingroup group_sdk_libraries_core_component_mesh
   * 
   * @brief
   * Basic component to manage any kind of mesh.
   */
  class CAMITK_API MeshComponent : public Component
  {
    Q_OBJECT
  public:

    /** Creates a top-level MeshComponent from a file.
     *  \note this is only to be used from a Component Extension open(...) or from an Action that creates data from a filter or transformation of a vtkPointSet.
     * 
     *  Please consider using MeshComponent(vtkSmartPointer<vtkPointSet> , const QString &)
     */
    MeshComponent(const QString & file) throw(AbortException);

    /** Creates a top-level MeshComponent from a vtkPointSet (and instanciate its 3D representation).
     * 
     * \note the filename is set to the empty string "".
     * 
     * \note if aPointSet is NULL, the representation is not initialized, it is the responsability of the subclass to initialize it later
     * 
     * @param aPointSet point set of the new MeshComponent
     * @param name name to be given to the Component
     */
    MeshComponent(vtkSmartPointer<vtkPointSet> aPointSet, const QString &name);

    /** Creates a MeshComponent as a sub component of another Component using a vtkPointSet (and instanciate its 3D representation).
     * 
     * \note if aPointSet is NULL, the representation is not initialized, it is the responsability of the subclass to initialize it later
     * 
     * @param parentComponent the parent component of the new MeshComponent
     * @param aPointSet point set of the new MeshComponent
     * @param name name to be given to the new MeshComponent
     */
    MeshComponent(Component *parentComponent, vtkSmartPointer<vtkPointSet> aPointSet, const QString &name);
    
    /// Destructor
    virtual ~MeshComponent();

    /// reimplemented to save the last pick point id
    void pointPicked ( vtkIdType pointId, bool );
    
    /// reimplemented to save the last pick point id
    void cellPicked ( vtkIdType cellId, bool );

    /// get the last pick point id, @return -1 if no point where picked
    vtkIdType getPickedPointId();
    
    /// get the last pick point id, @return -1 if no point where picked
    vtkIdType getPickedCellId();

    /** update property: empty method.
     */
    void updateProperty(QString , QVariant) {};   

  protected:
    /// build the instance of Geometry from the given vtkPointSet
    virtual void initRepresentation(vtkSmartPointer<vtkPointSet>);

    /// create and initialize dynamic properties
    virtual void initDynamicProperties();

  private:

    /// the concrete building of the 3D objects (Slice/Geometry): none in this case, everything is done by initRepresentation(vtkPointSet)
    virtual void initRepresentation() {};

    /// the last picked point
    vtkIdType pickedPointId;

    /// the last picked cell
    vtkIdType pickedCellId;
  };

}

#endif // MESHCOMPONENT_H
