/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef IMAGE_COMPONENT_H
#define IMAGE_COMPONENT_H

// -- Core image component stuff
#include "SingleImageComponent.h"
#include "CamiTKAPI.h"

// -- vtk stuff
#include <vtkPolyData.h>
#include <vtkSmartPointer.h>
#include <vtkImageData.h>
#include <vtkTransform.h>
#include <vtkImageFlip.h>
#include <vtkWindowLevelLookupTable.h>

// -- QT stuff classes
class QMenu;
#include <QVector3D>
#include <QVariant>

namespace camitk {

/**
 * @ingroup group_sdk_libraries_core_component_image
 * 
 * @brief
 * The manager of the Image Volume data.
 * An image volume data has no concrete 3D representation,
 *  but handles several sub-components for axial, sagittal and coronal slices.
 *
 * Maybe sometime we should add a volume rendering representation
 * See the method build3DVolume() in the former Image class...
 * Or the 3D representation could also be a 3D marching cube reconstruction...
 *
 * Vtk Pipeline:
 * \verbatim
 * \endverbatim
 *
 *
 */

class MeshComponent;

class CAMITK_API ImageComponent : public camitk::Component {
    Q_OBJECT

    /// The name of the image in the explorer
    Q_PROPERTY(QString imageName READ getImageName WRITE setImageName)

    /// the number of voxels in X, Y and Z direction
    Q_PROPERTY(QVariantMap imageSize READ getImageSize)

    /// The size of voxels in X, Y and Z direction (in mm)
    Q_PROPERTY(QVector3D voxelSize READ getVoxelSize)

    /// The 3D position of the image origin (X, Y, Z)
    Q_PROPERTY(QVector3D origin READ getImageOrigin)

    /// the type of data (uchar, short, ulong, ...)
    Q_PROPERTY(QString imageType READ getDataType)

    /// Set Axial, Coronal and Sagittal Slices visible in 3D
    Q_PROPERTY(bool viewIn3D READ getViewIn3D WRITE setViewIn3D);


public:

    /** \enum MedicalDataOrigin : There is no convension concerning the origin of the data. */
    enum MedicalDataOrigin {
        RAI_X,  /// Right Anterior Inferior  _ Axial normal X
        RAI_Y,  /// Right Anterior Inferior  _ Axial normal Y
        RAI_Z,  /// Right Anterior Inferior  _ Axial normal Z
        LAI_X,  /// Left Anterior Inferior   _ Axial normal X
        LAI_Y,  /// Left Anterior Inferior   _ Axial normal Y
        LAI_Z,  /// Left Anterior Inferior   _ Axial normal Z
        RPI_X,  /// Right Posterior Inferior _ Axial normal X
        RPI_Y,  /// Right Posterior Inferior _ Axial normal Y
        RPI_Z,  /// Right Posterior Inferior _ Axial normal Z
        LPI_X,  /// Left Posterior Inferior  _ Axial normal X
        LPI_Y,  /// Left Posterior Inferior  _ Axial normal Y
        LPI_Z,  /// Left Posterior Inferior  _ Axial normal Z
        RAS_X,  /// Right Anterior Superior  _ Axial normal X
        RAS_Y,  /// Right Anterior Superior  _ Axial normal Y
        RAS_Z,  /// Right Anterior Superior  _ Axial normal Z
        LAS_X,  /// Left Anterior Superior   _ Axial normal X
        LAS_Y,  /// Left Anterior Superior   _ Axial normal Y
        LAS_Z,  /// Left Anterior Superior   _ Axial normal Z
        RPS_X,  /// Right Posterior Superior _ Axial normal X
        RPS_Y,  /// Right Posterior Superior _ Axial normal Y
        RPS_Z,  /// Right Posterior Superior _ Axial normal Z
        LPS_X,  /// Left Posterior Superior  _ Axial normal X
        LPS_Y,  /// Left Posterior Superior  _ Axial normal Y
        LPS_Z  /// Left Posterior Superior  _ Axial normal Z
    };

    /// constructor
    ImageComponent(const QString & file) throw(AbortException) ;

    /** Creates an ImageComponent from a vtkImageData
     * @param anImageData : volume image of the new ImageComponent
     * @param name: name to be given to the Component (this name will apear in the explorer)
     * @param copy: perform or not a deep copy of the image given in parameters.
     *      By default, does not copy the original image, but references the corresponding
     *          smart pointer (for memory reasons, but if copy is set to true, performs a deep copy).
     */
    ImageComponent(vtkSmartPointer<vtkImageData> anImageData, const QString &name, bool copy=false)
    throw(AbortException) ;

    /// Destructor
    ~ImageComponent();

    /// set selected will select all the Image components (axial, sagittal and coronal).
    virtual void setSelected(const bool b, const bool recursive=false);

    /// getter/setter for the property
    QString getImageName() const;
    void setImageName(const QString& );

    /// get the image volume managed by this Component
    vtkSmartPointer<vtkImageData> getImageData();

    /** Method called when a pixel has been picked in the 3D view.
     * This method tells all the scene3D to display the slice containing the picked pixel.
     * The arguments are the ccordinates of the 3D point.
     */
    void pixelPicked(double x, double y, double z, SingleImageComponent *whoIsAsking);

    /// Get the last pixel picked using CTRL + LEFT/RIGHT CLICK
    void getLastPixelPicked( int * x, int * y, int * z );

    /** Number of colors: number of possible gray levels in the image
     *  computed from the min and the max of the data type ;
     *  e.g. for a volume coded on unsigned char, returns 256.
     */
    int getNumberOfColors() const;

    /** Min possible gray level of the image given its data type */
    double getMinColor() const;

    /** Max possible gray level of the image given its data type */
    double getMaxColor() const;

    /** Number of axial slices (i.e. dim[2]) */
    int getNumberOfSlices() const;

    /// Update the lookup table of the image viewer (see InterfaceBitMap).
    virtual void setLut(vtkSmartPointer<vtkWindowLevelLookupTable> lookupTable);

    /// get the current lookup table
    virtual vtkSmartPointer<vtkWindowLevelLookupTable>  getLut();

    /// force refresh of all interactive viewers that are displayng sub-components
    /// as ImageComponent is not itself displayed by any viewer
    virtual void refresh() const;

    /** Returns the axial slice */
    SingleImageComponent * getAxialSlices();
    /** Returns the coronal slice */
    SingleImageComponent * getCoronalSlices();
    /** Returns the sagittal slice */
    SingleImageComponent * getSagittalSlices();
    /** Returns the arbitrary slice */
    SingleImageComponent * getArbitrarySlices();
    /** Returns the MeshComponent which will contain the volume rendering actor */
    MeshComponent * getVolumeRenderingChild();

    /** Replaces the current image volume by the one given in parameters
     *  If copy is set to true, performs a deep copy before replacing the image
     *  If copy is set to false, only takes the smart pointer as input.
     */
    virtual void replaceImageData(vtkSmartPointer<vtkImageData> anImageData, bool copy=false);

    /// property getters
    QVariantMap getImageSize() const;
    QString getDataType() const;
    QVector3D getVoxelSize() const;
    QVector3D getImageOrigin() const;
    bool      getViewIn3D() const;
    void      setViewIn3D(bool);

protected:

    virtual void setImageData(vtkSmartPointer<vtkImageData> anImageData, bool copy);

private:

    /// the concrete building of the 3D objects (Slice/Geometry): none in this case!
    virtual void initRepresentation() {};

    // builds default lookup table
    void initLookupTable();

    /// build the SingleImage dcs (one for each image plane);
    void buildImageComponents();

    void updateImageComponents();

    /// internal method used to put a mesh in volumeRenderingChild
    /// and accessoiry display the bounding box
    vtkSmartPointer<vtkPolyData> getBoundingBox();

private:
    /// the core Image Volume that is managed here

    vtkSmartPointer<vtkImageData> originalImageData;

    SingleImageComponent * axialSlices;
    SingleImageComponent * sagittalSlices;
    SingleImageComponent * coronalSlices;
    SingleImageComponent * arbitrarySlices;

    /// When an action computes volume rendering for an image,
    ///		it stores the corresponding actor as a prop of this Component.
    MeshComponent * volumeRenderingChild;

    /// the current lookup table
    vtkSmartPointer<vtkWindowLevelLookupTable> lut;

    /// Store the last pixel selected, in original slices ref
    int currentPixelPicked[3];

    /// initialize pointers to NULL and other attributes
    virtual void init();

    /// Qt properties
    QString imageName;

    /// Display Axial, Coronal and Sagittal Slices in 3D viewer
    bool viewIn3D;
};

// -------------------- getImageData --------------------
inline vtkSmartPointer<vtkImageData> ImageComponent::getImageData() {
    return originalImageData;
}

}

#endif //IMAGE_COMPONENT_H
