/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include "Decimation.h"
#include "DecimationWidget.h"

//-- CamiTK core stuff
#include <MeshComponent.h>
#include <Application.h>
using namespace camitk;

//-- Vtk Stuff
#include <vtkSmartPointer.h>
#include <vtkCallbackCommand.h>
#include <vtkDecimatePro.h>
#include <vtkCleanPolyData.h>
#include <vtkPolyDataWriter.h>

// -------------------- Decimation constructor --------------------
Decimation::Decimation(ActionExtension *extension) : Action(extension){
    setName("Decimation");
    setDescription("Decimate mesh data");
    setComponent("MeshComponent");
    setFamily("Mesh Processing");
    addTag("Decimation");
    addTag("Simplify");
    
    // lazy instanciation
    actionWidget = NULL;

    // decimation properties
    setProperty("percentageOfReduction", 25); // default 25%
    setProperty("preserveTopology",false);
    setProperty("setDetails", false);
    setProperty("maxError",0.0);
    setProperty("featureAngle", 0.0);
    setProperty("splitting",false);
    setProperty("splitAngle", 0.0);
    setProperty("boundaryDeletion", false);
    setProperty("degree",0);
    setProperty("inflectionRatio", 0.0);
}

// -------------------- Decimation destructor  --------------------
Decimation::~Decimation() {
}

// -------------------- getWidget --------------------
QWidget * Decimation::getWidget() {
    //-- create the widget if it does not exist
    if (!actionWidget)
        actionWidget = new DecimationWidget(this);

    //-- update the widget (should not be really needed)
    actionWidget->updateTargets();

    //-- if it was hidden
    actionWidget->show();
    
    return actionWidget;
}

// -------------------- apply --------------------
Action::ApplyStatus Decimation::apply() {
    // set waiting cursor and status bar
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    Application::showStatusBarMessage("Performing Decimation");
    Application::resetProgressBar();

    MeshComponent * targetMesh = dynamic_cast<MeshComponent*>(getTargets().last());
    
    vtkSmartPointer<vtkDecimatePro> mcDecimate = vtkSmartPointer<vtkDecimatePro>::New();
    
    mcDecimate->SetInput ( targetMesh->getPointSet() );
    mcDecimate->SetTargetReduction ( property("percentageOfReduction").toDouble()/100.0 );
    mcDecimate->SetPreserveTopology ( property("preserveTopology").toBool() );
    
    if ( property("setDetails").toBool() ) {
        mcDecimate->SetMaximumError ( property("maxError").toDouble() );
        mcDecimate->SetFeatureAngle ( property("featureAngle").toDouble() );
        mcDecimate->SetSplitting ( property("splitting").toBool() ); 
        mcDecimate->SetSplitAngle ( property("splitAngle").toDouble() );
        mcDecimate->SetBoundaryVertexDeletion ( property("boundaryDeletion").toBool() );
        mcDecimate->SetDegree ( property("degree").toInt() ); 
        mcDecimate->SetInflectionPointRatio ( property("inflectionRatio").toDouble() );
    }
    
    vtkSmartPointer<vtkCallbackCommand> progressCallback = vtkSmartPointer<vtkCallbackCommand>::New();
    progressCallback->SetCallback(&Application::vtkProgressFunction);
    mcDecimate->AddObserver(vtkCommand::ProgressEvent, progressCallback);
    mcDecimate->Update();
    
    // clean
    vtkSmartPointer<vtkCleanPolyData> cleaner = vtkSmartPointer<vtkCleanPolyData>::New();
    cleaner->SetInput(mcDecimate->GetOutput());
    cleaner->AddObserver(vtkCommand::ProgressEvent, progressCallback);
    cleaner->Update();
    
    // Create a mesh Component
    vtkSmartPointer<vtkPointSet> resultPointSet = cleaner->GetOutput();
    new MeshComponent(resultPointSet, targetMesh->getName() + "_decimated");
    Application::refresh();
    
    Application::resetProgressBar();
    QApplication::restoreOverrideCursor();    
    Application::showStatusBarMessage("");
    
    return SUCCESS;
}
