(*********************************************************************************)
(*                Cameleon                                                       *)
(*                                                                               *)
(*    Copyright (C) 2005,2006 Institut National de Recherche en Informatique     *)
(*    et en Automatique. All rights reserved.                                    *)
(*                                                                               *)
(*    This program is free software; you can redistribute it and/or modify       *)
(*    it under the terms of the GNU Library General Public License as            *)
(*    published by the Free Software Foundation; either version 2 of the         *)
(*    License, or  any later version.                                            *)
(*                                                                               *)
(*    This program is distributed in the hope that it will be useful,            *)
(*    but WITHOUT ANY WARRANTY; without even the implied warranty of             *)
(*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              *)
(*    GNU Library General Public License for more details.                       *)
(*                                                                               *)
(*    You should have received a copy of the GNU Library General Public          *)
(*    License along with this program; if not, write to the Free Software        *)
(*    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA                   *)
(*    02111-1307  USA                                                            *)
(*                                                                               *)
(*    Contact: Maxence.Guesdon@inria.fr                                          *)
(*                                                                               *)
(*********************************************************************************)

(** Graphical interface for the OCaml toplevel *)

open Outcometree
open Parsetree

module O = Config_file

let _ = GMain.Main.init ()

let _ = Toploop.set_paths ()
let _ = Toploop.initialize_toplevel_env()

(** {2 Options} *)

let op_filename = Filename.concat Cam_messages.rc_dir "topcameleon"
let op_ini = new O.group
let save_options () = op_ini#write op_filename
let load_options () = op_ini#read op_filename

let color_caml_output = new O.string_cp ~group: op_ini
    ["colors" ; "ocaml_output"]
    "Black" ""

let color_error = new O.string_cp ~group: op_ini
    ["colors" ; "error_background"]
    "Yellow" ""

let auto_expand = new O.bool_cp ~group: op_ini
    ["auto_expand"] true ""

let elements_frame_width = new O.int_cp ~group: op_ini
    ["elements_frame_width"] 150 ""

let value_frame_width = new O.int_cp ~group: op_ini
    ["value_frame_width"] 150 ""

let output_frame_height = new O.int_cp ~group: op_ini
    ["output_frame_height"] 300 ""

let _ = load_options ()
let _ = Camtop_outvalue.auto_expand := auto_expand#get
let _ = save_options ()

module C = Configwin
module GSourceView = Camtop_installation.SourceView


let edit_options () =
  let col_oo = C.color ~f: color_caml_output#set
      Camtop_messages.color_ocaml_output color_caml_output#get
  in
  let col_er = C.color ~f: color_error#set
      Camtop_messages.color_error_background color_error#get
  in
  let autoex = C.bool
      ~f: (fun b -> auto_expand#set b; Camtop_outvalue.auto_expand := b)
      Camtop_messages.auto_expand auto_expand#get
  in
  let elements_frame_width = C.string
      ~f: (fun n -> try elements_frame_width#set (int_of_string n) with _ -> ())
      Camtop_messages.elements_frame_width (string_of_int elements_frame_width#get)
  in
  let value_frame_width = C.string
      ~f: (fun n -> try value_frame_width#set (int_of_string n) with _ -> ())
      Camtop_messages.value_frame_width (string_of_int value_frame_width#get)
  in
  let output_frame_height = C.string
      ~f: (fun n -> try output_frame_height#set (int_of_string n) with _ -> ())
      Camtop_messages.output_frame_height (string_of_int output_frame_height#get)
  in
  let (param_syntax, f_restore_syntax) = GSourceView.syntax_highlight_box () in
  let (param_svprops, f_restore_svprops) = GSourceView.source_view_props_box () in
  let sections =
    [
      C.Section ("Base",
		 [ col_oo ; col_er ; autoex ;
		   elements_frame_width ; value_frame_width ;
		   output_frame_height
		 ]) ;
      C.Section ("Source views", [param_svprops]) ;
      C.Section ("Syntax highlighting", [param_syntax]) ;
    ]
  in
  match C.get Camtop_messages.m_configuration sections with
    C.Return_ok -> save_options () ; true
  | _ -> f_restore_syntax (); f_restore_svprops () ; false

(** {2 Misc functions} *)

(** This function returns a file in the form of one string.*)
let input_file_as_string nom =
  try
    let chanin = open_in_bin nom in
    let len = 1024 in
    let s = String.create len in
    let buf = Buffer.create len in
    let rec iter () =
      try
	let n = input chanin s 0 len in
	if n = 0 then
          ()
	else
          (
           Buffer.add_substring buf s 0 n;
           iter ()
          )
      with
	End_of_file -> ()
    in
    iter ();
    close_in chanin;
    Buffer.contents buf
  with
    Sys_error _ ->
      ""

(** {2 Managment of elements} *)

type element_kind =
    Class
  | Class_type
  | Module
  | Module_type
  | Value
  | Type
  | Exception

let string_of_element_kind = function
    Class -> "class"
  | Class_type -> "class type"
  | Module -> "module"
  | Module_type -> "module type"
  | Value -> "val"
  | Type -> "type"
  | Exception -> "exception"

module G = Gmylist

type element = {
    mutable ele_name : string ;
    mutable ele_kind : element_kind ;
    mutable ele_pos : int ; (** position in the output text widget *)
  }

(** The list of defined elements *)
class elements value_box (text : GText.view) =
  object(self)
    inherit [element] G.plist
	`SINGLE
	[ (Some Cam_messages.kind,
	   G.String (fun e -> Glib.Convert.locale_to_utf8 (string_of_element_kind e.ele_kind))) ;

	  (Some Cam_messages.name,
	   G.String (fun e -> Glib.Convert.locale_to_utf8 e.ele_name)) ;
	]
	true

    val mutable data = []
    method compare e1 e2 = compare e1.ele_name e2.ele_name

    method add_element (e : element) =
      try
	data <-
	  (List.filter
	     (fun ele -> ele.ele_name <> e.ele_name or ele.ele_kind <> e.ele_kind)
	     data
	  ) @ [e];
	self#update_data data
      with
	Not_found ->
	  data <- data @ [e];
	  self#update_data data

    method on_deselect _ =
      value_box#clear

    method on_select e =
      let it = text#buffer#get_iter_at_char e.ele_pos in
      let it2 = text#buffer#get_iter_at_char (e.ele_pos + 1) in
      ignore(text#scroll_to_iter it);
      text#buffer#select_range it it2;
      match e.ele_kind with
	Value ->
	  (
	   try
	     let com = e.ele_name^" ;;" in
	     let lexbuf = Lexing.from_string com in
	     let phrase = !Toploop.parse_toplevel_phrase lexbuf in
	     ignore(Toploop.execute_phrase true Format.str_formatter phrase);
	     ignore (Format.flush_str_formatter ())
	   with _ ->
	     ()
	  )
      |	_ -> ()

  end

(** {2 Main box} *)

(** The main box *)
class gui () =
  let val_box = new Camtop_outvalue.out_value_box () in
  let view_edit = GSourceView.create_source_view () in
  let view_results = GSourceView.create_source_view ~editable: false () in
  object (self)
    inherit Camtop_toplevel_base.main
	~file: Camtop_installation.glade_file
	() as base

    val mutable file = (None : string option)

    val value_box = val_box
    val mutable elements = new elements val_box (GSourceView.create_source_view () :> GText.view)

    (** The correctly evaluated phrases, in reverse order *)
    val mutable phrases = ([] : string list)

    method print_out_value (fmt:Format.formatter) ov =
      value_box#update_data ov;
      while Glib.Main.pending () do
	ignore (Glib.Main.iteration false)
      done

    method save_to_file f =
      try
	let oc = open_out f in
	let l = List.rev phrases in
	List.iter
	  (fun s ->	Printf.fprintf oc "%s\n" s)
	  l;
	close_out oc
      with
	Sys_error s ->
	  GToolbox.message_box Cam_messages.error s
      | e ->
	  GToolbox.message_box Cam_messages.error (Printexc.to_string e)

    method save_as () =
      match GToolbox.select_file Camtop_messages.m_save () with
	None -> ()
      |	Some f ->
	  file <- Some f;
	  base#main#set_title (Camtop_messages.software^": "^f);
	  self#save_to_file f

    method save () =
      match file with
	None -> self#save_as ()
      |	Some f -> self#save_to_file f

    (** Get the characters of an error from a standard error message.
       Cut the first line (with the characters) and return also
       the rest of the message. *)
    method get_error_chars s =
      let characters = "Characters" in
      let lenc = String.length characters in
      let len = String.length s in
      try
	if len > lenc then
	  if String.sub s 0 lenc = characters then
	    (
	     let pos = String.index_from s (lenc + 1) '-' in
	     let pos2 = String.index_from s pos '\n' in
	     let s1 = String.sub s (lenc + 1) (pos - lenc - 1) in
	     let pos_start = int_of_string s1 in
	     let s2 = String.sub s (pos + 1) (pos2 - pos - 2) in
	     let pos_end = int_of_string s2 in
	     let rest = String.sub s (pos2+1) (len - pos2 - 1) in
	     (Some (pos_start, pos_end), rest)
	    )
	  else
	    (None, s)
	else
	  (None, s)
      with
	_ ->
	  (None, s)

    method handle_pattern pos pat =
      match pat.ppat_desc with
	Ppat_any -> ()
      |	Ppat_var s ->
	  elements#add_element { ele_name = s ; ele_kind = Value; ele_pos = pos }
      |	Ppat_tuple l ->
	  List.iter (self#handle_pattern pos) l
      |	_ -> ()

    method handle_structure_item pos i =
      match i.pstr_desc with
	Pstr_type l ->
	  let els = List.map
	      (fun (s,_) -> { ele_name = s; ele_kind = Type; ele_pos = pos })
	      l
	  in
	  List.iter elements#add_element els
      |	Pstr_value (_, pat_exp_list) ->
	  List.iter (fun (p,_) -> self#handle_pattern pos p) pat_exp_list
      |	Pstr_exception (s,_)
      |	Pstr_exn_rebind (s,_) ->
	  elements#add_element { ele_name = s; ele_kind = Exception; ele_pos = pos }
      |	Pstr_class l ->
	  let els = List.map
	      (fun pci -> { ele_name = pci.pci_name; ele_kind = Class; ele_pos = pos })
	      l
	  in
	  List.iter elements#add_element els
      |	Pstr_class_type l ->
	  let els = List.map
	      (fun pci -> { ele_name = pci.pci_name; ele_kind = Class_type; ele_pos = pos })
	      l
	  in
	  List.iter elements#add_element els
      |	Pstr_module (s,_) ->
	  elements#add_element { ele_name = s; ele_kind = Module; ele_pos = pos }
      |	Pstr_modtype (s,_) ->
	  elements#add_element { ele_name = s; ele_kind = Module_type; ele_pos = pos }
      |	 _ ->
	  ()

    method handle_phrase pos p =
      match p with
      |	Ptop_dir _ -> ()
      |	Ptop_def items -> List.iter (self#handle_structure_item pos) items

    method set_error_message = function
	None -> wl_error#set_text ""; wl_error#misc#hide ()
      |	Some s ->
	  wl_error#set_text (Glib.Convert.locale_to_utf8 s);
	  wl_error#misc#show ()

    (* Beware that this method does not catch exceptions.*)
    method execute_phrase com phrase =
      let b = view_edit#buffer in
      ignore(Toploop.execute_phrase true Format.str_formatter phrase);
      view_edit#buffer#delete ~start: b#start_iter ~stop: b#end_iter;
      phrases <- com :: phrases ;
      let s = Format.flush_str_formatter () in
      view_results#buffer#place_cursor ~where: view_results#buffer#end_iter;
      self#handle_phrase view_results#buffer#char_count phrase;
      view_results#buffer#insert (Glib.Convert.locale_to_utf8 (com^"\n"));
      let tag = view_results#buffer#create_tag [`FOREGROUND color_caml_output#get] in
      view_results#buffer#insert ~tags: [tag]
	(Glib.Convert.locale_to_utf8 s)

    method execute () =
      let b = view_edit#buffer in
      try
	self#set_error_message None;
	value_box#clear ;
	(*elements#wlist#unselect_all ();*)
	let com = b#get_text ~start: b#start_iter ~stop: b#end_iter () in
	let lexbuf = Lexing.from_string com in
	let phrase = !Toploop.parse_toplevel_phrase lexbuf in
	self#execute_phrase com phrase
      with
	e ->
	  try
	    Errors.report_error Format.str_formatter e;
	    let (pos_opt, mes) = self#get_error_chars (Format.flush_str_formatter ()) in
	    (
	     match pos_opt with
	     | None -> ()
	     | Some (st,en) ->
		 let start = b#get_iter_at_char st in
		 let stop = b#get_iter_at_char en in
		 let tag = b#create_tag
		     [`BACKGROUND color_error#get]
		 in
		 b#apply_tag tag ~start ~stop
	    );
	    self#set_error_message (Some mes)
	  with
	    e2 ->
	      let s = Printexc.to_string e2 in
	      GToolbox.message_box Cam_messages.error s

    method execute_file f =
      try
	let s = input_file_as_string f in
	let lexbuf = Lexing.from_string s in
	let pos = ref 0 in
	(
	 try
	   while true do
	     let phrase = !Toploop.parse_toplevel_phrase lexbuf in
	     let com = String.sub s !pos (lexbuf.Lexing.lex_curr_pos - !pos) in
	     self#execute_phrase com phrase;
	     pos := lexbuf.Lexing.lex_curr_pos;
	   done
	 with
	 | End_of_file -> ()
	 | e ->
	     try
	       Errors.report_error Format.str_formatter e;
	       let (pos_opt, mes) = self#get_error_chars (Format.flush_str_formatter ()) in
	       (
		match pos_opt with
		| None -> ()
		| Some (st,en) ->
		    let b = view_edit#buffer in
		    let start = b#get_iter_at_char st in
		    let stop = b#get_iter_at_char en in
		    let tag = view_results#buffer#create_tag
			[`BACKGROUND color_error#get]
		    in
		    b#apply_tag tag ~start ~stop
(*
		    wtext_in#insert (String.sub s !pos (st - !pos));
		    wtext_in#insert ~background: (`NAME !!color_error) (String.sub s st (en - st));
		    wtext_in#insert (String.sub s en (lexbuf.Lexing.lex_curr_pos - en))
*)
	       );
	       self#set_error_message (Some mes)
	     with
	       e2 ->
		 let s = Printexc.to_string e2 in
		 GToolbox.message_box Cam_messages.error s
	);

      with
      |	Failure s ->
	  GToolbox.message_box Cam_messages.error s

    method load_file f =
      self#execute_file f;
      file <- Some f;
      base#main#set_title (Camtop_messages.software^": "^f)

    method ask_load_file () =
      match GToolbox.select_file Camtop_messages.m_load_file () with
	None -> ()
      |	Some f -> self#load_file f

    method set_widths () =
      base#wf_elements#misc#set_size_request ~width: elements_frame_width#get ();
      base#wf_value#misc#set_size_request ~width: value_frame_width#get () ;
      base#wf_results#misc#set_size_request ~width: output_frame_height#get ();

    method edit_options () =
      if edit_options () then
	(self#set_widths ())

    method quit () = base#main#destroy ()

    method about () =
      GToolbox.message_box
        Camtop_messages.about
        Camtop_messages.software_about

    initializer
      wscroll_edit#add view_edit#coerce;
      wscroll_results#add view_results#coerce;
      elements <- new elements value_box (view_results :> GText.view) ;
      base#main#set_title Camtop_messages.software;
      self#set_error_message None;
      self#set_widths ();
      ignore(base#main#connect#destroy GMain.Main.quit);
      ignore(base#wb_execute#connect#clicked self#execute);

      let handlers =
	[ "on_quit_activate", `Simple self#quit;
          "on_preferences_activate", `Simple self#edit_options;
          "on_about_activate",`Simple self#about;
          "on_open_activate", `Simple self#ask_load_file;
          "on_save_activate", `Simple self#save ;
          "on_save_as_activate", `Simple self#save_as ;
	]
      in
      (* Finalize GUI *)
      Glade.bind_handlers ~extra:handlers ~warn:true self#xml;

(*
      ignore(itemConfig#connect#activate self#edit_options);
      ignore(itemQuit#connect#activate window#destroy);
      ignore(itemSave#connect#activate self#save);
      ignore(itemSaveAs#connect#activate self#save_as);
      ignore(itemLoadFile#connect#activate self#ask_load_file);
      ignore(itemAbout#connect#activate
	       (fun () -> GToolbox.message_box Cam_messages.m_about_top Cam_messages.software_top_about));
*)
      Okey.add view_edit
	~mods: [`CONTROL] GdkKeysyms._Return self#execute;

      wf_elements#add elements#box#coerce;
      wf_value#add value_box#coerce;
      let old_printer = !Toploop.print_out_value in
      Toploop.print_out_value :=
	(fun fmt ov -> old_printer fmt ov ; self#print_out_value fmt ov) ;

(*
      base#main#add_accel_group accelgroup;
*)
      (
       match file with
	 None -> ()
       | Some f -> base#main#set_title (Camtop_messages.software^": "^f)
      );

      base#main#show ()


  end

let usage = "Usage: "^Sys.argv.(0)^" <options> <object-files> [script-file]\noptions are:"

let preload_objects = ref []

let prepare ppf =
  Toploop.set_paths ();
  try List.for_all (Topdirs.load_file ppf) (List.rev !preload_objects)
  with x ->
    try Errors.report_error ppf x; false
    with x ->
      Format.fprintf ppf "Uncaught exception: %s\n" (Printexc.to_string x);
      false

let file_argument name =
  let ppf = Format.err_formatter in
  if Filename.check_suffix name ".cmo" || Filename.check_suffix name ".cma"
  then preload_objects := name :: !preload_objects
  else exit
      (
       if prepare ppf then
	 let gui = new gui () in gui#load_file name; GMain.Main.main (); 0
       else
	 2
      )

open Clflags

let main () =
    Arg.parse [
     "-I", Arg.String(fun dir ->
       let dir = Misc.expand_directory Config.standard_library dir in
       include_dirs := dir :: !include_dirs),
           "<dir>  Add <dir> to the list of include directories";
     "-labels", Arg.Clear classic, " Labels commute (default)";
     "-noassert", Arg.Set noassert, " Do not compile assertion checks";
     "-nolabels", Arg.Set classic, " Ignore labels and do not commute";
     "-nostdlib", Arg.Set no_std_include,
           " do not add default directory to the list of include directories";
     "-principal", Arg.Set principal, " Check principality of type inference";
     "-rectypes", Arg.Set recursive_types, " Allow arbitrary recursive types";
     "-unsafe", Arg.Set fast, " No bound checking on array and string access";
     "-w", Arg.String (Warnings.parse_options false),
           "<flags>  Enable or disable warnings according to <flags>:\n\
       \032    A/a enable/disable all warnings\n\
       \032    C/c enable/disable suspicious comment\n\
       \032    D/d enable/disable deprecated features\n\
       \032    F/f enable/disable partially applied function\n\
       \032    M/m enable/disable overriden method\n\
       \032    P/p enable/disable partial match\n\
       \032    S/s enable/disable non-unit statement\n\
       \032    U/u enable/disable unused match case\n\
       \032    V/v enable/disable hidden instance variable\n\
       \032    X/x enable/disable all other warnings\n\
       \032    default setting is \"Al\" (all warnings but labels enabled)";
     "-warn-error" , Arg.String (Warnings.parse_options true),
       "<flags>  Enable or disable fatal warnings according to <flags>\n\
         \032    (see option -w for the list of flags)\n\
         \032    default setting is a (all warnings are non-fatal)";

     "-dparsetree", Arg.Set dump_parsetree, " (undocumented)";
     "-drawlambda", Arg.Set dump_rawlambda, " (undocumented)";
     "-dlambda", Arg.Set dump_lambda, " (undocumented)";
     "-dinstr", Arg.Set dump_instr, " (undocumented)";
    ] file_argument usage;
  if not (prepare Format.err_formatter) then exit 2;
  let _gui = new gui () in
  GMain.Main.main ()
;;

main ()
