# Copyright (C) 2010 by Canonical Ltd
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

"""Tools for landing branches with Launchpad."""

import os

from launchpadlib.launchpad import Launchpad
from launchpadlib.uris import (
    DEV_SERVICE_ROOT, EDGE_SERVICE_ROOT,
    LPNET_SERVICE_ROOT, STAGING_SERVICE_ROOT)
from lazr.uri import URI
from bzrlib import msgeditor
from bzrlib.errors import BzrCommandError
from bzrlib.plugins.pqm import pqm_submit
from bzrlib import smtp_connection


class MissingReviewError(Exception):
    """Raised when we try to get a review message without enough reviewers."""


class LaunchpadBranchLander:

    name = 'launchpad-branch-lander'
    cache_dir = '~/.launchpadlib/cache'

    def __init__(self, launchpad):
        self._launchpad = launchpad

    @classmethod
    def load(cls, service_root=EDGE_SERVICE_ROOT):
        # XXX: JonathanLange 2009-09-24: No unit tests.
        cache_dir = os.path.expanduser(cls.cache_dir)
        # XXX: JonathanLange 2009-09-24 bug=435813: If cached data invalid,
        # there's no easy way to delete it and try again.
        launchpad = Launchpad.login_with(cls.name, service_root, cache_dir)
        return cls(launchpad)

    def load_merge_proposal(self, mp_url):
        """Get the merge proposal object for the 'mp_url'."""
        # XXX: JonathanLange 2009-09-24: No unit tests.
        web_mp_uri = URI(mp_url)
        api_mp_uri = self._launchpad._root_uri.append(
            web_mp_uri.path.lstrip('/'))
        return MergeProposal(self._launchpad.load(str(api_mp_uri)))

    def get_lp_branch(self, branch):
        """Get the launchpadlib branch based on a bzr branch."""
        # First try the public branch.
        branch_url = branch.get_public_branch()
        if branch_url:
            lp_branch = self._launchpad.branches.getByUrl(
                url=branch_url)
            if lp_branch is not None:
                return lp_branch
        # If that didn't work try the push location.
        branch_url = branch.get_push_location()
        if branch_url:
            lp_branch = self._launchpad.branches.getByUrl(
                url=branch_url)
            if lp_branch is not None:
                return lp_branch
        raise BzrCommandError(
            "No public branch could be found.  Please re-run and specify "
            "the URL for the merge proposal.")

    def get_merge_proposal_from_branch(self, branch):
        """Get the merge proposal from the branch."""

        lp_branch = self.get_lp_branch(branch)
        proposals = lp_branch.landing_targets
        if len(proposals) == 0:
            raise BzrCommandError(
                "The public branch has no source merge proposals.  "
                "You must have a merge proposal before attempting to "
                "land the branch.")
        elif len(proposals) > 1:
            raise BzrCommandError(
                "The public branch has multiple source merge proposals.  "
                "You must provide the URL to the one you wish to use.")
        return MergeProposal(proposals[0])


class MergeProposal:
    """Wrapper around launchpadlib `IBranchMergeProposal` for landing."""

    def __init__(self, mp):
        """Construct a merge proposal.

        :param mp: A launchpadlib `IBranchMergeProposal`.
        """
        self._mp = mp
        self._launchpad = mp._root

    @property
    def source_branch(self):
        """The push URL of the source branch."""
        return str(self._get_push_url(self._mp.source_branch)).rstrip('/')

    @property
    def target_branch(self):
        """The push URL of the target branch."""
        return str(self._get_push_url(self._mp.target_branch)).rstrip('/')

    @property
    def commit_message(self):
        """The commit message specified on the merge proposal."""
        return self._mp.commit_message

    @property
    def is_approved(self):
        """Is this merge proposal approved for landing."""
        return self._mp.queue_status == 'Approved'

    def get_stakeholder_emails(self):
        """Return a collection of people who should know about branch landing.

        Used to determine who to email with the ec2 test results.

        :return: A set of `IPerson`s.
        """
        # XXX: JonathanLange 2009-09-24: No unit tests.
        return set(
            map(get_email,
                [self._mp.source_branch.owner, self._launchpad.me]))

    def get_reviews(self):
        """Return a dictionary of all Approved reviews.

        Used to determine who has actually approved a branch for landing. The
        key of the dictionary is the type of review, and the value is the list
        of people who have voted Approve with that type.

        Common types include 'code', 'db', 'ui' and of course `None`.
        """
        reviews = {}
        for vote in self._mp.votes:
            comment = vote.comment
            if comment is None or comment.vote != "Approve":
                continue
            reviewers = reviews.setdefault(vote.review_type, [])
            reviewers.append(vote.reviewer)
        return reviews

    def get_bugs(self):
        """Return a collection of bugs linked to the source branch."""
        return self._mp.source_branch.linked_bugs

    def _get_push_url(self, branch):
        """Return the push URL for 'branch'.

        This function is a work-around for Launchpad's lack of exposing the
        branch's push URL.

        :param branch: A launchpadlib `IBranch`.
        """
        # XXX: JonathanLange 2009-09-24: No unit tests.
        return branch.composePublicURL(scheme="bzr+ssh")

    def get_commit_message(self, commit_text, testfix=False):
        """Get the Launchpad-style commit message for a merge proposal."""
        reviews = self.get_reviews()
        bugs = self.get_bugs()
        if testfix:
            testfix = '[testfix]'
        else:
            testfix = ''
        return '%s%s%s %s' % (
            testfix,
            get_reviewer_clause(reviews),
            get_bugs_clause(bugs),
            commit_text)


class Submitter(object):

    def __init__(self, branch, location):
        self.branch = branch
        self.config = self.branch.get_config()
        self.mail_to = self.config.get_user_option('pqm_email')
        if not self.mail_to:
            raise pqm_submit.NoPQMSubmissionAddress(branch)
        self.mail_from = self.config.get_user_option('pqm_user_email')
        if not self.mail_from:
            self.mail_from = self.config.username()
        self.lander = LaunchpadBranchLander.load()
        self.location = location

    def submission(self, mp):
        submission = pqm_submit.PQMSubmission(self.branch,
            mp.source_branch, mp.target_branch, '')
        return submission

    @staticmethod
    def check_submission(submission):
        submission.check_tree()
        submission.check_public_branch()

    @staticmethod
    def set_message(submission, mp):
        pqm_command = ''.join(submission.to_lines())
        commit_message = mp.commit_message or ''
        start_message = mp.get_commit_message(commit_message)
        message = msgeditor.edit_commit_message(
            'pqm command:\n%s' % pqm_command,
            start_message=start_message).rstrip('\n')
        submission.message = message.encode('utf-8')

    def run(self, outf):
        if self.location is None:
            mp = self.lander.get_merge_proposal_from_branch(self.branch)
        else:
            mp = self.lander.load_merge_proposal(self.location)
        submission = self.submission(mp)
        self.check_submission(submission)
        self.set_message(submission, mp)
        email = submission.to_email(self.mail_from, self.mail_to)
        if outf is not None:
            outf.write(email.as_string())
        else:
            smtp_connection.SMTPConnection(self.config).send_email(email)


def get_email(person):
    """Get the preferred email address for 'person'."""
    email_object = person.preferred_email_address
    # XXX: JonathanLange 2009-09-24 bug=319432: This raises a very obscure
    # error when the email address isn't set. e.g. with name12 in the sample
    # data. e.g. "httplib2.RelativeURIError: Only absolute URIs are allowed.
    # uri = tag:launchpad.net:2008:redacted".
    return email_object.email


def get_bugs_clause(bugs):
    """Return the bugs clause of a commit message.

    :param bugs: A collection of `IBug` objects.
    :return: A string of the form "[bug=A,B,C]".
    """
    if not bugs:
        return ''
    return '[bug=%s]' % ','.join(str(bug.id) for bug in bugs)


def get_reviewer_handle(reviewer):
    """Get the handle for 'reviewer'.

    The handles of reviewers are included in the commit message for Launchpad
    changes. Historically, these handles have been the IRC nicks. Thus, if
    'reviewer' has an IRC nickname for Freenode, we use that. Otherwise we use
    their Launchpad username.

    :param reviewer: A launchpadlib `IPerson` object.
    :return: unicode text.
    """
    irc_handles = reviewer.irc_nicknames
    for handle in irc_handles:
        if handle.network == 'irc.freenode.net':
            return handle.nickname
    return reviewer.name


def _comma_separated_names(things):
    """Return a string of comma-separated names of 'things'.

    The list is sorted before being joined.
    """
    return ','.join(sorted(thing.name for thing in things))


def get_reviewer_clause(reviewers):
    """Get the reviewer section of a commit message, given the reviewers.

    :param reviewers: A dict mapping review types to lists of reviewers, as
        returned by 'get_reviews'.
    :return: A string like u'[r=foo,bar][ui=plop]'.
    """
    # If no review type is specified it is assumed to be a code review.
    code_reviewers = reviewers.get(None, [])
    code_reviewers.extend(reviewers.get('', []))
    ui_reviewers = []
    rc_reviewers = []
    for review_type, reviewer in reviewers.items():
        if review_type is None:
            continue
        if review_type == '':
            code_reviewers.extend(reviewer)
        if 'code' in review_type or 'db' in review_type:
            code_reviewers.extend(reviewer)
        if 'ui' in review_type:
            ui_reviewers.extend(reviewer)
        if 'release-critical' in review_type:
            rc_reviewers.extend(reviewer)
    if not code_reviewers:
        raise MissingReviewError("Need approved votes in order to land.")
    if ui_reviewers:
        ui_clause = _comma_separated_names(ui_reviewers)
    else:
        ui_clause = 'none'
    if rc_reviewers:
        rc_clause = (
            '[release-critical=%s]' % _comma_separated_names(rc_reviewers))
    else:
        rc_clause = ''
    return '%s[r=%s][ui=%s]' % (
        rc_clause, _comma_separated_names(code_reviewers), ui_clause)


def get_bazaar_host(api_root):
    """Get the Bazaar service for the given API root."""
    # XXX: JonathanLange 2009-09-24 bug=435803: This is only needed because
    # Launchpad doesn't expose the push URL for branches.
    if api_root == EDGE_SERVICE_ROOT:
        return 'bazaar.launchpad.net'
    elif api_root == DEV_SERVICE_ROOT:
        return 'bazaar.launchpad.dev'
    elif api_root == STAGING_SERVICE_ROOT:
        return 'bazaar.staging.launchpad.net'
    elif api_root == LPNET_SERVICE_ROOT:
        return 'bazaar.launchpad.net'
    else:
        raise ValueError(
            'Cannot determine Bazaar host. "%s" not a recognized Launchpad '
            'API root.' % (api_root,))
