package buoy.widget;

import buoy.event.*;
import buoy.xml.*;
import buoy.xml.delegate.*;
import javax.swing.*;
import javax.swing.event.*;

/**
 * A BSlider is a Widget that allows the user to select a single value by dragging a "thumb"
 * along a bar.  It can optionally show tick marks, labels, or both along the bar.
 * <p>
 * In addition to the event types generated by all Widgets, BSliders generate the following event types:
 * <ul>
 * <li>{@link buoy.event.ValueChangedEvent ValueChangedEvent}</li>
 * </ul>
 *
 * @author Peter Eastman
 */

public class BSlider extends Widget
{
  private int suppressEvents;

  public static final Orientation HORIZONTAL = new Orientation(SwingConstants.HORIZONTAL);
  public static final Orientation VERTICAL = new Orientation(SwingConstants.VERTICAL);

  static
  {
    WidgetEncoder.setPersistenceDelegate(Orientation.class, new StaticFieldDelegate(BSlider.class));
  }
  
  /**
   * Create a new vertical BSlider.
   */
  
  public BSlider()
  {
    this(0, 0, 100, VERTICAL);
  }
  
  /**
   * Create a new BSlider.
   *
   * @param value        the BSlider's initial value
   * @param minimum      the minimum value for the range represented by the BSlider
   * @param maximum      the maximum value for the range represented by the BSlider
   * @param orientation  defines how the BSlider should be drawn and positioned.  This should be
   *                     HORIZONTAL or VERTICAL.
   */
  
  public BSlider(int value, int minimum, int maximum, Orientation orientation)
  {
    component = createComponent(orientation);
    getComponent().setModel(new DefaultBoundedRangeModel(value, 0, minimum, maximum));
    getComponent().addChangeListener(new ChangeListener() {
      public void stateChanged(ChangeEvent ev)
      {
        if (suppressEvents == 0)
          dispatchEvent(new ValueChangedEvent(BSlider.this, getComponent().getValueIsAdjusting()));
      }
    });
    setMajorTickSpacing(20);
    setMinorTickSpacing(5);
  }
  
  /**
   * Create the JSlider which serves as this Widget's Component.  This method is protected so that
   * subclasses can override it.
   *
   * @param orientation  defines how the BSlider should be drawn and positioned.  This should be
   *                     HORIZONTAL or VERTICAL.
   */
  
  protected JSlider createComponent(Orientation orientation)
  {
    return new JSlider(orientation.value);
  }

  public JSlider getComponent()
  {
    return (JSlider) component;
  }

  /**
   * Get the minimum value of the range represented by this BSlider.
   */
  
  public int getMinimum()
  {
    return getComponent().getMinimum();
  }
  
  /**
   * Set the minimum value of the range represented by this BSlider.
   */
  
  public void setMinimum(int value)
  {
    getComponent().setMinimum(value);
  }
  
  /**
   * Get the maximum value of the range represented by this BSlider.
   */
  
  public int getMaximum()
  {
    return getComponent().getMaximum();
  }
  
  /**
   * Set the maximum value of the range represented by this BSlider.
   */
  
  public void setMaximum(int value)
  {
    getComponent().setMaximum(value);
  }
  
  /**
   * Get the current value of this BSlider.
   */
  
  public int getValue()
  {
    return getComponent().getValue();
  }
  
  /**
   * Set the current value of this BSlider.
   */
  
  public void setValue(int value)
  {
    try
    {
      suppressEvents++;
      getComponent().setValue(value);
    }
    finally
    {
      suppressEvents--;
    }
  }
  
  /**
   * Get the orientation (HORIZONTAL or VERTICAL) of this BSlider.
   */
  
  public Orientation getOrientation()
  {
    int orient = getComponent().getOrientation();
    return (orient == HORIZONTAL.value ? HORIZONTAL : VERTICAL);
  }
  
  /**
   * Set the orientation (HORIZONTAL or VERTICAL) of this BSlider.
   */
  
  public void setOrientation(Orientation orientation)
  {
    getComponent().setOrientation(orientation.value);
    invalidateSize();
  }
  
  /**
   * Get the spacing between major tick marks on the slider.
   */
  
  public int getMajorTickSpacing()
  {
    return getComponent().getMajorTickSpacing();
  }
  
  /**
   * Set the spacing between major tick marks on the slider.
   * <p>
   * The major tick spacing can never be smaller than the minor tick spacing.  If the current
   * minor tick spacing is greater than the value passed to this method, then the minor tick
   * spacing will also be set to the same value.
   */
  
  public void setMajorTickSpacing(int spacing)
  {
    JSlider slider = getComponent();
    slider.setMajorTickSpacing(spacing);
    if (spacing < slider.getMinorTickSpacing())
      slider.setMinorTickSpacing(spacing);
    slider.setLabelTable(slider.createStandardLabels(spacing));
  }
  
  /**
   * Get the spacing between minor tick marks on the slider.
   */
  
  public int getMinorTickSpacing()
  {
    return getComponent().getMinorTickSpacing();
  }
  
  /**
   * Set the spacing between minor tick marks on the slider.
   * <p>
   * The minor tick spacing can never be larger than the major tick spacing.  If the current
   * major tick spacing is less than the value passed to this method, then the major tick
   * spacing will also be set to the same value.
   */
  
  public void setMinorTickSpacing(int spacing)
  {
    JSlider slider = getComponent();
    slider.setMinorTickSpacing(spacing);
    if (spacing > slider.getMajorTickSpacing())
      slider.setMajorTickSpacing(spacing);
  }
  
  /**
   * Get whether tick marks are shown on the slider.
   */
  
  public boolean getShowTicks()
  {
    return getComponent().getPaintTicks();
  }
  
  /**
   * Set whether tick marks are shown on the slider.
   */
  
  public void setShowTicks(boolean show)
  {
    getComponent().setPaintTicks(show);
  }
  
  /**
   * Get whether labels are shown on the slider at the major tick positions.
   */
  
  public boolean getShowLabels()
  {
    return getComponent().getPaintLabels();
  }
  
  /**
   * Set whether labels are shown on the slider at the major tick positions.
   */
  
  public void setShowLabels(boolean show)
  {
    getComponent().setPaintLabels(show);
  }
  
  /**
   * Get whether the thumb of the slider should always snap to the nearest tick position.
   */
  
  public boolean getSnapToTicks()
  {
    return getComponent().getSnapToTicks();
  }
  
  /**
   * Set whether the thumb of the slider should always snap to the nearest tick position.
   */
  
  public void setSnapToTicks(boolean snap)
  {
    getComponent().setSnapToTicks(snap);
  }
  
  /**
   * This inner class represents an orientation for the slider.
   */
  
  public static class Orientation
  {
    protected int value;
    
    private Orientation(int value)
    {
      this.value = value;
    }
  }
}
