# bug-triage -- bug triage and forward tool.
# Copyright (C) 2007  Gustavo R. Montesino
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

import pygtk
pygtk.require("2.0")
import gobject
import gtk

from btsutils import BugStatus
from btsutils.bug import Bug
from msg_exception import errorhandler

class bug_store(gtk.ListStore):
    """Debian Bugs TreeModel

    ListStore class for use with btsutils"""

    COLUMN_BUG = 0
    COLUMN_PACKAGE = 1
    COLUMN_SUMMARY = 2
    COLUMN_SEVERITY = 3
    COLUMN_SUBMITTER = 4
    COLUMN_TAGS = 5
    COLUMN_STATUS = 6
    COLUMN_USERTAGS = 7
    COLUMN_FORWARDED = 8
    COLUMN_URL = 9

    @errorhandler
    def __init__(self):
        """Initialise the ListStore instance"""

        gtk.ListStore.__init__(self, 
            gobject.TYPE_STRING, # Bug Number
            gobject.TYPE_STRING, # Package
            gobject.TYPE_STRING, # Summary
            gobject.TYPE_STRING, # Severity
            gobject.TYPE_STRING, # Submitter
            gobject.TYPE_STRING, # Tags
            gobject.TYPE_STRING, # Status
            gobject.TYPE_STRING, # Usertags
            gobject.TYPE_STRING, # Forwarded
            gobject.TYPE_STRING  # URL
            )

        self.set_sort_func(self.COLUMN_SEVERITY, self.sort_severity)
        self.set_sort_func(self.COLUMN_STATUS, self.sort_status)

        self.set_sort_column_id(self.COLUMN_SEVERITY, gtk.SORT_DESCENDING)

    @errorhandler
    def bug_to_list(self, bug):
        """Transforms a bug in a list proper to use with ListStore methods"""

        row = []
        row.append(bug.getBug())
        row.append(bug.getPackage())
        row.append(bug.getSummary())
        row.append(bug.getSeverity())
        row.append(bug.getSubmitter())
        row.append(", ".join(bug.getTags()))
        row.append(bug.getStatus())
        row.append(", ".join(bug.getUserTags()))
        row.append(bug.getForwarded())
        row.append(bug.getURL())
        return row
    
    @errorhandler
    def insert(self, position, bug):
        """Wrapper around gtk.ListStore insert to use an Bug object"""

        return gtk.ListStore.insert(self, position, self.bug_to_list(bug))
    
    @errorhandler
    def prepend(self, bug):
        """Wrapper around gtk.ListStore prepend to use an Bug object"""

        return gtk.ListStore.prepend(self, self.bug_to_list(bug))
    
    @errorhandler
    def append(self, bug):
        """Wrapper around gtk.ListStore append to use an Bug object"""

        return gtk.ListStore.append(self, self.bug_to_list(bug))

    @errorhandler
    def get(self, iter):
        """Returns the bug pointed by a TreeIter"""

        bug = Bug()
        bug.setBug(gtk.ListStore.get(self, iter, self.COLUMN_BUG)[0])
        bug.setPackage(gtk.ListStore.get(self, iter, self.COLUMN_PACKAGE)[0])
        bug.setSummary(gtk.ListStore.get(self, iter, self.COLUMN_SUMMARY)[0])
        bug.setSeverity(gtk.ListStore.get(self, iter, self.COLUMN_SEVERITY)[0])
        bug.setSubmitter(gtk.ListStore.get(self, iter, self.COLUMN_SUBMITTER)[0])
        bug.setStatus(gtk.ListStore.get(self, iter, self.COLUMN_STATUS)[0])
        bug.setForwarded(gtk.ListStore.get(self, iter, self.COLUMN_FORWARDED)[0])
        bug.setURL(gtk.ListStore.get(self, iter, self.COLUMN_URL)[0])

        tags = gtk.ListStore.get(self, iter, self.COLUMN_TAGS)[0].split(', ')
        if tags != ['']:
            bug.setTags(tags)

        usertags = gtk.ListStore.get(self, iter, self.COLUMN_USERTAGS)[0].split(', ')
        if usertags != ['']:
            bug.setUserTags(tags)

        return bug

    @errorhandler
    def sort_severity(self, model, iter1, iter2, data=None):
        """Sort method to sort bugs by severity"""

        severity = dict(wishlist=0,minor=1,normal=2,important=3,serious=4,grave=5,critical=6)

        try:
            sev1 = severity[model.get_value(iter1, self.COLUMN_SEVERITY)]
            sev2 = severity[model.get_value(iter2, self.COLUMN_SEVERITY)]

        except KeyError:
            # GTK calls this function one time for every column of an added bug, which means
            # we will get some calls before severity is set :/
            
            return 0

        if sev1 < sev2:
            return -1
        elif sev1 == sev2:
            return 0
        else:
            return 1

    @errorhandler
    def sort_status(self, model, iter1, iter2, data=None):
        """Sort method to sort bugs by status"""

        status1 = model.get_value(iter1, self.COLUMN_STATUS)
        status1 = BugStatus.VALID.index(status1)

        status2 = model.get_value(iter2, self.COLUMN_STATUS)
        status2 = BugStatus.VALID.index(status2)

        if status1 < status2:
            return -1
        elif status1 == status2:
            return 0
        else:
            return 1

# vim: tabstop=4 expandtab shiftwidth=4
