/***************************************************************************
                          bridgecfg.cpp  -  description
                             -------------------
    begin                : Mon Nov 18 2002
    copyright            : (C) 2002 by Ken Schenke
    email                : kenschenke at yahoo dot com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   In addition, as a special exception, Ken Schenke gives permission to  *
 *   link the code of this program with the Qt non-commercial edition (or  *
 *   with modified versions of the Qt non-commercial edition that use the  *
 *   same license as the Qt non-commercial edition, and distribute linked  *
 *   combinations including the two.  You must obey the GNU General Public *
 *   License in all respects for all of the code used other than the Qt    *
 *   Non-Commercial edition.  If you modify this file, you may extend this *
 *   exception to your version of the file, but you are not obligated to   *
 *   do so.  If you do not wish to do so, delete this exception statement  *
 *   from your version.                                                    *
 *                                                                         *
 ***************************************************************************/

#include "bridgecfg.h"
#include "browserlist.h"

#ifdef Q_WS_X11
#include "config.h"
#endif

#include "xmlparser.h"
#include "xbelwrite.h"
#include "xbelread.h"

#include <qdir.h>
#include <qmessagebox.h>
#include <iostream>
#include <fstream>

#ifdef _WIN32
#include <shlobj.h>
#include <io.h>
#else
#include <unistd.h>
#endif

using namespace std;

extern BROWSERLIST BrowserList[];

/***************************************************************************
 *                                                                         *
 *   Constants                                                             *
 *                                                                         *
 ***************************************************************************/

#define SAXSTATE_BRIDGECFG		(SAXSTATE_USER+ 1)
#define SAXSTATE_BROWSER		(SAXSTATE_USER+ 2)
#define SAXSTATE_DESCRIP		(SAXSTATE_USER+ 3)
#define SAXSTATE_NAME			(SAXSTATE_USER+ 4)
#define SAXSTATE_BOOKMARKS		(SAXSTATE_USER+ 5)
#define SAXSTATE_ORDINAL		(SAXSTATE_USER+ 6)
#define SAXSTATE_READONLY		(SAXSTATE_USER+ 7)
#define SAXSTATE_BROWSERNUM     (SAXSTATE_USER+ 8)
#define SAXSTATE_SOURCES        (SAXSTATE_USER+ 9)
#define SAXSTATE_SOURCE         (SAXSTATE_USER+10)
#define SAXSTATE_DESTINATIONS   (SAXSTATE_USER+11)
#define SAXSTATE_DESTINATION    (SAXSTATE_USER+12)

/***************************************************************************
 *                                                                         *
 *   XML Parser Call-back Functions                                        *
 *                                                                         *
 ***************************************************************************/

static bool xmlStartBridgeCfg(void *, const xmlChar *, const xmlChar **, short);
static bool xmlStartBrowser(void *, const xmlChar *, const xmlChar **, short);
static void xmlEndBookmarks(void *, const xmlChar *, const xmlChar *, short, short);
static void xmlEndBrowser(void *, const xmlChar *, const xmlChar *, short, short);
static void xmlEndBrowserNum(void *, const xmlChar *, const xmlChar *, short, short);
static void xmlEndDescrip(void *, const xmlChar *, const xmlChar *, short, short);
static void xmlEndDestination(void *, const xmlChar *, const xmlChar *, short, short);
static void xmlEndName(void *, const xmlChar *, const xmlChar *, short, short);
static void xmlEndOrdinal(void *, const xmlChar *, const xmlChar *, short, short);
static void xmlEndReadonly(void *, const xmlChar *, const xmlChar *, short, short);
static void xmlEndSource(void *, const xmlChar *, const xmlChar *, short, short);

/***************************************************************************
 *                                                                         *
 *   XML Handlers                                                          *
 *                                                                         *
 *   These handlers tell the XML Parser (see xmlparser.h) what callback    *
 *   functions to call for each XML element encountered.  They also define *
 *   the states used for each element.                                     *
 *                                                                         *
 ***************************************************************************/

static ELEMHANDLER xmlHandlers[] = {
	{
		SAXSTATE_BRIDGECFG,
		"bridgecfg",
		xmlStartBridgeCfg,
		NULL,
	},{
		SAXSTATE_BROWSER,
		"browser",
		xmlStartBrowser,
		xmlEndBrowser,
	},{
		SAXSTATE_DESCRIP,
		"descrip",
		NULL,
		xmlEndDescrip,
	},{
		SAXSTATE_NAME,
		"name",
		NULL,
		xmlEndName,
	},{
		SAXSTATE_BOOKMARKS,
		"bookmarks",
		NULL,
		xmlEndBookmarks,
	},{
		SAXSTATE_ORDINAL,
		"ordinal",
		NULL,
		xmlEndOrdinal,
	},{
		SAXSTATE_READONLY,
		"readonly",
		NULL,
		xmlEndReadonly,
	},{
		SAXSTATE_BROWSERNUM,
		"browsernum",
		NULL,
		xmlEndBrowserNum,
	},{
		SAXSTATE_SOURCES,
		"sources",
		NULL,
		NULL,
	},{
		SAXSTATE_SOURCE,
		"source",
		NULL,
		xmlEndSource,
	},{
		SAXSTATE_DESTINATIONS,
		"destinations",
		NULL,
		NULL,
	},{
		SAXSTATE_DESTINATION,
		"destination",
		NULL,
		xmlEndDestination,
	},
	// this element marks the end of the list
	{ 0, NULL, NULL, NULL }
};

/***************************************************************************
 *                                                                         *
 *   Structure Definitions                                                 *
 *                                                                         *
 ***************************************************************************/

typedef struct {
	vector<BrowserCfg> *browsers;
	vector<BRWSNUM> *sources;
	vector<BRWSNUM> *destinations;
	BrowserCfg  *browser;
} SAXDATA;

/***************************************************************************
 *                                                                         *
 *   Function Prototypes                                                   *
 *                                                                         *
 ***************************************************************************/

static	void	xmlStartDocument(void *);
#ifdef Q_WS_WIN
static HKEY GetRegKey(HKEY Parent, const char *Path);
#endif

/***************************************************************************
 *                                                                         *
 *   GetRegKey()                                                           *
 *                                                                         *
 *   Parameters:                                                           *
 *      HKEY Parent                                                        *
 *      const char *Path                                                   *
 *   Return:                                                               *
 *      HKEY (NULL if error)                                               *
 *   Description:                                                          *
 *      This function is a helper to Windows' RegOpenKeyEx() which is very *
 *      tedious to use if the caller wants to open a key several levels    *
 *      deep in the registry.  This function can take a string in Path     *
 *      containing key levels separated by backslashes.  For example,      *
 *      "Software\\Microsoft\\Windows\\CurrentVersion".  It recurses       *
 *      through the string, finally returning the key at the end of the    *
 *      string.  As it opens keys and subkeys, it closes the parent keys   *
 *      as soon as a child key is opened.  All keys are opened read-only.  *
 *                                                                         *
 ***************************************************************************/

#ifdef Q_WS_WIN
static HKEY GetRegKey(HKEY Parent, const char *Path)
{
	const char	*p;
	HKEY		hkey;
	char		*key;
	short		len;
	LONG		retval;

	// Look for the first backslash

	if(p = strchr(Path, '\\'))
		len = p - Path;
	else
		len = strlen(Path);

	if((key = new char[len+1]) == NULL)
		return NULL;
	memcpy(key, Path, len);
	key[len] = 0;

	retval = RegOpenKeyEx(Parent, key, 0, KEY_READ, &hkey);
	delete [] key;
	if(retval != ERROR_SUCCESS)
		return NULL;

	if(p)
	{
		HKEY subkey = GetRegKey(hkey, p+1);
		RegCloseKey(hkey);
		return subkey;
	}
	else
		return hkey;
}
#endif

/***************************************************************************
 *                                                                         *
 *   EncodeString()                                                        *
 *                                                                         *
 *   Parameters:                                                           *
 *      const char *str                                                    *
 *   Return:                                                               *
 *      encoded string                                                     *
 *   Description:                                                          *
 *      This function changes XML's special characters into XML entities.  *
 *                                                                         *
 ***************************************************************************/

static string EncodeString(const char *str)
{
	int	len = strlen(str);

	string outstr = "";
	for(int i=0; i<len; i++)
	{
		switch(str[i])
		{
		case  '<':	outstr += "&lt;";	break;
		case  '>':	outstr += "&gt;";	break;
		case  '&':	outstr += "&amp;";	break;
		case '\"':	outstr += "&quot;";	break;
		case '\'':	outstr += "&apos;";	break;
		default:	outstr += str[i];	break;
		}
	}

	return outstr;
}

/***************************************************************************
 *                                                                         *
 *   EncodeString()                                                        *
 *                                                                         *
 *   Parameters:                                                           *
 *      const string &str                                                  *
 *   Return:                                                               *
 *      encoded string                                                     *
 *   Description:                                                          *
 *      Overloaded version of above function.                              *
 *                                                                         *
 ***************************************************************************/

static string EncodeString(const string &str)
{
	return EncodeString(str.c_str());
}

/***************************************************************************
 *                                                                         *
 *   BridgeCfg::findConfigFile()   (X11 Version)                           *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function attempts to locate BookmarkBridge's configuration    *
 *      files.  It first looks for the directory holding the files.  If    *
 *      directory does not exist, it creates it.  It then looks to see if  *
 *      the directory contains the files.                                  *
 *                                                                         *
 ***************************************************************************/

#if defined(Q_WS_X11)
void BridgeCfg::findConfigFile(void) throw(BkException)
{
	extern QString gConfigFile;
	extern QString gTreeFile;
	string	configPath;

	configFileExists = false;
	xbelFileExists = false;

	if(gConfigFile == QString::null)
	{
		// figure out where the configuration directory resides

		configPath = string(static_cast<const char *>(QDir::homeDirPath())) + "/.bookmarkbridge";
		QDir homeDir(QDir::homeDirPath());

		// create it if necessary

		if( homeDir.exists(configPath.c_str()) == false
		 && homeDir.mkdir(configPath.c_str()) == false)
		{
			QString msg;

			msg.sprintf("Unable to Create Configuration Directory\n%s",
				configPath.c_str());
			BKEXCEPT(msg);
		}

		// the configuration file

		configFileName = configPath + "/bridgeconf.xml";

		QDir configDir(configPath.c_str());
		configFileExists = configDir.exists("bridgeconf.xml");
	}
	else
	{
		configFileName = static_cast<const char *>(gConfigFile);
		if(access(configFileName.c_str(), 0) == 0)
			configFileExists = true;
	}

	// the bookmark tree file (XBEL)

	if(gTreeFile == QString::null)
		xbelFileName = configPath + "/bookmarks.xbel";
	else
		xbelFileName = static_cast<const char *>(gTreeFile);
	if(access(xbelFileName.c_str(), 0) == 0)
		xbelFileExists = true;

	datadir = DATADIR;
}
#elif defined(Q_WS_WIN)
void BridgeCfg::findConfigFile(void) throw(BkException)
{
	char	appdata[_MAX_PATH];
	extern QString gConfigFile;
	extern QString gTreeFile;

	QString	configPath;

	configFileExists = false;
	xbelFileExists = false;

	if(gConfigFile == QString::null)
	{
		InitMemberVars();

		// figure out where the configuration directory resides

		configPath.sprintf("%s\\BookmarkBridge", appdatadir.c_str());
		QDir appdataDir(appdatadir.c_str());

		// create it if necessary

		if( appdataDir.exists(configPath) == false
		 && appdataDir.mkdir(configPath) == false)
		{
			QString msg;

			msg.sprintf("Unable to Create Configuration Directory\n%s",
				static_cast<const char *>(configPath));
			BKEXCEPT(msg);
		}

		// the configuration file

		configFileName = configPath + "/bridgeconf.xml";

		QDir configDir(configPath);
		configFileExists = configDir.exists("bridgeconf.xml");
	}
	else
	{
		configFileName = gConfigFile;
		if(access(configFileName.c_str(), 0) == 0)
			configFileExists = true;
	}

	// the bookmark tree file (XBEL)

	if(gTreeFile == QString::null)
		xbelFileName = configPath + "/bookmarks.xbel";
	else
		xbelFileName = gTreeFile;
	if(access(xbelFileName.c_str(), 0) == 0)
		xbelFileExists = true;

	// the data directory

	if(GetModuleFileName(NULL, appdata, sizeof(appdata)) == 0)
		BKEXCEPT("Unable to Determine Location of Data Files");
	char *ch = strrchr(appdata, '\\');
	if(ch == NULL)
		BKEXCEPT("Malformed Path Name of Executable");
	*ch = 0;
	datadir = appdata;
}
#else
#error "Must Define Code to Find Config File"
#endif

/***************************************************************************
 *                                                                         *
 *   BridgeCfg::InitDefaults()                                             *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function is called when the configuration file was not found. *
 *      It asks each browser if it can detect an installation of its       *
 *      particular browser on the system.  If it can, it then adds that    *
 *      browser to the configuration and adds it as a source and           *
 *      destination.                                                       *
 *                                                                         *
 ***************************************************************************/

void BridgeCfg::InitDefaults(void) throw(BkException)
{
	// Look at each of the known browsers and attempt to
	// detect their presence on the system.

	int			i, p;
	BrowserBk	*browser;
	QString		temp;
	QStringList	pathlist;

	for(i=0; ;i++)
	{
		// are we at the end of the browsers?
		
		if(BrowserList[i].Name==NULL || BrowserList[i].Factory==NULL)
			break;

		// create an instance of the browser class
		
		browser = BrowserList[i].Factory();
		if(browser == NULL)
			BKEXCEPT("Unable to Allocate Memory");

		// see if it can detect an installation of the browser
		
		if(	!browser->DetectBrowser(*this, pathlist)
		 ||	pathlist.count() < 1)
//		 ||	!browser->AreBookmarksValid(path))
		{
				delete browser;
				continue;
		}

		// since it found at least one, create a configuration entry for each

		for(p=0; p<static_cast<int>(pathlist.count()); p++)
		{
			BrowserCfg *cfg = new BrowserCfg;

			// This test should never happen since the DetectBrowser()
			// function is supposed to have already validated each
			// bookmark location before adding it to the list.

			if(!browser->AreBookmarksValid(pathlist[p]))
				continue;
			
			cfg->setBrowser(BrowserList[i].Name);
			cfg->setBrowserNum(i+1);
			temp.sprintf("%s Bookmarks", BrowserList[i].Name);
			cfg->setDescrip(temp);
			cfg->setLocation(pathlist[p]);
			cfg->setReadOnly(browser->AreBookmarksReadonly(pathlist[p]));

			// add it to the configuration and to the sources and destinations
			
			browsers.push_back(*cfg);
			sources.push_back(cfg->ordinal());
			destinations.push_back(cfg->ordinal());

			delete cfg;
		}

		delete browser;
	}

	QMessageBox::information(NULL, "Notice",
		"BookmarkBridge has attempted to detect all browsers\n"
		"on your system.  Any browsers it has detected were\n"
		"automatically added as both a source and destination\n"
		"for bookmark synchronization.  Please take the time\n"
		"to click the Settings button and verify your configuration.");

	writeConfig();
}

/***************************************************************************
 *                                                                         *
 *   InitMemberVars()                                                      *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None.  Exception thrown if error occurs.                           *
 *   Description:                                                          *
 *      This function initializes the following three member variables:    *
 *          progfilesdir                                                   *
 *          appdatadir                                                     *
 *          favoritesdir                                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef Q_WS_WIN
void BridgeCfg::InitMemberVars(void) throw(BkException)
{
	HINSTANCE	hShell32;
	char		progdir[_MAX_PATH];
	HKEY		winkey;
	
	winkey = GetRegKey(HKEY_LOCAL_MACHINE, "Software\\Microsoft\\Windows\\CurrentVersion");
	if(winkey == NULL)
		strcpy(progdir, "C:\\Program Files");
	else
	{
		DWORD	datasize = sizeof(progdir);
		DWORD	type;

		if(	RegQueryValueEx(
				winkey,
				"ProgramFilesDir",
				NULL,
				&type,
				(UCHAR *)progdir,
				&datasize) != ERROR_SUCCESS
		 ||	type != REG_SZ)
			strcpy(progdir, "C:\\Program Files");

		RegCloseKey(winkey);
	}
	progfilesdir = progdir;

	if(hShell32 = LoadLibrary("SHELL32.DLL"))
	{
		char	appdata[_MAX_PATH];

		typedef BOOL (CALLBACK* SHGetSpecialFolderPathAProc)(HWND, LPCSTR, int, BOOL);
		SHGetSpecialFolderPathAProc SHgsfp = (SHGetSpecialFolderPathAProc)
			GetProcAddress(hShell32, "SHGetSpecialFolderPathA");
		if(SHgsfp)
		{
			if(SHgsfp(NULL, appdata, CSIDL_APPDATA, FALSE))
				appdatadir = appdata;
			else
				appdatadir = progfilesdir;
			if(SHgsfp(NULL, appdata, CSIDL_FAVORITES, FALSE))
				favoritesdir = appdata;
			else
			{
				if(GetWindowsDirectory(appdata, sizeof(appdata)) == 0)
					favoritesdir = "";
				else
				{
					strcat(appdata, "\\Favorites");
					favoritesdir = appdata;
				}
			}
		}
		else
			appdatadir = progfilesdir;

		FreeLibrary(hShell32);
	}
	else
		appdatadir = progfilesdir;
}
#endif

/***************************************************************************
 *                                                                         *
 *   BridgeCfg::readConfig()                                               *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function attempts to read BookmarkBridge's configuration      *
 *      file, bridgeconf.xml.                                              *
 *                                                                         *
 ***************************************************************************/

void BridgeCfg::readConfig(void) throw(BkException)
{
	if(configFileName.empty())
		findConfigFile();

	if(!configFileExists)
	{
		InitDefaults();   // we'll go with defaults
		return;
	}

	SAXDATA		data;

	data.browsers = &browsers;
	data.sources = &sources;
	data.destinations = &destinations;

	// parse the file

	ParseXmlDocument(
		configFileName.c_str(),
		&data,
		xmlHandlers,
		xmlStartDocument,
		NULL);
}

/***************************************************************************
 *                                                                         *
 *   BridgeCfg::readBookmarksXbel()                                        *
 *                                                                         *
 *   Parameters:                                                           *
 *      BkFolder **root                                                    *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function attempts to read the XBEL file containing a          *
 *      persistant representation of BookmarkBridge's bookmark tree.  This *
 *      tree stores each bookmark, folder, and other information.          *
 *                                                                         *
 ***************************************************************************/

void BridgeCfg::readBookmarksXbel(BkFolder **root) throw(BkException)
{
	if(xbelFileName.empty())
		findConfigFile();
	
	// If the bookmark tree file hasn't been created yet, we'll just
	// pretend we read it, but exit with an empty tree.

	if(access(xbelFileName.c_str(), 0))
		return;

	ParseXbelDocument(xbelFileName.c_str(), root);
}

/***************************************************************************
 *                                                                         *
 *   BridgeCfg::writeBookmarksXbel()                                       *
 *                                                                         *
 *   Parameters:                                                           *
 *      const BkFolder &root                                               *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function writes the bookmark tree to an XBEL file.            *
 *                                                                         *
 ***************************************************************************/

void BridgeCfg::writeBookmarksXbel(const BkFolder &root) throw(BkException)
{
	if(xbelFileName.empty())
		findConfigFile();

	WriteXbelDocument(xbelFileName.c_str(), root, *this);
}

/***************************************************************************
 *                                                                         *
 *   BridgeCfg::writeConfig()                                              *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function saves the configuration consisting of the browsers,  *
 *      sources, and destinations to an XML configuration file.            *
 *                                                                         *
 ***************************************************************************/

void BridgeCfg::writeConfig(void) throw(BkException)
{
	// locate the existing configuration file (if any)
	
	if(configFileName.empty())
		findConfigFile();

	// open the file for writing
	
	ofstream file(configFileName.c_str(), ios::out|ios::trunc);
	if(!file)
	{
		QString	msg;

		msg.sprintf("Unable to Open Config File \"%s\" for Writing",
			configFileName.c_str());
		BKEXCEPT(msg);
	}

	file << "<bridgecfg version=\"1.0\">" << endl;

	// loop through the browsers
	
	vector<BrowserCfg>::const_iterator it;
	for(it=browsers.begin(); it!=browsers.end(); ++it)
	{
		file << "   <browser>" << endl;
		if(it->descrip().size())
			file
				<< "      "
				<< "<descrip>"
				<< EncodeString(it->descrip())
				<< "</descrip>"
				<< endl;
		file
			<< "      "
			<< "<name>"
			<< EncodeString(BrowserList[it->browserNum()-1].Name)
			<< "</name>"
			<< endl;
		file
			<< "      "
			<< "<bookmarks>"
			<< EncodeString(it->location())
			<< "</bookmarks>"
			<< endl;
		file
			<< "      "
			<< "<ordinal>"
			<< it->ordinal()
			<< "</ordinal>"
			<< endl;
		file
			<< "      "
			<< "<browsernum>"
			<< it->browserNum()
			<< "</browsernum>"
			<< endl;
		if(it->readOnly())
			file << "      <readonly/>" << endl;
		file << "   </browser>" << endl;
	}

	// loop through the sources
	
	vector<BRWSNUM>::const_iterator oit;
	int Any = 0;
	for(oit=sources.begin(); oit!=sources.end(); ++oit)
	{
		if(!Any)
		{
			Any = 1;
			file
				<< "   "
				<< "<sources>"
				<< endl;
		}

		file
			<< "      "
			<< "<source>"
			<< *oit
			<< "</source>"
			<< endl;

	}
	if(Any)
		file
			<< "   "
			<< "</sources>"
			<< endl;

	// loop through the destinations
	
	Any = 0;
	for(oit=destinations.begin(); oit!=destinations.end(); ++oit)
	{
		if(!Any)
		{
			Any = 1;
			file
				<< "   "
				<< "<destinations>"
				<< endl;
		}

		file
			<< "      "
			<< "<destination>"
			<< *oit
			<< "</destination>"
			<< endl;

	}
	if(Any)
		file
			<< "   "
			<< "</destinations>"
			<< endl;

	file << "</bridgecfg>" << endl;
}

/***************************************************************************
 *                                                                         *
 *   xmlStartDocument()                                                    *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      This function is called by the SAX2 parsing engine before any      *
 *      others.  It uses the opportunity to initialize state data.  It     *
 *      also pushes the current state on to a state stack.  This stack is  *
 *      pushed when a new XML element is encountered and popped at the end *
 *      of the element.  This allows the callback functions to always      *
 *      know where they are in the XML document.                           *
 *                                                                         *
 ***************************************************************************/

static void xmlStartDocument(void *ctx)
{
	SAXDATA *data = (SAXDATA *)ctx;

	data->browser = NULL;
}

/***************************************************************************
 *                                                                         *
 *   xmlStartBridgeCfg()                                                   *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *             (not used)                                      *
 *      const xmlChar *    (not used)                                      *
 *      const xmlChar **   (not used)                                      *
 *      short prevState                                                    *
 *   Return:                                                               *
 *      False on success, true on failure                                  *
 *   Description:                                                          *
 *      This function is called by the XML parser (see xmlparser.h) when   *
 *      it encounters the beginning of the <bridgecfg> element.            *
 *                                                                         *
 ***************************************************************************/

static bool xmlStartBridgeCfg(void *, const xmlChar *, const xmlChar **, short prevState)
{
	if(prevState != SAXSTATE_STARTEND)
		return true;

	return false;
}

/***************************************************************************
 *                                                                         *
 *   xmlStartBrowser()                                                     *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *    (not used)                                      *
 *      const xmlChar **   (not used)                                      *
 *      short              (not used)                                      *
 *   Return:                                                               *
 *      False on success, true on failure                                  *
 *   Description:                                                          *
 *      This function is called by the XML parser (see xmlparser.h) when   *
 *      it encounters the beginning of the <browser> element.              *
 *                                                                         *
 ***************************************************************************/

static bool xmlStartBrowser(void *ctx, const xmlChar *, const xmlChar **, short)
{
	SAXDATA *data = (SAXDATA *)ctx;

	data->browser = new BrowserCfg;
	if(data->browser == NULL)
	{
		delete data->browser;
		return true;
	}

	return false;
}

/***************************************************************************
 *                                                                         *
 *   xmlEndBrowser()                                                       *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *    (not used)                                      *
 *      const xmlChar *    (not used)                                      *
 *      short state                                                        *
 *      short              (not used)                                      *
 *   Return:                                                               *
 *      False on success, true on failure                                  *
 *   Description:                                                          *
 *      This function is called by the XML parser (see xmlparser.h) when   *
 *      it encounters the end of the <browser> element.                    *
 *                                                                         *
 ***************************************************************************/

static void xmlEndBrowser(void *ctx, const xmlChar *, const xmlChar *, short state, short)
{
	SAXDATA *data = (SAXDATA *)ctx;

	ASSERT(state == SAXSTATE_BROWSER);

	data->browsers->push_back(*(data->browser));
	delete data->browser;
	data->browser = NULL;
}

/***************************************************************************
 *                                                                         *
 *   xmlEndBrowserNum()                                                    *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *    (not used)                                      *
 *      const xmlChar *pChars                                              *
 *      short state                                                        *
 *      short              (not used)                                      *
 *   Return:                                                               *
 *      False on success, true on failure                                  *
 *   Description:                                                          *
 *      This function is called by the XML parser (see xmlparser.h) when   *
 *      it encounters the end of the <browsernum> element.                 *
 *                                                                         *
 ***************************************************************************/

static void xmlEndBrowserNum(void *ctx, const xmlChar *, const xmlChar *pChars, short state, short)
{
	SAXDATA *data = (SAXDATA *)ctx;

	ASSERT(state == SAXSTATE_BROWSERNUM);

	if(pChars)
		data->browser->setBrowserNum(atol((const char *)pChars));
}

/***************************************************************************
 *                                                                         *
 *   xmlEndBookmarks()                                                     *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *    (not used)                                      *
 *      const xmlChar *pChar                                               *
 *      short state                                                        *
 *      short              (not used)                                      *
 *   Return:                                                               *
 *      False on success, true on failure                                  *
 *   Description:                                                          *
 *      This function is called by the XML parser (see xmlparser.h) when   *
 *      it encounters the end of the <bookmarks> element.                  *
 *                                                                         *
 ***************************************************************************/

static void xmlEndBookmarks(void *ctx, const xmlChar *, const xmlChar *pChars, short state, short)
{
	SAXDATA *data = (SAXDATA *)ctx;

	ASSERT(state == SAXSTATE_BOOKMARKS);

	if(pChars)
		data->browser->setLocation(QString::fromUtf8((const char *)pChars));
}

/***************************************************************************
 *                                                                         *
 *   xmlEndDescrip()                                                       *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *    (not used)                                      *
 *      const xmlChar *pChars                                              *
 *      short state                                                        *
 *      short              (not used)                                      *
 *   Return:                                                               *
 *      False on success, true on failure                                  *
 *   Description:                                                          *
 *      This function is called by the XML parser (see xmlparser.h) when   *
 *      it encounters the end of the <descrip> element.                    *
 *                                                                         *
 ***************************************************************************/

static void xmlEndDescrip(void *ctx, const xmlChar *, const xmlChar *pChars, short state, short)
{
	SAXDATA *data = (SAXDATA *)ctx;

	ASSERT(state == SAXSTATE_DESCRIP);

	if(pChars)
		data->browser->setDescrip(QString::fromUtf8((const char *)pChars));
}

/***************************************************************************
 *                                                                         *
 *   xmlEndDestination()                                                   *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *    (not used)                                      *
 *      const xmlChar *pChars                                              *
 *      short state                                                        *
 *      short              (not used)                                      *
 *   Return:                                                               *
 *      False on success, true on failure                                  *
 *   Description:                                                          *
 *      This function is called by the XML parser (see xmlparser.h) when   *
 *      it encounters the end of the <destination> element.                *
 *                                                                         *
 ***************************************************************************/

static void xmlEndDestination(void *ctx, const xmlChar *, const xmlChar *pChars, short state, short prevState)
{
	SAXDATA *data = (SAXDATA *)ctx;

	ASSERT(state == SAXSTATE_DESTINATION);

	if(prevState==SAXSTATE_DESTINATIONS && pChars)
		data->destinations->push_back(atoi((const char *)pChars));
}

/***************************************************************************
 *                                                                         *
 *   xmlEndName()                                                          *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *    (not used)                                      *
 *      const xmlChar *pChars                                              *
 *      short state                                                        *
 *      short              (not used)                                      *
 *   Return:                                                               *
 *      False on success, true on failure                                  *
 *   Description:                                                          *
 *      This function is called by the XML parser (see xmlparser.h) when   *
 *      it encounters the end of the <name> element.                       *
 *                                                                         *
 ***************************************************************************/

static void xmlEndName(void *ctx, const xmlChar *, const xmlChar *pChars, short state, short)
{
	SAXDATA *data = (SAXDATA *)ctx;

	ASSERT(state == SAXSTATE_NAME);

	if(pChars)
		data->browser->setBrowser(QString::fromUtf8((const char *)pChars));
}

/***************************************************************************
 *                                                                         *
 *   xmlEndOrdinal()                                                       *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *    (not used)                                      *
 *      const xmlChar *pChars                                              *
 *      short state                                                        *
 *      short              (not used)                                      *
 *   Return:                                                               *
 *      False on success, true on failure                                  *
 *   Description:                                                          *
 *      This function is called by the XML parser (see xmlparser.h) when   *
 *      it encounters the end of the <ordinal> element.                    *
 *                                                                         *
 ***************************************************************************/

static void xmlEndOrdinal(void *ctx, const xmlChar *, const xmlChar *pChars, short state, short)
{
	SAXDATA *data = (SAXDATA *)ctx;

	ASSERT(state == SAXSTATE_ORDINAL);

	if(pChars)
		data->browser->setOrdinal(atoi((const char *)pChars));
}

/***************************************************************************
 *                                                                         *
 *   xmlEndReadonly()                                                      *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *    (not used)                                      *
 *      const xmlChar *    (not used)                                      *
 *      short state                                                        *
 *      short              (not used)                                      *
 *   Return:                                                               *
 *      False on success, true on failure                                  *
 *   Description:                                                          *
 *      This function is called by the XML parser (see xmlparser.h) when   *
 *      it encounters the the <readonly/> element.                         *
 *                                                                         *
 ***************************************************************************/

static void xmlEndReadonly(void *ctx, const xmlChar *, const xmlChar *, short state, short)
{
	SAXDATA *data = (SAXDATA *)ctx;

	ASSERT(state == SAXSTATE_READONLY);

	data->browser->setReadOnly(true);
}

/***************************************************************************
 *                                                                         *
 *   xmlEndSource()                                                        *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *    (not used)                                      *
 *      const xmlChar *pChars                                              *
 *      short state                                                        *
 *      short              (not used)                                      *
 *   Return:                                                               *
 *      False on success, true on failure                                  *
 *   Description:                                                          *
 *      This function is called by the XML parser (see xmlparser.h) when   *
 *      it encounters the end of the <source> element.                     *
 *                                                                         *
 ***************************************************************************/

static void xmlEndSource(void *ctx, const xmlChar *, const xmlChar *pChars, short state, short prevState)
{
	SAXDATA *data = (SAXDATA *)ctx;

	ASSERT(state == SAXSTATE_SOURCE);

	if(prevState==SAXSTATE_SOURCES && pChars)
		data->sources->push_back(atoi((const char *)pChars));
}
