//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <iostream>
#include <cstring>
#include <cstdio>

#include <glibmm.h>
#include <glibmm/i18n.h>

#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#include <libhal.h>
#include <libhal-storage.h>

#include <boost/format.hpp>
#include <boost/algorithm/string.hpp>
#include "uri++.hh"

#include "database.hh"
#include "x_service_core.hh"

#include "debug.hh"
#include "paths.hh"

#include "hal.hh"
#include "hal++.hh"

using namespace Hal;

namespace
{

  const char* HALBus[] =
  {
    "<Unknown>",
    "IDE",
    "SCSI",
    "USB",
    "IEEE1394",
    "CCW"
  };

  struct HALDriveType
  {
    const char  * name;
    const char  * icon_default;
    const char  * icon_usb;
    const char  * icon_1394;
  };

  const HALDriveType drive_type_data[] =
  {
    { N_("Removable Disk"),       "device-removable.png",   "device-removable-usb.png", "device-removable-1394.png"   },
    { N_("Disk"),                 "device-harddrive.png",   "device-harddrive-usb.png", "device-harddrive-1394.png"   },
    { N_("CD-ROM"),               "device-cdrom.png",        NULL,                       NULL                         },
    { N_("Floppy"),               "device-floppy.png",       NULL,                       NULL                         }, 
    { N_("Tape"),                 "device-removable.png",    NULL,                       NULL                         },  
    { N_("CF Card"),              "device-cf-card.png",      NULL,                       NULL                         }, 
    { N_("MemoryStick (TM)"),     "device-ms.png",           NULL,                       NULL                         },  
    { N_("SmartMedia Card"),      "device-sm-card.png",      NULL,                       NULL                         },  
    { N_("SD/MMC Card"),          "device-sdmmc-card.png",   NULL,                       NULL                         },  
    { N_("Camera"),               "device-camera.png",       NULL,                       NULL                         },  
    { N_("Portable Player"),      "device-ipod.png",         NULL,                       NULL                         },   
    { N_("ZIP Drive"),            "device-floppy.png",       NULL,                       NULL                         },    
    { N_("JAZ Drive"),            "device-floppy.png",       NULL,                       NULL                         },     
    { N_("FlashKey"),             "device-removable.png",    NULL,                       NULL                         },     
  };
}

namespace Bmp
{
  namespace Library
  {
      ::Bmp::DB::DatumDefine volume_attrs[] = 
      {
          {
              N_("Volume UDI"),
              N_("Volume UDIs"),
              N_("The HAL UDI of the volume"),
              "volume_udi",
              DB::VALUE_TYPE_STRING,
          },

          {
              N_("Device UDI"),
              N_("Device UDIs"),
              N_("The HAL UDI of the device the volume is on"),
              "device_udi",
              DB::VALUE_TYPE_STRING,
          },

          {
              N_("Mount Path"),
              N_("Mount Paths"),
              N_("The current mount path of the volume"),
              "mount_point",
              DB::VALUE_TYPE_STRING,
          },

          {
              N_("Device Serial"),
              N_("Device Serials"),
              N_("The device's serial number"),
              "drive_serial",
              DB::VALUE_TYPE_STRING,
          },

          {
              N_("Volume Name"),
              N_("Volume Names"),
              N_("The name of the volume"),
              "label",
              DB::VALUE_TYPE_STRING,
          },

          {
              N_("Device File"),
              N_("Device Files"),
              N_("The UNIX Device File of this volume"),
              "device_node",
              DB::VALUE_TYPE_STRING,
          },

          {
              N_("Drive Bus"),
              N_("Drive Buses"),
              N_("The device bus the volume's device is on (e.g. USB)"),
              "drive_bus",
              DB::VALUE_TYPE_INT,
          },

          {
              N_("Drive Type"),
              N_("Drive Types"),
              N_("The type of the drive this volume is on"),
              "drive_type",
              DB::VALUE_TYPE_INT,
          },

          {
              N_("Volume Size"),
              N_("Volume Sizes"),
              N_("The storage size of the volume"), 
              "size",
              DB::VALUE_TYPE_INT,
          },

          {
              N_("Drive Size"),
              N_("Drive Sizes"),
              N_("The storage size of the drive this volume is on"), 
              "drive_size",
              DB::VALUE_TYPE_INT,
          },

          {
              N_("Drive Size"),
              N_("Drive Sizes"),
              N_("The storage size of the drive this volume is on"), 
              "mount_time",
              DB::VALUE_TYPE_INT,
          },

      };

      enum VolumeAttribute
      {
        VATTR_VOLUME_UDI,
        VATTR_DEVICE_UDI,
        VATTR_MOUNT_PATH,
        VATTR_DEVICE_SERIAL,
        VATTR_VOLUME_NAME,
        VATTR_DEVICE_FILE,
        VATTR_DRIVE_BUS,         
        VATTR_DRIVE_TYPE,         
        VATTR_SIZE,         
        VATTR_DRIVE_SIZE,         
        VATTR_MOUNT_TIME,         
      };

      HAL::Volume::Volume ( Hal::RefPtr<Hal::Context> const& context,
                            Hal::RefPtr<Hal::Volume>  const& volume) throw ()
      {
        using namespace Bmp::DB;
        using namespace Glib; 

        Hal::RefPtr<Hal::Drive> drive = Hal::Drive::create_from_udi (context, volume->get_storage_device_udi());

        volume_udi      = volume->get_udi();
        device_udi      = volume->get_storage_device_udi(); 

        size            = volume->get_size ();
        label           = volume->get_label(); 

        mount_time      = time( NULL ); 
        mount_point     = volume->get_mount_point(); 

        drive_bus       = drive->get_bus();
        drive_type      = drive->get_type();

#ifdef HAVE_HAL_058
        drive_size      = drive->get_size(); 
#else
        drive_size      = 0; 
#endif //HAVE_HAL_058

        drive_serial    = drive->get_serial(); 
        device_file     = drive->get_device_file(); 
      }


      HAL::Volume::Volume (Bmp::DB::Row const& row)
      {
        using namespace Bmp::DB;
        using namespace Glib; 

        Row::const_iterator i;
        Row::const_iterator end = row.end();

        i = row.find (volume_attrs[VATTR_VOLUME_UDI].id);
        volume_udi = boost::get <std::string> (i->second).c_str();

        i = row.find (volume_attrs[VATTR_DEVICE_UDI].id);
        device_udi = boost::get <std::string> (i->second).c_str();

        i = row.find (volume_attrs[VATTR_DRIVE_BUS].id);
        drive_bus = Hal::DriveBus(boost::get <uint64_t> (i->second));

        i = row.find (volume_attrs[VATTR_DRIVE_TYPE].id);
        drive_type = Hal::DriveType (boost::get <uint64_t> (i->second));

        i = row.find (volume_attrs[VATTR_SIZE].id);
        size = boost::get <uint64_t> (i->second);

        i = row.find (volume_attrs[VATTR_DRIVE_SIZE].id);
        drive_size = boost::get <uint64_t> (i->second);

        i = row.find (volume_attrs[VATTR_MOUNT_TIME].id);
        mount_time = boost::get <uint64_t> (i->second);

        i = row.find (volume_attrs[VATTR_MOUNT_PATH].id);
        if (i != end)
          mount_point = boost::get <std::string> (i->second).c_str();

        i = row.find (volume_attrs[VATTR_DEVICE_SERIAL].id);
        if (i != end)
          drive_serial = boost::get <std::string> (i->second).c_str();

        i = row.find (volume_attrs[VATTR_VOLUME_NAME].id);
        if (i != end)
          label = boost::get <std::string> (i->second).c_str();

        i = row.find (volume_attrs[VATTR_DEVICE_FILE].id);
        if (i != end)
          device_file = boost::get <std::string> (i->second).c_str();
      }

      ///////////////////////////////////////////////////////

      void
      HAL::register_update_volume (Volume const& volume)
      {
        Bmp::DB::Row row;

        row[volume_attrs[VATTR_VOLUME_UDI].id]    = volume.volume_udi;
        row[volume_attrs[VATTR_DEVICE_UDI].id]    = volume.device_udi;
        row[volume_attrs[VATTR_MOUNT_PATH].id]    = volume.mount_point;
        row[volume_attrs[VATTR_DEVICE_SERIAL].id] = volume.drive_serial;
        row[volume_attrs[VATTR_VOLUME_NAME].id]   = volume.label;
        row[volume_attrs[VATTR_DEVICE_FILE].id]   = volume.device_file;
        row[volume_attrs[VATTR_DRIVE_BUS].id]     = uint64_t(volume.drive_bus);
        row[volume_attrs[VATTR_DRIVE_TYPE].id]    = uint64_t(volume.drive_type);
        row[volume_attrs[VATTR_SIZE].id]          = uint64_t(volume.size);
        row[volume_attrs[VATTR_DRIVE_SIZE].id]    = uint64_t(volume.drive_size);
        row[volume_attrs[VATTR_MOUNT_TIME].id]    = uint64_t(volume.mount_time);

        m_hal_db->add ("volumes", row);
      }

      void
      HAL::volumes (VVolumes & volumes) const
      {
        using namespace Bmp::DB;
        VAttributes a;
        VRows rows;

        m_hal_db->get ("volumes", a, rows);

        for (VRows::const_iterator r = rows.begin(); r != rows.end(); ++r)
        {
          Volume volume ((*r));
          volumes.push_back (volume);
        }
      }

      HAL::HAL ()
      {
        using namespace Bmp::DB;
  
        m_initialized = hal_init ();
        if (!m_initialized)
          {
            Gtk::MessageDialog dialog (_("Volume/Device Management could not be m_initialized; HAL monitoring of"
                                         "volumes in the library, permanent addition of items to the library"
                                         "and other parts of BMP will not be functional.\n\nPlease check "
                                         "your HAL installation and/or contact your distributor."),
                                          false, Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK, true);
            dialog.run();
            dialog.hide();
            return;
          }
      }

      void
      HAL::cdrom_policy (Hal::RefPtr<Hal::Volume> const& volume)
      {
        using namespace Hal;

        Hal::RefPtr<Hal::Drive> drive = Hal::Drive::create_from_udi (m_context, volume->get_storage_device_udi ());

        if (drive->property_exists  ("info.locked") &&
            drive->get_property <bool> ("info.locked"))
        {
          debug ("hal", "drive with udi %s is locked through hal; skipping policy", volume->get_udi().c_str());
          return;
        }

        Hal::DriveType drive_type; 

        try{
          drive_type = drive->get_type ();
          }
        catch (...)
          {
            debug ("hal", "cannot get drive_type for %s", volume->get_storage_device_udi().c_str());
            return;
          }

        if (drive_type == Hal::DRIVE_TYPE_CDROM)
        {
            Hal::DiscProperties props; 

            try{
              props = volume->get_disc_properties();
              }
            catch (...)
              {
                return;
              }

            std::string device_udi (volume->get_storage_device_udi());

            if ((props & Hal::DISC_HAS_AUDIO) && !(props & Hal::DISC_HAS_DATA))
            {
              debug ("hal", "CD is CDDA and has no data: %s at %s",
                volume->get_udi().c_str(), device_udi.c_str());
              signal_cdda_inserted_.emit (volume->get_udi(), volume->get_device_file());
            }
            else
            if ((props & Hal::DISC_HAS_AUDIO) && (props & Hal::DISC_HAS_DATA))
            {
              debug ("hal", "CD is CDDA/Mixed-Mode: %s at %s",
                volume->get_udi().c_str(), device_udi.c_str());
              signal_cdda_inserted_.emit (volume->get_udi(), volume->get_device_file());
            }
            else
            if ((props & Hal::DISC_HAS_DATA))
            {
              debug ("hal", "CD is CDROM");
            }
            else
            if ((props & Hal::DISC_IS_BLANK))
            {
              debug ("hal", "CD is a blank medium");
            }
        }
      }

      void
      HAL::process_volume (std::string const &udi)
      {
        using namespace Bmp;

        Hal::RefPtr<Hal::Volume> volume = Hal::Volume::create_from_udi (m_context, udi);

        if (volume->get_device_file().empty())
        {
          debug ("hal", "cannot get device file: %s", udi.c_str()); 
          return;
        }

        if (volume->get_fsusage() != Hal::VOLUME_USAGE_MOUNTABLE_FILESYSTEM)
        {
          debug ("hal", "no mountable filesystem for %s", udi.c_str());
          if (volume->is_disc())
          {
            cdrom_policy (volume);
            return;
          }
        }
        else
        {
          process_udi (udi);
        }
      }

      Glib::ustring
      HAL::get_volume_type (Volume const& volume) const
      {
        return Glib::ustring (drive_type_data[guint(volume.drive_type)].name);
      }
 
      Glib::ustring
      HAL::get_volume_drive_bus (Volume const& volume) const
      {
        return Glib::ustring (HALBus[volume.drive_bus]);
      }

      Glib::RefPtr<Gdk::Pixbuf> 
      HAL::get_volume_icon (Volume const& volume) const
      {
        guint index (volume.drive_type);
        std::string baseicon;

        if (volume.drive_bus == Hal::DRIVE_BUS_USB)
          {
            baseicon = drive_type_data[index].icon_usb  ? drive_type_data[index].icon_usb 
                                                        : drive_type_data[index].icon_default;
          }
        else
        if (volume.drive_bus == Hal::DRIVE_BUS_IEEE1394)
          {
            baseicon = drive_type_data[index].icon_1394 ? drive_type_data[index].icon_1394 
                                                        : drive_type_data[index].icon_default;
          }
        else
          {
            baseicon = drive_type_data[index].icon_default;
          }
        
        return Gdk::Pixbuf::create_from_file
          (Glib::build_filename (BMP_IMAGE_DIR_HAL, baseicon));
      }

      bool
      HAL::volume_is_mounted  (Volume const& volume) const
      {
        try{

            Hal::RefPtr<Hal::Volume> volume_hal_instance =
              Hal::Volume::create_from_udi (m_context, volume.volume_udi);

            return volume_hal_instance->is_mounted();
          }
        catch (...)
          {
            throw InvalidVolumeSpecifiedError (volume.volume_udi.c_str());
          }
      }

      HAL::Volume
      HAL::get_volume_for_uri (Glib::ustring const& uri) const
      {
        using namespace Glib;
        using namespace std;

        if (!m_initialized)
          throw NotInitializedError(); 

        string filename (filename_from_uri (uri));
        string::size_type l = 0;

        Volumes::const_iterator m = m_volumes.end();

        for (Volumes::const_iterator i = m_volumes.begin (); i != m_volumes.end (); ++i)
        {
          std::string::size_type lv (i->second.first.mount_point.length());

          if ((i->second.first.mount_point == filename.substr (0, lv)) && (lv > l))
          {
            m = i;
            l = lv;
          } 
	      }

        if (m != m_volumes.end())
          return m->second.first; 
        else
          throw NoVolumeForUriError ((boost::format ("No volume for Uri %s") % uri.c_str()).str());
      }

      std::string
      HAL::get_mount_point_for_volume (std::string const& volume_udi, std::string const& device_udi) const
      {
        if (!m_initialized)
          throw NotInitializedError(); 

        VolumeKey match (volume_udi, device_udi);
        Volumes::const_iterator v_iter = m_volumes.find (match);

        if (v_iter != m_volumes.end())
          return v_iter->second.first.mount_point; 
 
        throw NoMountPathForUdiError
            ((boost::format ("No mount path for given volume: device: %s, volume: %s") % device_udi % volume_udi).str());
      }

      void
      HAL::process_udi (std::string const& udi)
      {
        using boost::algorithm::split;
        using boost::algorithm::is_any_of;
        using boost::algorithm::to_lower;
        using boost::algorithm::find_last;
        using boost::iterator_range;

        debug ("hal", "Processing UDI '%s'", udi.c_str());  

        Hal::RefPtr<Hal::Volume> volume_hal_instance;
        try{
          volume_hal_instance = Hal::Volume::create_from_udi (m_context, udi);
          }
        catch (...)
          {
            debug ("hal", "Error constructing Hal::Volume from UDI: %s", udi.c_str());
            return;
          }

        if (!(volume_hal_instance->get_fsusage() == Hal::VOLUME_USAGE_MOUNTABLE_FILESYSTEM))
          return;

        if (!(volume_hal_instance->is_mounted()))
          return;

        try{

            Volume volume (m_context, volume_hal_instance);
            VolumeKey key (volume.volume_udi, volume.device_udi);

            m_volumes.insert (std::make_pair (key, StoredVolume (volume, volume_hal_instance)));

            if (!volume_hal_instance->is_disc())
              {
                register_update_volume (volume);  // we don't store information for discs permanently
              }

            Hal::RefPtr<Hal::Volume> const& vol (m_volumes.find (key)->second.second);

            vol->signal_condition().connect
              (sigc::mem_fun (this, &Bmp::Library::HAL::device_condition));

            vol->signal_property_modified().connect
              (sigc::mem_fun (this, &Bmp::Library::HAL::device_property));

            signal_volume_added_.emit (volume); 
          }
        catch (...) {}
      }

      bool
      HAL::hal_init ()
      {
        using namespace Bmp::DB;
        using namespace Glib;

        DBusError error;
        dbus_error_init (&error);

        DBusConnection * c = dbus_bus_get (DBUS_BUS_SYSTEM, &error);

        if (dbus_error_is_set (&error))
          {
            debug ("hal", G_STRLOC ": Error conntecting to DBus.");
            dbus_error_free (&error);
            return false;
          }

        dbus_connection_setup_with_g_main (c, NULL);
        dbus_connection_set_exit_on_disconnect (c, FALSE);

        try{
          m_context = Hal::Context::create (c);
          }
        catch (...) 
          {
            debug ("hal", G_STRLOC ": Unable to create a Hal::Context");
            return false;
          }

        try{
          m_context->signal_device_added().connect
            (sigc::mem_fun (this, &Bmp::Library::HAL::device_added));
          m_context->signal_device_removed().connect
            (sigc::mem_fun (this, &Bmp::Library::HAL::device_removed));
          }
        catch (...)
          {
            debug ("hal", G_STRLOC ": Not able to connect device-added/device-removed signals.");
            m_context.clear();
            return false;
          }

        Hal::StrV list;

        try{
          list = m_context->find_device_by_capability ("volume");
          }
        catch (...)
          {
            debug ("hal", G_STRLOC ": Hal::Context::get_all_devices failed()");
            m_context.clear();
            return false;
          }

        for (unsigned int n = 0; n < G_N_ELEMENTS(volume_attrs); ++n)
          {
            m_hal_map.insert (std::make_pair (std::string (volume_attrs[n].id), volume_attrs[n].type));
          }

        m_hal_db = new ::Bmp::DB::Database ("hal", BMP_PATH_DATA_DIR, ::Bmp::DB::DB_OPEN);

        try {
            ustring sql;
            sql.append ("CREATE TABLE IF NOT EXISTS volumes (");
            for (unsigned int n = 0; n < G_N_ELEMENTS(volume_attrs); ++n)
            {
              sql.append ("'");
              sql.append (volume_attrs[n].id);
              sql.append ("' ");

              switch (volume_attrs[n].type)
              {

                  case VALUE_TYPE_STRING:
                      sql.append (" TEXT ");
                      break;

                  case VALUE_TYPE_INT:
                      sql.append (" INTEGER DEFAULT 0 ");
                      break;
              }

              sql.append (", ");
            }

            sql.append (" PRIMARY KEY ('");
            sql.append (volume_attrs[VATTR_DEVICE_UDI].id);
            sql.append ("', '"); 
            sql.append (volume_attrs[VATTR_VOLUME_UDI].id);
            sql.append ("') ON CONFLICT REPLACE);"); 

            m_hal_db->sqlite_exec_simple (sql.c_str());

          }
        catch (Bmp::DB::SqlError& cxe)
          {
          }

        m_hal_db->insert_map_for_table ("volumes", m_hal_map);

        for (Hal::StrV::const_iterator n = list.begin(); n != list.end(); ++n) 
          process_udi (*n);

        return true;
      }

      HAL::~HAL ()
      {
        delete m_hal_db;
      }

      void
      HAL::device_condition  (std::string const& udi,
                              std::string const& cond_name,
                              std::string const& cond_details)
      {
        if (cond_name == "EjectPressed")
          {
            signal_ejected_.emit (udi);
            debug ("hal", "ejected: %s", udi.c_str());
          }
      }

      void
      HAL::device_added (std::string const& udi)
      {
        try{
            if (!m_context->device_query_capability (udi, "volume"))
              return;
            process_volume (udi);       
          }
        catch (...)
          {
            debug ("hal", "Error in device-added signal handler!");
            return;
          }
      }

      void
      HAL::device_removed (std::string const& udi) 
      {
        //FIXME: Remove from m_volumes 
        signal_device_removed_.emit (udi);
        debug ("hal", "device removed: %s", udi.c_str());
      }

      void
      HAL::device_property	
          (std::string const& udi,
           std::string const& key,
           bool               is_removed,
           bool               is_added)
      {
        using namespace Bmp;
        using namespace Bmp::Library;

        if (!m_context->device_query_capability (udi, "volume"))
        {
          debug ("hal", "'%s' does not have 'volume' capability!");
          return;
        }

        Hal::RefPtr<Hal::Volume> volume = Hal::Volume::create_from_udi (m_context, udi);

        debug ("hal", "'%s' is %s", (volume->is_mounted()?"mounted":"not mounted"));

        if (volume->is_mounted())
          {
            process_udi (udi);
          }
        else
          {
            Volumes::iterator n (m_volumes.find (VolumeKey (udi, volume->get_storage_device_udi ())));
            if (n != m_volumes.end())
            {
              Volume vol (n->second.first);
              signal_volume_removed_.emit (vol); 
            }
          }
      }

      HAL::SignalVolume&
      HAL::signal_volume_removed ()
      {
        return signal_volume_removed_;
      }

      HAL::SignalVolume&
      HAL::signal_volume_added ()
      {
        return signal_volume_added_;
      }

      HAL::SignalDeviceRemoved&
      HAL::signal_device_removed ()
      {
        return signal_device_removed_;
      }

      HAL::SignalCDDAInserted&
      HAL::signal_cdda_inserted ()
      {
        return signal_cdda_inserted_;
      }

      HAL::SignalEjected&
      HAL::signal_ejected ()
      {
        return signal_ejected_;
      }
  }
}
