//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifndef BMP_DATABASE_HH
#define BMP_DATABASE_HH

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <glibmm.h>
#include <sqlite3_bmp.h>
#include <queue>
#include <vector>

#include <boost/variant.hpp>
#include <boost/tuple/tuple.hpp>

#include <bmp/database_types.hh>

namespace Bmp
{
  Glib::ustring
  sql_uprintf (const char *format, ...);
 
  namespace DB
  {

#include "exception.hh"

    EXCEPTION(DbInitError)
    EXCEPTION(SqlError)

#define FORMAT_ERROR(sqlerr, sqlcode, sql)  ((sql_err_fmt % (sqlerr) % (sqlcode) % (sql)).str ().c_str ())

    void
    append_attributes (Glib::ustring &sql, VAttributes const& attributes);

	  /** Slot for notifying about changes in a table
	   *
	   */
	  typedef sigc::slot < void, sqlite_int64 > SlotSQLHook;
	  typedef std::vector < SlotSQLHook > Hooks;
	  typedef std::map < std::string, Hooks > TableHookMap;

	  enum HookType
	  {
	    HOOK_INSERT,
	    HOOK_DELETE,
	    HOOK_UPDATE,

	    N_HOOKS,
	  };

	  typedef boost::tuple  < HookType, sqlite_int64, std::string > QueueHook;
	  typedef std::queue    < QueueHook >				HookQueue;

    enum DbOpenMode
    {
      DB_TRUNCATE,
      DB_OPEN,
    };

    enum TableRowMode
    {
      ROW_MODE_DEFAULT,
      ROW_MODE_REPLACE,
    };
	
	  class Database
	  {
	    public:

	      Database (std::string const& name, std::string const& path, DbOpenMode mode = DB_TRUNCATE);
	      ~Database ();

	      void
	      create_table     (std::string const&  name,
                          std::string const&  pkey,
                          ValueMap    const&  map,
                          TableRowMode        row_mode = ROW_MODE_DEFAULT);

	      void
	      create_attribute_view

                        (std::string const& name,
                         std::string const& view_name,
                         std::string const& attribute,
                         ValueMap const& map);	

	      void
	      create_view_manual  

                        (std::string const& name,
                         std::string const& view_name,
                         std::string const& select,
                         ValueMap const& map);

	      void
	      drop_table      (std::string const& name);

	      bool
	      table_exists    (std::string const& name);
    
        bool
        attr_exists     (std::string const& name, 
                         std::string const& pkey_name,
                         Glib::ustring const& pkey_value);

	      void 
	      project	        (std::string const& name,
                         std::string const& project, 
                         VRows & vector,
                         VAttributes const& attributes = VAttributes ());

	      void 
	      get	            (std::string const& name,
                         VAttributes const& attributes,
                         VRows & vector,
                         std::string const& suffix = std::string()); 

	      void 
	      get             (std::string const& name,
                         sqlite_int64	rowid,
                         Row & row); 

	      void 
	      set	            (std::string const& name,
                         std::string const& key,
                         std::string const& value,
                         VAttributesSet const& attributes);

	      void 
	      set             (std::string const& name,
                         Glib::ustring const& where_clause,
                         VAttributesSet const& attributes);

	      void
	      add	            (std::string const& name,
                         Row const& row);
			       

	      void
	      del	            (std::string const&   name,
			                   VAttributes const& attributes);

	      unsigned int
	      sqlite_exec_simple 
                        (Glib::ustring const& sql); 

        void
        insert_map_for_table
                        (std::string const& name,
                         ValueMap const& map);

	    private:

	      sqlite3	* sqlite;

	      static void sqlite_trace (void * data, const char * sqltext);

	      TableMapMap  mapmap;
	      std::string  name;
	      std::string  path;        //sqlite3 filename = path+name+".mlib";
	  };
  } // namespace DB
}; // namespace Bmp 

#endif // !BMP_DATABASE_HH
