//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2006 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non GPL-compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <iostream>
#include <sstream>

#include <glibmm.h>
#include <glib/gstdio.h>

#include <sqlite3_bmp.h>

#include <boost/tuple/tuple.hpp>
#include <boost/variant.hpp>
#include <boost/format.hpp>

#include "debug.hh"
#include "database.hh"

namespace
{
  const char* sql_valuenames[] =
  {
      "BOOL", 
      "INTEGER",
      "REAL",
      "TEXT"
  };

  static boost::format format_int  ("%llu");
  static boost::format format_bool ("%d");
  static boost::format format_real ("%f");

  static boost::format sql_err_fmt ("SQLite Error: %s, SQLite Error Code: %d, SQL String: '%s'");

  int statement_prepare (sqlite3_stmt **pStmt, Glib::ustring const& sql, sqlite3 *sqlite)
  {
      int status;
      const char *tail;
      status = sqlite3_prepare (sqlite,
                                sql.c_str(), 
                                strlen (sql.c_str()),
                                pStmt,
                                &tail);      
  
      if (status)
        {
          g_warning ("SQL Error: '%s', SQL Statement: '%s'", sqlite3_errmsg (sqlite), sql.c_str ());
          sqlite3_reset (*pStmt);
        };

      return status;
  }

  void get_row_from_statement (sqlite3_stmt *pStmt, Bmp::DB::Row& row, Bmp::DB::ValueMap const& map)
  {
      using namespace Bmp::DB;
      using namespace Glib; 
      using namespace std;

      unsigned int counter = 0;
      for (ValueMap::const_iterator i = map.begin (); i != map.end (); ++i)
      {
        switch (i->second)
          {
            case VALUE_TYPE_BOOL:
              row.insert (make_pair (i->first, bool(sqlite3_column_int (pStmt, counter++)))); break;

            case VALUE_TYPE_INT:
              row.insert (make_pair (i->first, uint64_t(sqlite3_column_int (pStmt, counter++)))); break;

            case VALUE_TYPE_REAL:
              row.insert (make_pair (i->first, double(sqlite3_column_double (pStmt, counter++)))); break;

            case VALUE_TYPE_STRING:
              {
                const char* value = (const char*)sqlite3_column_text (pStmt, counter++);        
                if (value && strlen(value))
                  {
                    row.insert (make_pair (i->first, ustring(value)));
                    break;
                  }
              }
            default: break;
          }
      }
  }

  void
  print_compiled_statement (sqlite3_stmt *stmt)
  {
    int count = sqlite3_bind_parameter_count (stmt);
    for (int n = 0; n < count; ++n)
    {
      g_message ("Parameter %d: %s", n, sqlite3_bind_parameter_name (stmt, n));
    }
  }

};

namespace Bmp
{
  Glib::ustring
  sql_uprintf (const char *format, ...)
  {
    va_list args;
    va_start (args, format);
    char *v = sqlite3_vmprintf (format, args);
    va_end (args);
    Glib::ustring r (v);
    sqlite3_free (v);
    return r;
  }

  namespace DB
  {
      void
      append_attributes (Glib::ustring & sql, Bmp::DB::VAttributes const& attributes)
      {
          using namespace Bmp;
          using namespace Bmp::DB;
          using namespace Glib; 

          VAttributes::const_iterator i_last (attributes.end ());
          i_last--;

          for (VAttributes::const_iterator i = attributes.begin(); i != attributes.end(); ++i)
          {
            MatchStyle    const   match_style     (boost::get<0>(*i));
            Variant       const&  value           (boost::get<2>(*i));
            std::string   const&  attribute_name  (boost::get<1>(*i));
          
            bool use_op = true;

            if (match_style == FUZZY)
              {
                  switch (value.which())
                    {
                      case VALUE_TYPE_STRING: 
                        {
                          sql.append (sql_uprintf (" %s LIKE '%%%q%%' ",
                              attribute_name.c_str(), boost::get<std::string>(value).c_str())); 
                          break;
                        }

                      case VALUE_TYPE_INT:
                        {
                          sql .append (attribute_name)  
                              .append (" LIKE '%")
                              .append ((format_int % boost::get<uint64_t>(value)).str()).append ("%' ");
                          break;
                        }

                      case VALUE_TYPE_BOOL:
                        {
                          sql .append (attribute_name)
                              .append (" LIKE '%")
                              .append ((format_int % uint64_t(boost::get<bool>(value))).str()).append ("%' ");
                          break;
                        }
                    }
              }
            else
            if (match_style == NOT_NULL)
              {
                sql.append (sql_uprintf (" %s NOT NULL ", attribute_name.c_str()));
                use_op = false;
              }
            else
            if (match_style == IS_NULL)
              {
                sql.append (sql_uprintf (" (%s IS NULL OR length(%s) = 0) ",
                    attribute_name.c_str(), attribute_name.c_str()));
                use_op = false;
              }
            else
              {
                ustring op;

                switch (match_style)
                {
                  case EXACT:
                  {
                    op = "="; break;
                  }

                  case LESSER_THAN:
                  {
                    op = "<"; break;
                  }

                  case GREATER_THAN:
                  {
                    op = ">"; break;
                  }

                  case LESSER_THAN_OR_EQUAL:
                  {
                    op = "<="; break;
                  }

                  case GREATER_THAN_OR_EQUAL:
                  {
                    op = ">="; break;
                  }

                  case IS_NULL:
                  case NOT_NULL:
                  case FUZZY:
                  {
                      // do nothing, all are handled above
                  }
                }

                if (use_op)
                  {
                    switch (value.which())
                      {
                        case VALUE_TYPE_STRING:
                          {
                            sql.append (sql_uprintf (" %s %s '%q' ",
                                attribute_name.c_str(), op.c_str(), boost::get<std::string>(value).c_str())); 
                            break;
                          }

                        case VALUE_TYPE_INT:
                          {
                            sql .append (attribute_name)
                                .append (" ")
                                .append (op)
                                .append (" '")
                                .append ((format_int % boost::get<uint64_t>(value)).str()).append ("' ");
                            break;
                          }

                        case VALUE_TYPE_BOOL:
                          {
                            sql .append (attribute_name)
                                .append (" ")
                                .append (op)
                                .append (" '")
                                .append ((format_int % int(boost::get<bool>(value))).str()).append ("' ");
                            break;
                          }
                      }
                  }
              }
          if (i != i_last) sql.append (" AND ");
        }
      }

      void
      Database::sqlite_trace   (void         *data,
                                const char   *sqltext)
      {
        g_message (G_STRLOC ": %s", sqltext);
      }

      unsigned int
      Database::sqlite_exec_simple (Glib::ustring const& sql) 
      {
        int            status = 0;
        sqlite3_stmt  *pStmt  = 0;
        const char    *tail   = 0;
        unsigned int   rows   = 0;

        debug ("sql", "exec-simple: '%s'", sql.c_str());

        status = sqlite3_prepare (sqlite,
                                  sql.c_str(),    
                                  strlen (sql.c_str()),
                                  &pStmt,
                                  &tail);      

        if (status != SQLITE_OK)
          {
            throw SqlError (FORMAT_ERROR(sqlite3_errmsg (sqlite), sqlite3_errcode (sqlite), sql));
          }

        for (status = -1; status != SQLITE_DONE; )
          {
            status = sqlite3_step (pStmt);
            if (status == SQLITE_BUSY) continue;
            if (status == SQLITE_DONE) break;
            if (status == SQLITE_ROW) { rows++; continue; }
            if ((status == SQLITE_ERROR) || (status == SQLITE_MISUSE))
              {
                int errcode = sqlite3_errcode (sqlite);
                if (errcode != 0)
                  {
                    sqlite3_reset (pStmt);
                    throw SqlError (FORMAT_ERROR(sqlite3_errmsg (sqlite), sqlite3_errcode (sqlite), sql));
                  }
                else break;
              }
          }
        sqlite3_finalize (pStmt);
        return rows;
      }

      Database::Database (std::string const& name, std::string const& path, DbOpenMode openmode)
        : name (name), path (path)
      {
        using namespace std;
        using namespace Glib;

        string db_filename (build_filename (path, name) + ".mlib");        

        if (file_test (db_filename, FILE_TEST_EXISTS) && openmode == DB_TRUNCATE)
          {
            g_unlink (db_filename.c_str());
          }

        if (sqlite3_open (db_filename.c_str(), &sqlite))
          {
            throw DbInitError ((boost::format("Unable to open database at '%s'") % db_filename).str());
          }

#ifdef SQLITE_TRACE
          sqlite3_trace (sqlite, Database::sqlite_trace, this);
#endif //SQLITE_TRACE

      };

      Database::~Database ()
      {
        sqlite3_close (sqlite);
      };

      void
      Database::create_attribute_view (std::string const& name,
                                       std::string const& view_name,
                                       std::string const& attribute,
                                       ValueMap    const& map)  
      {
        using namespace Glib;

        mapmap.insert (std::make_pair (name, map));
        if (table_exists (view_name)) return;
        ustring sql;
        sql .append ("CREATE VIEW ")
            .append (name)
            .append (".")
            .append (view_name)
            .append (" AS SELECT DISTINCT ")
            .append (attribute)
            .append (" FROM " )
            .append (name)
            .append (";");
        sqlite_exec_simple (sql.c_str());
      }

      void
      Database::create_view_manual (std::string const& name,
                                    std::string const& view_name,
                                    std::string const& select,
                                    ValueMap    const& map)
      {
        using namespace Glib;

        mapmap.insert (std::make_pair (view_name, map));
        if (table_exists (view_name)) return;
        ustring sql;
        sql .append ("CREATE VIEW ")
            .append (name)
            .append (".")
            .append (view_name)
            .append (" AS ")
            .append (select)
            .append (";");
        sqlite_exec_simple (sql.c_str());
      }

      void
      Database::add  (std::string   const& name,
                      Bmp::DB::Row  const& row)
      {
        using namespace Glib;

        if (mapmap.find (name) == mapmap.end ())
          {
            g_error ("No value-map for table '%s'", name.c_str());
          }

        ValueMap const& map (mapmap.find(name)->second);

        ustring sql_keys,
                sql_values;

        bool had_value = false; 

        sql_keys    .append ("(");
        sql_values  .append ("("); 

        for (ValueMap::const_iterator i = map.begin() ; i != map.end() ; ++i)
        {
          Bmp::DB::Row::const_iterator data_iter = row.find (i->first);

          if (data_iter != row.end())
            {
              Variant const& value (data_iter->second);

              if (had_value)
                {
                  sql_keys    .append (" , ");
                  sql_values  .append (" , ");
                }

                //add key
                sql_keys.append (i->first);

                //add its value
                sql_values.append ("'");
                switch (i->second)
                {
                  case VALUE_TYPE_STRING:
                    sql_values.append (sql_uprintf ("%q", boost::get<std::string>(value).c_str())); 
                    break;
  
                  case VALUE_TYPE_INT:
                    sql_values.append ((format_int % boost::get<uint64_t>(value)).str());
                    break;

                  case VALUE_TYPE_BOOL:
                    sql_values.append ((format_bool % boost::get<bool>(value)).str()); 
                    break;

                  case VALUE_TYPE_REAL:
                    sql_values.append ((format_real % boost::get<double>(value)).str()); 
                    break;

                }
              sql_values.append ("'");
              had_value = true;
            }
        }

        sql_keys.append (")");
        sql_values.append (")");

        ustring sql;
        sql .append ("INSERT INTO ")
            .append (name)
            .append (" ")
            .append (sql_keys)
            .append (" VALUES ")
            .append (sql_values)
            .append (";");
        sqlite_exec_simple (sql.c_str());
      }

      void
      Database::del   (std::string const&     name,
                        VAttributes  const& attributes)
      {
        using namespace Glib;

        ustring sql;
        sql.append ("DELETE FROM ").append(name).append(" WHERE ");
        append_attributes (sql, attributes);
        sql.append (";");
        sqlite_exec_simple (sql.c_str());
      }

      void 
      Database::project (std::string const& name,
                         std::string const& project, 
                         VRows & vector,
                         VAttributes const&  attributes)
      {
        using namespace std;
        using namespace Glib;

        if (mapmap.find (name) == mapmap.end ())
          {
            g_error ("No value-map for table '%s'", name.c_str());
          }

        ValueMap const& map = (*(mapmap.find(name))).second;
        ValueMap::const_iterator i_last (map.end());   
        i_last--;

        ustring sql;

        sql . append ("SELECT DISTINCT ")
            . append (project)
            . append (" FROM ")
            . append (name);

        if (!attributes.empty ())
          {
            sql.append (" WHERE ");
            append_attributes (sql, attributes);
          }

        sql.append (";");

        sqlite3_stmt *pStmt = 0;
        if (statement_prepare (&pStmt, sql, sqlite))
          {
            throw SqlError (FORMAT_ERROR(sqlite3_errmsg (sqlite), sqlite3_errcode (sqlite), sql));
          }

        for ( int status = -1 ; status != SQLITE_DONE ; )
          {
            status = sqlite3_step (pStmt);
            switch (status)
              {
                case SQLITE_BUSY: continue;
                case SQLITE_DONE: break;
                case SQLITE_ROW:            
                {
                  Bmp::DB::Row row;
                  Bmp::DB::ValueType type (map.find (project)->second);

                  switch (type)
                  {
                    case VALUE_TYPE_BOOL:
                      row.insert (make_pair (project, bool(sqlite3_column_int (pStmt, 0)))); break;

                    case VALUE_TYPE_INT:
                      row.insert (make_pair (project, uint64_t(sqlite3_column_int (pStmt, 0)))); break;

                    case VALUE_TYPE_REAL:
                      row.insert (make_pair (project, double(sqlite3_column_double (pStmt, 0)))); break;

                    case VALUE_TYPE_STRING:
                    {
                      const char* v = (const char*)sqlite3_column_text (pStmt, 0); 
                      if (v)
                        {
                          row.insert (make_pair (project, ustring(v)));
                        }
                      break;
                    }
                    default: break;
                  }

                  vector.push_back (row);
                  break;
                }
                case SQLITE_ERROR:  sqlite3_reset (pStmt); return; 
                case SQLITE_MISUSE: sqlite3_reset (pStmt); return; 
              } 
          } 
        sqlite3_finalize (pStmt);
      }

      void 
      Database::set     (std::string const&       name,
                         std::string const&       pkey,
                         std::string const&       pkey_value,
                         VAttributesSet const&    attributes)
      {
        using namespace Glib;

        ustring sql ("UPDATE ");

        sql . append (name) . append (" SET ");
            
        VAttributesSet::const_iterator i_last = attributes.end();
        i_last--;

        for (VAttributesSet::const_iterator i = attributes.begin() ; i != attributes.end() ; ++i)
          {
            std::string const& attribute_name = boost::get<0>(*i);
            Variant     const& value          = boost::get<1>(*i);

            switch (value.which())
              {
                case VALUE_TYPE_STRING:
                  {
                    sql .append (sql_uprintf ("%s='%q' ", attribute_name.c_str(), boost::get<std::string>(value).c_str())); 
                    break;
                  }

                case VALUE_TYPE_INT:
                  {
                    sql .append (attribute_name)
                        .append ("='")
                        .append ((format_int % boost::get<uint64_t>(value)).str()).append("' ");
                    break;
                  }

                case VALUE_TYPE_BOOL:
                  {
                    sql .append (attribute_name)
                        .append ("='")
                        .append ((format_int % int(boost::get<bool>(value))).str()).append("' ");
                    break;
                  }
              }

            if (i != i_last) sql.append (" , ");

          }

        sql.append (sql_uprintf (" WHERE %s='%q';", pkey.c_str(), pkey_value.c_str())); 
        sqlite_exec_simple (sql.c_str());
      }

      void 
      Database::set (std::string    const& name,
                     Glib::ustring  const& where_clause,
                     VAttributesSet const& attributes)
      {
        using namespace Glib;

        ustring sql ("UPDATE ");
        sql . append (name) . append (" SET ");

        VAttributesSet::const_iterator i_last = attributes.end();
        i_last--;

        for (VAttributesSet::const_iterator i = attributes.begin() ; i != attributes.end() ; ++i)
          {
            std::string const& attribute_name = boost::get<0>(*i);
            Variant     const& value          = boost::get<1>(*i);

            switch (value.which())
            {
                case VALUE_TYPE_STRING:
                  {
                    sql.append (sql_uprintf ("%s='%q' ", attribute_name.c_str(), boost::get<std::string>(value).c_str())); 
                    break;
                  }

                case VALUE_TYPE_INT:
                  {
                    sql .append (attribute_name)
                        .append ("='")
                        .append ((format_int % boost::get<uint64_t>(value)).str()).append("' ");
                    break;
                  }

                case VALUE_TYPE_BOOL:
                  {
                    sql .append (attribute_name)
                        .append ("='")
                        .append((format_int % int(boost::get<bool>(value))).str()).append("' ");
                    break;
                  }
            }

            if (i != i_last) sql.append (" , ");
        }

        sql.append (where_clause);
        sql.append (";");

        sqlite_exec_simple (sql.c_str());
      }

      void 
      Database::get (std::string const& name,
                     sqlite_int64 rowid,
                     Row & row)
      {
        using namespace Glib;
        static boost::format lld ("%lld");

        if (mapmap.find (name) == mapmap.end ())
          {
            g_error ("No value-map for table '%s'", name.c_str());
          }

        ValueMap const& map (mapmap.find(name)->second);
        ValueMap::const_iterator i_last (map.end());   
        i_last--;

        ustring sql ("SELECT ");

        for (ValueMap::const_iterator i = map.begin() ; i != map.end() ; ++i)
          {
            sql.append (i->first);
            if (i != i_last) sql.append (",");
          }

        sql .append (" FROM ")
            .append (name)
            .append (" WHERE rowid='")
            .append ((lld % rowid).str())
            .append ("';"); 

        sqlite3_stmt *pStmt = 0;
        if (statement_prepare (&pStmt, sql, sqlite))
          { 
            throw SqlError (FORMAT_ERROR(sqlite3_errmsg (sqlite), sqlite3_errcode (sqlite), sql));
          }

        for ( int status = -1 ; status != SQLITE_DONE ; )
          {
            status = sqlite3_step (pStmt);
            switch (status)
              {
                case SQLITE_BUSY:   continue;
                case SQLITE_DONE:   break;
                case SQLITE_ERROR:  sqlite3_reset (pStmt); return;
                case SQLITE_MISUSE: sqlite3_reset (pStmt); return;
                case SQLITE_ROW:            
                  {
                    get_row_from_statement (pStmt, row, map);
                    sqlite3_finalize (pStmt);
                    return;
                  }
              } 
          } 
        sqlite3_finalize (pStmt);
      }

      void 
      Database::get (std::string const& name,
                     VAttributes const& attributes,
                     VRows & vector,
                     std::string const& suffix)
      {
        using namespace Glib;

        if (mapmap.find (name) == mapmap.end ())
          {
            g_error ("No value-map for table '%s'", name.c_str());
          }

        ValueMap const& map = (*(mapmap.find(name))).second;
        ValueMap::const_iterator i_last (map.end());   
        i_last--;

        ustring sql ("SELECT ");

        for (ValueMap::const_iterator i = map.begin() ; i != map.end() ; ++i)
          {
            sql.append (i->first);
            if (i != i_last) sql.append (",");
          }

        if (!attributes.empty ())
          {
            sql.append (" FROM ").append(name).append(" WHERE ");
            append_attributes (sql, attributes);
            if (suffix.empty())
              {
                sql.append (";");
              }  
            else
              {
                sql.append (" ").append(suffix).append(" ").append(";");
              }
          }
        else
          {
            sql.append (" FROM ").append(name);

            if (suffix.empty())
              {
                sql.append (";");
              }
            else
              {
                sql.append (" ").append(suffix).append(";");
              }
          }

        debug ("sql", "query: %s", sql.c_str());

        sqlite3_stmt *pStmt = 0;
        if (statement_prepare (&pStmt, sql, sqlite))
          {
            throw SqlError (FORMAT_ERROR(sqlite3_errmsg (sqlite), sqlite3_errcode (sqlite), sql));
          }
 
        for (int status = -1 ; status != SQLITE_DONE ; )
          {
            status = sqlite3_step (pStmt);
            switch (status)
              {
                case SQLITE_BUSY:   continue;
                case SQLITE_DONE:   break;
                case SQLITE_MISUSE: sqlite3_reset (pStmt); return;
                case SQLITE_ERROR:  sqlite3_reset (pStmt); return;
                case SQLITE_ROW:            
                  {
                    Bmp::DB::Row row;
                    get_row_from_statement (pStmt, row, map);
                    vector.push_back (row);
                    break;
                  }
              } 
          } 
          sqlite3_finalize (pStmt);
      }

      void
      Database::insert_map_for_table
                        (std::string const& name,
                         ValueMap const& map)
      {
        mapmap.insert (std::make_pair (name, map));
      }

      void
      Database::create_table (std::string const&  name,
                              std::string const&  pkey,
                              ValueMap    const&  map,
                              TableRowMode        row_mode)
      {
        using namespace Glib;

        mapmap.insert (std::make_pair (name, map));

        if (table_exists (name)) return;

        ustring sql;
        sql .append ("CREATE TABLE IF NOT EXISTS ")
            .append (name)
            .append (" (");

        ValueMap::const_iterator i_last;
        i_last = map.end ();
        i_last--;

        for (ValueMap::const_iterator i = map.begin (); i != map.end (); ++i)
          {
            if (!pkey.compare (i->first))
              {
                sql .append ("'")
                    .append (i->first)
                    .append ("' ")
                    .append (sql_valuenames[i->second])
                    .append (" NOT NULL PRIMARY KEY ");

                if (row_mode == ROW_MODE_REPLACE)
                  {
                    sql .append ("ON CONFLICT REPLACE ");
                  }
              }
            else
              {
                if (i->second == VALUE_TYPE_STRING)
                  {
                    sql .append ("'")
                        .append (i->first)
                        .append ("' ")
                        .append (sql_valuenames[i->second])
                        .append (" ");
                  }
                else
                  {
                    sql .append ("'")
                        .append (i->first)
                        .append ("' ")
                        .append (sql_valuenames[i->second])
                        .append (" DEFAULT 0");
                  }
              }

            if (i != i_last) sql.append (",");

          }

        sql.append (");");
        sqlite_exec_simple (sql.c_str());
      }

      void
      Database::drop_table (std::string const& name)
      {
        using namespace Glib;

        ustring sql;

        sql .append ("DROP TABLE ")
            .append (name)
            .append (";");

        sqlite_exec_simple (sql.c_str());
      }

      bool
      Database::table_exists    (std::string const& name)
      {
        using namespace Glib;

        ustring sql; 

        sql .append ("SELECT name FROM sqlite_master WHERE name='")
            .append (name)
            .append("';");

        unsigned int nrow = sqlite_exec_simple (sql.c_str()); 
        return (nrow != 0); 
      }

      bool
      Database::attr_exists (std::string const&   name, 
                             std::string const&   key,
                             Glib::ustring const& value)
      {
        using namespace Glib;

        ustring sql ("SELECT "); 

        sql .append (key)
            .append (" FROM ")
            .append (name)
            .append (" WHERE ")
            .append (key)
            .append ("='")
            .append (sql_uprintf ("%q", value.c_str()))
            .append ("';");

        unsigned int nrow = sqlite_exec_simple (sql.c_str()); 

        debug ("sql", "attr_exists: %u", nrow);
  
        return (nrow != 0); 
      }
  } 
}; 
