//
// Author: 
//   Mikael Hallendal <micke@imendio.com>
//
// (C) 2004 Imendio HB
// 

using Rss;
using System;
using System.Collections;
using System.IO;
using System.Reflection;
using System.Threading;
using System.Xml;
using System.Xml.Serialization;

namespace Imendio.Blam {

    public class ChannelCollection {

	public event ChannelEventHandler ChannelAdded;
	public event ChannelEventHandler ChannelUpdated;
	public event ChannelEventHandler ChannelRemoved;
	
	public event ChannelEventHandler ChannelRefreshStarted;
	public event ChannelEventHandler ChannelRefreshFinished;

	public string FileName;

	private Queue mQueue;
	private IList mRunningList;

	private bool mDirty = false;
	private uint mTimeoutId = 0;
	private static uint WRITE_TIMEOUT = 5 * 60 * 1000; // Every 5 minutes
	
	static XmlSerializer serializer = new XmlSerializer (typeof (ChannelCollection));
	
	private ArrayList mChannels;
	[XmlElement ("Channel", typeof (Channel))]
	public ArrayList Channels {
	    get {
		return mChannels;
	    }
	    set {
		mChannels = value;
	    }
	}

	public int NrOfUnreadItems {
	    get {
		int unread = 0;

		foreach (Channel channel in mChannels) {
		    unread += channel.NrOfUnreadItems;
		}
		
		return unread;
	    }
	}

	public ChannelCollection ()
	{
	    mQueue = Queue.Synchronized (new Queue ());
	    mRunningList = ArrayList.Synchronized (new ArrayList ());
	}

	public static ChannelCollection LoadFromFile (string file)
	{
	    ChannelCollection collection;

	    try {
		collection = RealLoadFromFile (file);
	    } catch {
		try {
		    collection = RealLoadFromFile (Defines.APP_DATADIR + "/collection.xml");
		} catch {
		    collection = new ChannelCollection ();
		    collection.mChannels = new ArrayList ();
		}
	    }

	    collection.FileName = file;

	    return collection;
	}

	private static ChannelCollection RealLoadFromFile (string file)
	{
	    ChannelCollection collection;
	    XmlTextReader reader = null;

	    try {
		reader = new XmlTextReader (file);
	    } catch (Exception) {
		reader = new XmlTextReader (GetTempFile (file));
	    }
	    
            collection = (ChannelCollection) serializer.Deserialize (reader);
            
	    reader.Close ();

            if (collection.Channels == null) {
                collection.mChannels = new ArrayList ();
            }

            foreach (Channel channel in collection.Channels) {
		channel.Setup ();
	    }

	    return collection;
	}
	
	public void SaveToFile ()
	{
	    lock (this) {
		if (!mDirty) {
		    return;
		}

		string tmpFile = GetTempFile (this.FileName);

		try {
		    Stream writer = new FileStream (tmpFile, FileMode.Create);
		    serializer.Serialize (writer, this);
		    writer.Close();
		} catch (Exception) {
		    System.Console.WriteLine ("Failed to save to temporary file");
		    return;
		}

		// Move the file to the real one
		try {
		    if (File.Exists (this.FileName)) {
			File.Delete (this.FileName);
		    }

		    File.Move (tmpFile, this.FileName);
		} catch (Exception e) {
		    System.Console.WriteLine ("Move failed: " + e.Message);
		    return;
		}
		
		MarkAsDirty (false);
	    }
	}

	public void Add (Channel channel)
	{
	    // Not the most efficiant way of doing things :)
	    foreach (Channel ch in mChannels) {
		if (channel.Url == ch.Url) {
		    return;
		}
	    }

	    mChannels.Add (channel);
	    Refresh (channel);

	    if (ChannelAdded != null) {
		ChannelAdded (channel);
	    }
	}

	public void Update (Channel channel)
	{
	    MarkAsDirty (true);

	    if (ChannelUpdated != null) {
		ChannelUpdated (channel);
	    }
	}

	public void Remove (Channel channel)
	{
	    mChannels.Remove (channel);

	    if (ChannelRemoved != null) {
		ChannelRemoved (channel);
	    }
	}

	public void Refresh (Channel channel) 
	{
	    mQueue.Enqueue (channel);
	    EmitChannelRefreshStarted (channel);

	    Thread thread = new Thread (new ThreadStart (UpdateThread));
	    mRunningList.Add (thread);

	    thread.Start ();
	}

	private void QueueChannelRefresh (Channel channel)
	{
	    mQueue.Enqueue (channel);
	    EmitChannelRefreshStarted (channel);
	}

	private void StartRefreshThreads (int maxNrOfThreads)
	{
	    // Only start a maximum of five threads
	    for (int i = 0; i < 5 && i < maxNrOfThreads; ++i) { 
		Thread thread = new Thread (new ThreadStart (UpdateThread));
		mRunningList.Add (thread);
		thread.Start ();
	    }
	}

	public void RefreshAll ()
	{
	    int nrOfChannels = 0;
	    
	    foreach (Channel channel in mChannels) {
		QueueChannelRefresh (channel);
		nrOfChannels++;
	    }

	    StartRefreshThreads (nrOfChannels);
	}

	public void RefreshAll (int refreshRate) 
	{
	    int nrOfChannels = 0;

	    foreach (Channel channel in mChannels) {
		TimeSpan span = DateTime.Now.Subtract (channel.LastRefreshed);
		
		if (span.TotalSeconds >= refreshRate * 60) {
		    QueueChannelRefresh (channel);
		    nrOfChannels++;
		}
	    }
	    
	    StartRefreshThreads (nrOfChannels);
	}
	
	/* Used to cross-mark as read */
	public void MarkItemIdAsReadInAllChannels (Channel channel, string id)
	{
	    foreach (Channel ch in mChannels) {
		if (ch != channel) {
		    ch.MarkItemIdAsRead (id);
		}
	    }
	}

	private void UpdateThread ()
	{
	    while (true) {
		try {
		    Channel channel = (Channel) mQueue.Dequeue ();
		    bool updated = FeedUpdater.Update (channel);

		    if (updated) {
			MarkAsDirty (true);
		    }	

		    new MainloopEmitter (this.ChannelRefreshFinished, channel).Emit ();
		} catch (InvalidOperationException) {
		    break;
		}
	    }

	    mRunningList.Remove (Thread.CurrentThread);
	}

	private void EmitChannelRefreshStarted (Channel channel)
	{
	    if (ChannelRefreshStarted != null) {
		ChannelRefreshStarted (channel);
	    }
	}
	
	private bool WriteTimeoutHandler ()
	{
	    SaveToFile ();
	    
	    return false;
	}

	private void MarkAsDirty (bool dirty)
	{
	    lock (this) {
		if (dirty) {
		    if (mTimeoutId != 0) {
			GLib.Source.Remove (mTimeoutId);
		    } 
		
		    mTimeoutId = GLib.Timeout.Add (WRITE_TIMEOUT, new GLib.TimeoutHandler (WriteTimeoutHandler));
		} else {
		    if (mTimeoutId != 0) {
			GLib.Source.Remove (mTimeoutId);
			mTimeoutId = 0;
		    }
		}

		mDirty = dirty;
	    }
	}

	private static string GetTempFile (string fileName) 
	{
	    return fileName + ".tmp";
	}
    }
}

