# $Id: LargeSeq.pm,v 1.10 2001/12/10 16:10:22 heikki Exp $
#
# BioPerl module for Bio::Seq::LargeSeq
#
# Cared for by Ewan Birney, Jason Stajich
#
# Copyright Ewan Birney, Jason Stajich
#
# You may distribute this module under the same terms as perl itself

# POD documentation - main docs before the code

=head1 NAME

Bio::Seq::LargeSeq - SeqI compliant object that stores sequence as files in /tmp 

=head1 SYNOPSIS

  # normal primary seq usage

=head1 DESCRIPTION

This object stores a sequence as a series of files in a temporary
directory. The aim is to allow someone the ability to store very large
sequences (eg, E<gt> 100MBases) in a file system without running out of memory
(eg, on a 64 MB real memory machine!). 

Of course, to actually make use of this functionality, the programs
which use this object B<must> not call $primary_seq-E<gt>seq otherwise the
entire sequence will come out into memory and probably paste your
machine. However, calls $primary_seq-E<gt>subseq(10,100) will cause only
90 characters to be brought into real memory.

=head1 FEEDBACK

=head2 Mailing Lists

User feedback is an integral part of the evolution of this
and other Bioperl modules. Send your comments and suggestions preferably
 to one of the Bioperl mailing lists.
Your participation is much appreciated.

  bioperl-l@bioperl.org          - General discussion
  http://www.bioperl.org/MailList.html             - About the mailing lists

=head2 Reporting Bugs

Report bugs to the Bioperl bug tracking system to help us keep track
 the bugs and their resolution.
 Bug reports can be submitted via email or the web:

  bioperl-bugs@bio.perl.org
  http://bio.perl.org/bioperl-bugs/

=head1 AUTHOR - Ewan Birney

Email birney@ebi.ac.uk

=head1 APPENDIX

The rest of the documentation details each of the object methods. Internal methods are usually preceded with a _

=cut


# Let the code begin...


package Bio::Seq::LargeSeq;
use vars qw($AUTOLOAD @ISA);
use strict;

# Object preamble - inherits from Bio::Root::Objecttest 8, 

use Bio::Seq::LargePrimarySeq;
use Bio::Seq;

@ISA = qw(Bio::Seq);


sub new {
    my ($class, @args) = @_;
    my $self = $class->SUPER::new(@args);
    
    my ($pseq) = $self->_rearrange([qw(PRIMARYSEQ)], @args);

    if( ! defined $pseq ) {
	$pseq = new Bio::Seq::LargePrimarySeq(@args);
    }
    $self->primary_seq($pseq);

    return $self;
}


=head2 trunc

 Title   : trunc
 Usage   : $subseq = $myseq->trunc(10,100);
 Function: Provides a truncation of a sequence,

 Example :
 Returns : a fresh Bio::SeqI object
 Args    :

=cut

sub trunc {
    my ($self, $s, $e) = @_;
    return new Bio::Seq::LargeSeq(
				  '-display_id' => $self->display_id,
				  '-accession_number' => $self->accession_number,
				  '-desc' => $self->desc,
				  '-alphabet' => $self->alphabet,
				  -primaryseq => 
				  $self->primary_seq->trunc($s,$e));

}

=head2 Bio::Seq::LargePrimarySeq methods

=cut

=head2 add_sequence_as_string

 Title   : add_sequence_as_string
 Usage   : $seq->add_sequence_as_string("CATGAT");
 Function: Appends additional residues to an existing LargePrimarySeq object.  
           This allows one to build up a large sequence without storing
           entire object in memory.
 Returns : Current length of sequence
 Args    : string to append

=cut

sub add_sequence_as_string {
    my ($self,$str) = @_;
    return $self->primary_seq->add_sequence_as_string($str);
}

1;
