/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <gio/gio.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#include "metadata-defines.h"

#define BUFFER_SIZE 8192
#define THUMBNAIL_SIZE 256

static void
set_thumbnail_size (GdkPixbufLoader *loader,
                    int              width,
                    int              height,
                    gpointer         userdata)
{
    int nw, nh;

    if (width <= THUMBNAIL_SIZE && height <= THUMBNAIL_SIZE) {
        return;
    }

    if (width > height) {
        float ratio;

        nw = THUMBNAIL_SIZE;
        ratio = (float) width / (float) THUMBNAIL_SIZE;
        nh = (int) ((float) height / ratio);
    } else {
        float ratio;

        nh = THUMBNAIL_SIZE;
        ratio = (float) height / (float) THUMBNAIL_SIZE;
        nw = (int) ((float) width / ratio);
    }

    gdk_pixbuf_loader_set_size (loader, nw, nh);
}

#if __WORDSIZE == 64
#define G_GSIZE_MODIFIER "l"
#endif
gboolean
bkl_task_pixbuf_thumbnail (GFile      *file,
                           GFileInfo  *info,
                           const char *mimetype,
                           GHashTable *metadata)
{
    GdkPixbufLoader *loader;
    GdkPixbuf *pixbuf;
    char data[BUFFER_SIZE];
    GInputStream *stream;
    GError *error = NULL;
    gssize count;
    char *uri, *filename, *fullname;

    stream = (GInputStream *) g_file_read (file, NULL, &error);
    if (stream == NULL) {
        char *uri;

        uri = g_file_get_uri (file);
        g_warning ("Error creating input stream for %s: %s", uri,
                   error->message);
        g_free (uri);
        g_error_free (error);

        return FALSE;
    }

    loader = gdk_pixbuf_loader_new ();
    g_signal_connect (loader, "size-prepared",
                      G_CALLBACK (set_thumbnail_size), NULL);

    count = g_input_stream_read (stream, data, BUFFER_SIZE, NULL, &error);
    if (count == -1) {
        uri = g_file_get_uri (file);
        g_warning ("Error reading from stream for %s: %s", uri, error->message);
        g_free (uri);
        g_error_free (error);

        g_input_stream_close (stream, NULL, NULL);
        g_object_unref (stream);
        g_object_unref (loader);

        return FALSE;
    }

    while (count > 0) {
        gdk_pixbuf_loader_write (loader, (guchar *) data, count, &error);
        if (error != NULL) {
            uri = g_file_get_uri (file);
            g_warning ("Error writing %" G_GSIZE_MODIFIER "d bytes to pixbuf for %s: %s",
                       count, uri, error->message);
            g_free (uri);
            g_error_free (error);

            g_input_stream_close (stream, NULL, NULL);
            g_object_unref (stream);
            g_object_unref (loader);

            return FALSE;
        }

        count = g_input_stream_read (stream, data, BUFFER_SIZE, NULL, &error);
        if (count == -1) {
            uri = g_file_get_uri (file);
            g_warning ("Error reading from stream for %s: %s",
                       uri, error->message);
            g_free (uri);
            g_error_free (error);

            g_input_stream_close (stream, NULL, NULL);
            g_object_unref (stream);
            g_object_unref (loader);

            return FALSE;
        }
    }

    g_input_stream_close (stream, NULL, NULL);
    g_object_unref (stream);

    gdk_pixbuf_loader_close (loader, &error);
    if (error != NULL) {
        g_warning ("Error closing pixbuf: %s", error->message);
        g_error_free (error);

        g_object_unref (loader);
        return FALSE;
    }

    pixbuf = gdk_pixbuf_loader_get_pixbuf (loader);

    uri = g_file_get_uri (file);
    filename = g_compute_checksum_for_string (G_CHECKSUM_MD5, uri, -1);

    fullname = g_build_filename (g_get_home_dir (), ".bkl-thumbnails",
                                 filename, NULL);
    g_free (filename);

    if (gdk_pixbuf_save (pixbuf, fullname, "png", &error, NULL) == FALSE) {
        g_warning ("Error writing file %s for %s: %s", fullname, uri,
                   error->message);
        g_error_free (error);
        g_free (uri);
        g_free (fullname);
        g_object_unref (loader);

        return FALSE;
    }

    g_object_unref (loader);
    g_free (uri);

    uri = g_filename_to_uri (fullname, NULL, &error);
    if (uri == NULL) {
        g_warning ("Error converting %s to uri: %s", fullname, error->message);
        g_error_free (error);
        g_free (fullname);

        return FALSE;
    }
    g_free (fullname);

    g_hash_table_insert (metadata, METADATA_EXTENDED_THUMBNAIL, uri);

    return TRUE;
}
