#include <errno.h>
#include <string.h>
#include <unistd.h>

#include "sysdeps.h"
#include "misc/misc.h"
#include "net/socket.h"

#include "dns.h"

#define MAX_PORT 65535
#define MIN_PORT 1025

static int bind_random(int sock)
{
  int i;
  for (i = 0; i < 10; ++i) {
    ipv4port port;
    port = random_scale(MAX_PORT - MIN_PORT) + MIN_PORT;
    if (socket_bind4(sock, &IPV4ADDR_ANY, port)) return 1;
  }
  return socket_bind4(sock, &IPV4ADDR_ANY, 0);
}

static void randomize_query(str* query)
{
  query->s[0] = random_trunc(0xff);
  query->s[1] = random_trunc(0xff);
}

static int bind_connect(int sock, const ipv4addr* ip)
{
  return bind_random(sock) && socket_connect4(sock, ip, 53);
}

static int transmit_udp(str* query, const ipv4addr* ip)
{
  int sock;
  if ((sock = socket_udp()) == -1) return sock;
  if (bind_connect(sock, ip)) {
    randomize_query(query);
    if (socket_send4(sock, query->s, query->len, 0, 0) == (long)query->len)
      return sock;
  }
  close(sock);
  return -1;
}

static int transmit_tcp(str* query, const ipv4addr* ip)
{
  int sock;
  if ((sock = socket_tcp()) == -1) return sock;
  if (bind_connect(sock, ip)) {
    randomize_query(query);
    if (write(sock, query->s, query->len) == (long)query->len)
      return sock;
  }
  close(sock);
  return -1;
}

static const unsigned timeouts[4] = { 1, 3, 11, 45 }; /* Total of 60 sec */
#define NUM_TIMEOUTS 4

static int receive_udp(int sock, int timeout, const str* query, str* response)
{
  char buf[512];
  long rd;
  for (;;) {
    iopoll_fd io;
    io.fd = sock;
    io.events = IOPOLL_READ;
    io.revents = 0;
    switch (iopoll(&io, 1, timeout*1000) == 1) {
    case -1: return -1;		/* System error */
    case 0:  return 0;		/* Timed out */
    default:
      if ((rd = socket_recv4(sock, buf, 512, 0, 0)) == -1)
	return -1;
      if (rd >= (long)query->len && memcmp(buf, query->s, 2) == 0) {
	if (!str_copyb(response, buf, rd)) {
	  errno = ENOMEM;
	  return -1;
	}
	return 1;
      }
    }
  }
}

static int query_udp(str* query, str* response)
{
  unsigned t;
  for (t = 0; t < NUM_TIMEOUTS; ++t) {
    unsigned i;
    for (i = 0; i < dns_server_count; ++i) {
      int sock;
      int r;
      if ((sock = transmit_udp(query, &dns_servers[i])) == -1) return -1;
      r = receive_udp(sock, timeouts[t], query, response);
      close(sock);
      if (r > 0) return r;
      if (r == 0) continue;
      return -1;
    }
  }
  return 0;
}

int dns_txrx(str* query, str* response)
{
  if (!dns_try_reload_config()) return -1;
  return query_udp(query, response);
}
