/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "WakeupPipe.h"
#include "Reactor.h"
#include <ace/config-lite.h>
#include <ace/IPC_SAP.h>
#include <ace/OS_NS_unistd.h>
#include <ace/OS_NS_sys_socket.h>
#include <ace/os_include/netinet/os_tcp.h> // FOR TCP_NODELAY

#ifdef ACE_WIN32
#  define READ_FUN  ACE_OS::recv
#  define WRITE_FUN ACE_OS::send
#  define CLOSE_FUN ACE_OS::closesocket
#else
#  define READ_FUN  ACE_OS::read
#  define WRITE_FUN ACE_OS::write
#  define CLOSE_FUN ACE_OS::close
#endif

static void setNonBlockingMode(ACE_HANDLE handle);

#ifdef ACE_WIN32
static int createSocketPairWin32(ACE_HANDLE handles[2]);
#endif

namespace ReactorHelpers {

WakeupPipe::WakeupPipe()
{
#ifdef ACE_WIN32
	if (createSocketPairWin32(m_handles) == -1) {
		throw Reactor::Exception("creating a socket pair failed");
	}
#else
	if (ACE_OS::pipe(m_handles) == -1) {
		throw Reactor::Exception("pipe() failed");
	}
#endif
	setNonBlockingMode(m_handles[0]);
	setNonBlockingMode(m_handles[1]);
}

WakeupPipe::~WakeupPipe()
{
	CLOSE_FUN(m_handles[0]);
	CLOSE_FUN(m_handles[1]);
}

void
WakeupPipe::activate()
{
	char const data = '\0';
	WRITE_FUN(m_handles[1], &data, 1);
}

void
WakeupPipe::deactivate()
{
	char buf[128];
	while (READ_FUN(m_handles[0], buf, sizeof(buf)) == sizeof(buf)) {}
}

} // namespace ReactorHelpers


void setNonBlockingMode(ACE_HANDLE handle)
{
	// ACE_IPC_SAP has a protected ctor.
	class SAP : public ACE_IPC_SAP {};
	
	SAP sap;
	sap.set_handle(handle);
	sap.enable(ACE_NONBLOCK);
}


#ifdef ACE_WIN32

#include "AutoClosingSAP.h"
#include <ace/INET_Addr.h>
#include <ace/SOCK_Acceptor.h>
#include <ace/SOCK_Connector.h>
#include <ace/SOCK_Stream.h>
#include <ace/Time_Value.h>
#include <ace/os_include/netinet/os_in.h>
#include <winsock2.h>
#include <stdlib.h>

/*
On Windows, the only way to create a pipe that would work with select(),
is to make a listening socket and connect another socket to it.
The following code is more complex than it could be, but it handles
a situation when something else connects to our listening socket before
we do it. Unfortunately, it doesn't handle a situation when so many
connections are made that our connection gets dropped. 
*/
int createSocketPairWin32(ACE_HANDLE handles[2])
{
	int const err = -1;
	
	AutoClosingSAP<ACE_SOCK_Acceptor> acceptor;

#if 0
	// This triggers a serious bug in Sunbelt Kerio Personal Firewall, as of 2006-05-25 
	ACE_INET_Addr bind_addr((u_short)0, (ACE_UINT32)INADDR_LOOPBACK);
	int const reuse_addr = 1;
	if (acceptor.open(bind_addr, reuse_addr) == err) {
		return err;
	}
#else
	sockaddr_in sa_in;
	memset(&sa_in, 0, sizeof(sa_in));
	sa_in.sin_family = AF_INET;
	sa_in.sin_port = 0;
	sa_in.sin_addr.s_addr = htonl(INADDR_LOOPBACK);
	
	int const reuse_addr = 1;
	int res = static_cast<ACE_SOCK&>(acceptor).open(
		SOCK_STREAM, AF_INET, 0, reuse_addr
	);
	if (res == err) {
		return err;
	}
	
	res = bind(
		(SOCKET)acceptor.get_handle(),
		(struct sockaddr*)&sa_in, sizeof(sa_in)
	);
	if (res != 0) {
		return err;
	}
	
	res = listen((SOCKET)acceptor.get_handle(), ACE_DEFAULT_BACKLOG);
	if (res != 0) {
		return err;
	}
#endif
	
	ACE_INET_Addr listen_addr;
	if (acceptor.get_local_addr(listen_addr) == err) {
		return err;
	}
	
	ACE_SOCK_Connector connector;
	AutoClosingSAP<ACE_SOCK_Stream> writer;
	
	int conn_res = connector.connect(writer, listen_addr, &ACE_Time_Value::zero);
	if (conn_res == err && errno != EWOULDBLOCK) {
		return err;
	}
	
	AutoClosingSAP<ACE_SOCK_Stream> reader;
	ACE_INET_Addr connected_from;
	ACE_INET_Addr connected_as;
	
	while (true) {
		reader.close();
		if (acceptor.accept(reader, &connected_from) == err) {
			if (errno == ECONNRESET) {
				// Something else has tried to connect,
				// but didn't complete the connection.
				continue;
			} else {
				return err;
			}
		}
		
		if (conn_res == err) { // we could already be connected
			conn_res = connector.complete(writer, 0, &ACE_Time_Value::zero);
			if (conn_res == err) {
				if (errno == EWOULDBLOCK) {
					// something else has connected
					continue;
				} else {
					return err;
				}
			}
		}
		
		if (writer.get_local_addr(connected_as) == err) {
			return err;
		}
		
		if (connected_from != connected_as) {
			// something else has connected
			continue;
		}
		
		break; // connection is established
	}
	
	int one = 1;
	writer.set_option(ACE_IPPROTO_TCP, TCP_NODELAY, &one, sizeof(one));
	
	handles[0] = reader.get_handle();
	handles[1] = writer.get_handle();
	
	// prevent auto-closing
	reader.release();
	writer.release();
		
	return 0;
}

#endif // ACE_WIN32
