/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "TrayMenu.h"
#include "TrayIcon.h"
#include "Application.h"
#include "GlobalState.h"
#include "Forwarding.h"
#include "ForwardingConfigFile.h"
#include <wx/menu.h>
#include <wx/event.h>
#include <wx/strconv.h>

namespace wxGUI
{

static ForwardingConfigFile& FCF()
{
	return wxGetApp().forwardingConfigFile();
}

static wxString utf8ToWxString(Forwarding::Utf8String const& utf8)
{
	return wxString(wxConvUTF8.cMB2WC(utf8.raw().c_str()), *wxConvCurrent);
}


class TrayMenu::Impl : public wxMenu
{
public:
	Impl();
private:
	void onBypass(wxCommandEvent& evt);
	
	void onConfigBasic(wxCommandEvent& evt);
	
	void onConfigAdvanced(wxCommandEvent& evt);

	void onConfigFilters(wxCommandEvent& evt);
	
	void onConfigForwarding(wxCommandEvent& evt);
	
	void onOperationLog(wxCommandEvent& evt);
	
	void onRequestLog(wxCommandEvent& evt);
	
	void onAbout(wxCommandEvent& evt);

	void onQuit(wxCommandEvent& evt);

	void onAny(wxCommandEvent& evt);

	void onFwdOptionSelected(wxCommandEvent& evt);

	DECLARE_EVENT_TABLE()
};


enum {
	ID_BYPASS,
	ID_CONFIG,
	ID_CONFIG_BASIC,
	ID_CONFIG_ADVANCED,
	ID_CONFIG_FILTERS,
	ID_CONFIG_FORWARDING,
	ID_LOG,
	ID_LOG_OPERATIONS,
	ID_LOG_REQUESTS,
	ID_ABOUT,
	ID_QUIT,
	ID_FIRST_FWD_OPTION = wxID_HIGHEST + 1 // must be the last option
};

BEGIN_EVENT_TABLE(TrayMenu::Impl, wxMenu)
	EVT_MENU(ID_BYPASS, onBypass)
	EVT_MENU(ID_CONFIG_BASIC, onConfigBasic)
	EVT_MENU(ID_CONFIG_ADVANCED, onConfigAdvanced)
	EVT_MENU(ID_CONFIG_FILTERS, onConfigFilters)
	EVT_MENU(ID_CONFIG_FORWARDING, onConfigForwarding)
	EVT_MENU(ID_LOG_OPERATIONS, onOperationLog)
	EVT_MENU(ID_LOG_REQUESTS, onRequestLog)
	EVT_MENU(ID_ABOUT, onAbout)
	EVT_MENU(ID_QUIT, onQuit)
	EVT_MENU(wxID_ANY, onAny)
END_EVENT_TABLE()


TrayMenu::TrayMenu()
{
}

TrayMenu::~TrayMenu()
{
}

void
TrayMenu::popup(TrayIcon& here)
{
	Impl impl;
	here.PopupMenu(&impl);
}


/*============================ TrayMenu::Impl ===========================*/

TrayMenu::Impl::Impl()
{
	AppendCheckItem(ID_BYPASS, _T("Bypass"));
	if (!GlobalState::ReadAccessor()->isFilteringEnabled()) {
		Check(ID_BYPASS, true);
	}
	
	AppendSeparator();
	
	if (!FCF().getForwardingConfig().options().empty()) {
		Forwarding::Config const& config = FCF().getForwardingConfig();
		Forwarding::OptionList::const_iterator opt(config.options().begin());
		Forwarding::OptionList::const_iterator const end(config.options().end());
		for (int i = 0; opt != end; ++opt, ++i) {
			int const id = ID_FIRST_FWD_OPTION + i;
			AppendRadioItem(id, utf8ToWxString(opt->getName()));
			if (opt->isSelected()) {
				Check(id, true);
			}
		}
		AppendSeparator();
	}

	wxMenu* config_submenu = new wxMenu;
	Append(ID_CONFIG, _T("Configuration"), config_submenu);
	config_submenu->Append(ID_CONFIG_BASIC, _T("Basic"));
	config_submenu->Append(ID_CONFIG_ADVANCED, _T("Advanced"));
	config_submenu->Append(ID_CONFIG_FORWARDING, _T("Forwarding"));
	config_submenu->Append(ID_CONFIG_FILTERS, _T("Filters"));
	
	wxMenu* log_submenu = new wxMenu;
	Append(ID_LOG, _T("Log"), log_submenu);
	log_submenu->Append(ID_LOG_OPERATIONS, _T("Operations"));
	log_submenu->Append(ID_LOG_REQUESTS, _T("Requests"));
	
	AppendSeparator();
	
	Append(ID_ABOUT, _T("About"));

	Append(ID_QUIT, _T("Quit"));
}

void
TrayMenu::Impl::onBypass(wxCommandEvent& evt)
{
	bool const bypass = evt.IsChecked();
	Check(ID_BYPASS, bypass);
	GlobalState::WriteAccessor()->setFilteringEnabled(!bypass);
}

void
TrayMenu::Impl::onConfigBasic(wxCommandEvent&)
{
	wxGetApp().showBasicConfigDialog();
}

void
TrayMenu::Impl::onConfigAdvanced(wxCommandEvent&)
{
	wxGetApp().showAdvancedConfigWindow();
}

void
TrayMenu::Impl::onConfigFilters(wxCommandEvent&)
{
	wxGetApp().showFilterConfigWindow();
}

void
TrayMenu::Impl::onConfigForwarding(wxCommandEvent&)
{
	wxGetApp().showForwardingConfigWindow();
}

void
TrayMenu::Impl::onOperationLog(wxCommandEvent&)
{
	wxGetApp().showLogDialog();
}

void
TrayMenu::Impl::onRequestLog(wxCommandEvent&)
{
	wxGetApp().showRequestLogWindow();
}

void
TrayMenu::Impl::onAbout(wxCommandEvent&)
{
	wxGetApp().showAboutDialog();
}

void
TrayMenu::Impl::onQuit(wxCommandEvent&)
{
	wxGetApp().requestAppExit();
}

void
TrayMenu::Impl::onAny(wxCommandEvent& evt)
{
	if (evt.GetId() >= ID_FIRST_FWD_OPTION) {
		onFwdOptionSelected(evt);
	} else {
		evt.Skip();
	}
}

void
TrayMenu::Impl::onFwdOptionSelected(wxCommandEvent& evt)
{
	int const option_idx = evt.GetId() - ID_FIRST_FWD_OPTION;
	assert(option_idx >= 0);

	Forwarding::Config new_config(FCF().getForwardingConfig());
	Forwarding::OptionList::iterator it(new_config.options().begin());
	Forwarding::OptionList::iterator const end(new_config.options().end());
	for (int i = 0; it != end; ++i, ++it) {
		it->setSelected(i == option_idx);
	}

	if (!FCF().applyAndSave(new_config)) {
		return;
	}
	
	Check(evt.GetId(), true);
}

} // namespace wxGUI
