/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "TrayMenu.h"
#include "Application.h"
#include "GlobalState.h"
#include "Forwarding.h"
#include "ForwardingConfigFile.h"
#include "ScopedIncDec.h"
#include <gtkmm/menu.h>
#include <gtkmm/stock.h>
#include <gtkmm/radiobuttongroup.h>
#include <gtkmm/radiomenuitem.h>
#include <memory>

using namespace std;

namespace GtkGUI
{

static ForwardingConfigFile& FCF()
{
	return Application::instance()->forwardingConfigFile();
}

class TrayMenu::Impl : public Gtk::Menu
{
public:
	Impl();
	
	void update();
private:
	void onBypass();
	
	void onFwdOptionSelected(int option_idx);
	
	void onBasicConfig();
	
	void onAdvancedConfig();
	
	void onForwardingConfig();
	
	void onFilterConfig();
	
	void onOperationLog();
	
	void onRequestLog();
#ifdef DEBUG
	void onDebug();
#endif
	void onAbout();
	
	void onQuit();
	
	Gtk::CheckMenuItem* m_pBypassItem;
	Gtk::RadioButtonGroup m_forwardingRadioGroup;
	Gtk::Menu_Helpers::MenuList::iterator m_forwardingBegin;
	Gtk::Menu_Helpers::MenuList::iterator m_forwardingEnd;
	int m_ignoreFwdSelection;
};


TrayMenu::TrayMenu()
:	m_ptrImpl(new Impl)
{
}

TrayMenu::~TrayMenu()
{
}

void
TrayMenu::popup(guint button, guint32 activate_time)
{
	m_ptrImpl->update();
	m_ptrImpl->popup(button, activate_time);
}


/*=========================== TrayMenu::Impl ============================*/

TrayMenu::Impl::Impl()
:	m_pBypassItem(0),
	m_ignoreFwdSelection(0)
{
	items().push_back(
		Gtk::Menu_Helpers::CheckMenuElem(
			"Bypass",
			sigc::mem_fun(*this, &Impl::onBypass)
		)
	);
	m_pBypassItem = dynamic_cast<Gtk::CheckMenuItem*>(&items().back());
	
	m_forwardingBegin = m_forwardingEnd = items().insert(
		items().end(), Gtk::Menu_Helpers::SeparatorElem()
	);
	// m_forwardingEnd will not be modified.
	
#ifdef DEBUG
	items().push_back(
		Gtk::Menu_Helpers::MenuElem(
			"Debug",
			sigc::mem_fun(*this, &Impl::onDebug)
		)
	);
#endif
	
	auto_ptr<Gtk::Menu> config_submenu(new Gtk::Menu);
	config_submenu->items().push_back(
		Gtk::Menu_Helpers::MenuElem(
			"Basic",
			sigc::mem_fun(*this, &Impl::onBasicConfig)
		)
	);
	config_submenu->items().push_back(
		Gtk::Menu_Helpers::MenuElem(
			"Advanced",
			sigc::mem_fun(*this, &Impl::onAdvancedConfig)
		)
	);
	config_submenu->items().push_back(
		Gtk::Menu_Helpers::MenuElem(
			"Forwarding",
			sigc::mem_fun(*this, &Impl::onForwardingConfig)
		)
	);
	config_submenu->items().push_back(
		Gtk::Menu_Helpers::MenuElem(
			"Filters",
			sigc::mem_fun(*this, &Impl::onFilterConfig)
		)
	);
	
	items().push_back(
		Gtk::Menu_Helpers::MenuElem(
			"Configuration", *manage(config_submenu.release())
		)
	);
	
	auto_ptr<Gtk::Menu> log_submenu(new Gtk::Menu);
	log_submenu->items().push_back(
		Gtk::Menu_Helpers::MenuElem(
			"Operations",
			sigc::mem_fun(*this, &Impl::onOperationLog)
		)
	);
	log_submenu->items().push_back(
		Gtk::Menu_Helpers::MenuElem(
			"Requests",
			sigc::mem_fun(*this, &Impl::onRequestLog)
		)
	);
	
	items().push_back(
		Gtk::Menu_Helpers::MenuElem(
			"Log", *manage(log_submenu.release())
		)
	);
	
	items().push_back(Gtk::Menu_Helpers::SeparatorElem());
	
	items().push_back(
		Gtk::Menu_Helpers::MenuElem(
			"About",
			sigc::mem_fun(*this, &Impl::onAbout)
		)
	);
	items().push_back(
		Gtk::Menu_Helpers::StockMenuElem(
			Gtk::Stock::QUIT,
			sigc::mem_fun(*this, &Impl::onQuit)
		)
	);
	
	bool filtering_enabled = GlobalState::ReadAccessor()->isFilteringEnabled();
	// Calling set_active(GlibalState::ReadAccessor()->...) would lead to a deadlock.
	m_pBypassItem->set_active(!filtering_enabled);
}

void
TrayMenu::Impl::update()
{
	ScopedIncDec<int> guard(m_ignoreFwdSelection);
	
	items().erase(m_forwardingBegin, m_forwardingEnd);
	// m_forwardingEnd should still be valid
	m_forwardingBegin = m_forwardingEnd;
	
	// This shouldn't be necessary, but for some reason (bug in gtkmm?) it is.
	m_forwardingRadioGroup = Gtk::RadioButtonGroup();
	
	if (!FCF().getForwardingConfig().options().empty()) {
		m_forwardingBegin = items().insert(
			m_forwardingEnd, Gtk::Menu_Helpers::SeparatorElem()
		);
		
		Forwarding::Config const& config = FCF().getForwardingConfig();
		Forwarding::OptionList::const_iterator opt(config.options().begin());
		Forwarding::OptionList::const_iterator const end(config.options().end());
		for (int i = 0; opt != end; ++opt, ++i) {
			sigc::slot<void> slot(
				sigc::bind(sigc::mem_fun(*this, &Impl::onFwdOptionSelected), i)
			);
			
			Gtk::MenuItem& item = *items().insert(
				m_forwardingEnd,
				Gtk::Menu_Helpers::RadioMenuElem(
					m_forwardingRadioGroup,
					opt->getName().raw(), slot
				)
			);
			
			if (opt->isSelected()) {
				dynamic_cast<Gtk::RadioMenuItem*>(&item)->set_active();
			}
		}
	}
}

void
TrayMenu::Impl::onBypass()
{
	bool const bypass = m_pBypassItem->get_active();
	GlobalState::WriteAccessor()->setFilteringEnabled(!bypass);
}

void
TrayMenu::Impl::onFwdOptionSelected(int const option_idx)
{
	if (m_ignoreFwdSelection) {
		return;
	}
	
	Forwarding::Config new_config(FCF().getForwardingConfig());
	Forwarding::OptionList::iterator it(new_config.options().begin());
	Forwarding::OptionList::iterator const end(new_config.options().end());
	for (int i = 0; it != end; ++it, ++i) {
		it->setSelected(i == option_idx);
	}

	FCF().applyAndSave(new_config);
}

void
TrayMenu::Impl::onBasicConfig()
{
	Application::instance()->showBasicConfigDialog();
}

void
TrayMenu::Impl::onAdvancedConfig()
{
	Application::instance()->showAdvancedConfigWindow();
}

void
TrayMenu::Impl::onForwardingConfig()
{
	Application::instance()->showForwardingConfigWindow();
}

void
TrayMenu::Impl::onFilterConfig()
{
	Application::instance()->showFilterConfigWindow();
}

void
TrayMenu::Impl::onOperationLog()
{
	Application::instance()->showLogDialog();
}

void
TrayMenu::Impl::onRequestLog()
{
	Application::instance()->showRequestLogWindow();
}

#ifdef DEBUG
void
TrayMenu::Impl::onDebug()
{
	Application::instance()->showDebugWindow();
}
#endif

void
TrayMenu::Impl::onAbout()
{
	Application::instance()->showAboutDialog();
}

void
TrayMenu::Impl::onQuit()
{
	Application::instance()->requestAppExit();
}

} // namespace GtkGUI
