/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "PortSet.h"
#include "StringUtils.h"
#include "BString.h"
#include <ostream>

using namespace std;

class PortSet::Parser
{
public:
	bool parse(char const* begin, char const* end);
	
	std::vector<Range> result() const { return m_result; }
private:
	bool parseRange();
	
	bool parsePossibleComma();
	
	bool parsePort(unsigned& port);
	
	std::vector<Range> m_result;
	char const* m_pCur;
	char const* m_pEnd;
};

bool
PortSet::contains(unsigned port) const
{
	vector<Range>::const_iterator it(m_portRanges.begin());
	vector<Range>::const_iterator const end(m_portRanges.end());
	for (; it != end; ++it) {
		if (it->isInRange(port)) {
			return true;
		}
	}
	return false;
}

void
PortSet::toStream(std::ostream& strm) const
{
	vector<Range>::const_iterator it(m_portRanges.begin());
	vector<Range>::const_iterator const end(m_portRanges.end());
	bool first = true;
	for (; it != end; ++it) {
		if (first) {
			first = false;
		} else {
			strm << ", ";
		}
		it->toStream(strm);
	}
}

bool
PortSet::fromString(std::string const& str)
{
	char const* begin = str.c_str();
	char const* end = begin + str.size();
	Parser parser;
	if (!parser.parse(begin, end)) {
		return false;
	}
	parser.result().swap(m_portRanges);
	return true;
}

void
PortSet::swap(PortSet& other)
{
	m_portRanges.swap(other.m_portRanges);
}


/*========================== PortSet::Range ============================*/

bool
PortSet::Range::isInRange(unsigned port) const
{
	return port >= from && port <= to;
}

void
PortSet::Range::toStream(std::ostream& strm) const
{
	strm << from;
	if (from != to) {
		strm << "..";
		strm << to;
	}
}


/*========================== PortSet::Parser ===========================*/

bool
PortSet::Parser::parse(char const* begin, char const* end)
{
	m_result.clear();
	m_pCur = begin;
	m_pEnd = end;
	
	while (m_pCur != m_pEnd) {
		m_pCur = StringUtils::ltrim(m_pCur, m_pEnd);
		if (!parseRange()) {
			return false;
		}
		if (!parsePossibleComma()) {
			return false;
		}
	}
	return true;
}

bool
PortSet::Parser::parsePossibleComma()
{
	m_pCur = StringUtils::ltrim(m_pCur, m_pEnd);
	if (m_pCur == m_pEnd) {
		return true;
	}
	
	if (*m_pCur == ',') {
		++m_pCur;
		return true;
	} else {
		return false;
	}
}

bool
PortSet::Parser::parseRange()
{
	unsigned port1 = 0;
	unsigned port2 = 0;
	if (!parsePort(port1)) {
		return false;
	}
	
	BString const sep("..");
	if (!StringUtils::startsWith(m_pCur, m_pEnd, sep.begin(), sep.end())) {
		port2 = port1;
	} else {
		m_pCur += sep.size();
		if (!parsePort(port2)) {
			return false;
		}
	}
	m_result.push_back(Range(port1, port2));
	return true;
}

bool
PortSet::Parser::parsePort(unsigned& port)
{
	char const* end = m_pEnd;
	unsigned prt = StringUtils::parseUnsigned<unsigned>(m_pCur, end);
	if (end == m_pCur) {
		return false;
	} else {
		m_pCur = end;
		port = prt;
		return true;
	}
}
