/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "LimitedSpaceManager.h"
#include <algorithm>
#include <assert.h>

namespace HttpCache
{

/*========================= LimitedSpaceObject =========================*/

LimitedSpaceObject::LimitedSpaceObject(
	detail::ManagerHandle* manager_handle, off_t const size)
:	m_confirmedSize(size),
	m_ptrManagerHandle(manager_handle)
{
}

LimitedSpaceObject::~LimitedSpaceObject()
{
}

void
LimitedSpaceObject::unref() const
{
	int32_t const refs = --m_numRefs;
	if (refs == m_sGhostOffset) {
		m_ptrManagerHandle->onGhostGone(m_confirmedSize);
		delete this;
	} else if (refs == 0) {
		delete this;
	}
}

off_t
LimitedSpaceObject::getPotentialSize() const
{
	if (m_reservations.empty()) {
		return m_confirmedSize;
	}
	
	off_t const largest_reservation = (--m_reservations.end())->getRequestedSize();
	return std::max(m_confirmedSize, largest_reservation);
}

off_t
LimitedSpaceObject::getPotentialSizeWith(off_t const size_request) const
{
	return std::max(getPotentialSize(), size_request);
}

void
LimitedSpaceObject::addReservation(
	detail::ResizeRequestBase& request, IntrusiveListNode& node)
{
	ReservationList::iterator const begin(m_reservations.begin());
	ReservationList::iterator const end(m_reservations.end());
	ReservationList::iterator next(end);
	ReservationList::iterator prev(next);
	for (; next != begin; --next) {
		--prev;
		if (prev->getRequestedSize() <= request.getRequestedSize()) {
			break;
		}
	}
	m_reservations.insert(next, request, node); // insert before next
}


/*======================= LimitedSpaceManagerBase ========================*/

LimitedSpaceManagerBase::LimitedSpaceManagerBase(
	uint64_t const space_limit, IntrusivePtr<AbstractCommand> const& gc_invoker)
:	m_ptrGcInvoker(gc_invoker),
	m_spaceLimit(space_limit),
	m_usedSpace(0)
{
}

LimitedSpaceManagerBase::~LimitedSpaceManagerBase()
{
}

void
LimitedSpaceManagerBase::ref() const
{
	++m_refCounter;
}

void
LimitedSpaceManagerBase::unref() const
{
	if (--m_refCounter == 0) {
		delete this;
	}
}

void
LimitedSpaceManagerBase::setSpaceLimit(uint64_t const limit)
{
	ACE_Guard<Mutex> guard(m_mutex);
	m_spaceLimit = limit;
}

bool
LimitedSpaceManagerBase::requestResize(
	LimitedSpaceObject& obj, detail::ResizeRequestBase& request,
	IntrusiveListNode& node, off_t const requested_size)
{
	assert(requested_size >= 0);
	
	for (int i = 0; i < 2; ++i) {
		if (i != 0) {
			assert(m_ptrGcInvoker.get());
			(*m_ptrGcInvoker)();
		}
		
		ACE_Guard<Mutex> guard(m_mutex);
		
		off_t const old_potential_size = obj.getPotentialSize();
		off_t const new_potential_size = obj.getPotentialSizeWith(requested_size);
		off_t const extra_space = new_potential_size - old_potential_size;
		assert(extra_space >= 0);
		
		if (extra_space > 0 && m_usedSpace + extra_space > m_spaceLimit) {
			if (m_ptrGcInvoker.get()) {
				continue;
			} else {
				break;
			}
		}
		
		obj.addReservation(request, node);
		m_usedSpace += extra_space;
		return true;
	}
	
	return false;
}
	
void
LimitedSpaceManagerBase::commitResize(
	LimitedSpaceObject& obj, detail::ResizeRequestBase& request,
	IntrusiveListNode& node, off_t const new_size, bool const only_grow)
{
	assert(new_size >= -1); // -1 means no resize took place
	
	ACE_Guard<Mutex> guard(m_mutex);
	
	off_t const old_potential_size = obj.getPotentialSize();
	node.unlink();
	
	if (only_grow) {
		if (new_size > obj.getConfirmedSize()) {
			obj.setConfirmedSize(new_size);
		}
	} else {
		if (new_size != -1) {
			obj.setConfirmedSize(new_size);
		}
	}
	
	off_t const new_potential_size = obj.getPotentialSize();
	off_t const freed_space = old_potential_size - new_potential_size;
	assert(freed_space >= 0);
	m_usedSpace -= freed_space;
}

void
LimitedSpaceManagerBase::onGhostGone(off_t const confirmed_size)
{
	ACE_Guard<Mutex> guard(m_mutex);
	m_usedSpace -= confirmed_size;
}


/*============================= ResizeRequest ============================*/

ResizeRequest::ResizeRequest(
	IntrusivePtr<LimitedSpaceObject> const& obj, off_t const requested_size)
:	detail::ResizeRequestBase(requested_size),
	m_ptrObj(obj),
	m_requestGranted(false),
	m_confirmed(false)
{
	assert(requested_size >= 0);
	
	m_requestGranted = m_ptrObj->requestResize(*this, m_listNode, requested_size);
}

ResizeRequest::~ResizeRequest()
{	
	if (m_requestGranted) {
		off_t const new_size = m_confirmed ? getRequestedSize() : -1;
		m_ptrObj->commitResize(*this, m_listNode, new_size, false);
	}
}


/*============================== GrowRequest =============================*/

GrowRequest::GrowRequest(
	IntrusivePtr<LimitedSpaceObject> const& obj, off_t const requested_size)
:	detail::ResizeRequestBase(requested_size),
	m_newSize(-1),
	m_ptrObj(obj),
	m_requestGranted(false)
{
	assert(requested_size >= 0);
	
	m_requestGranted = m_ptrObj->requestResize(*this, m_listNode, requested_size);
}

GrowRequest::~GrowRequest()
{
	assert(m_newSize <= getRequestedSize());
	
	if (m_requestGranted) {
		m_ptrObj->commitResize(*this, m_listNode, m_newSize, true);
	}
}

} // namespace HttpCache
