/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ResponseCacheControl.h"
#include "HttpResponseMetadata.h"
#include "HttpHeadersCollection.h"
#include "HttpHeader.h"
#include "HttpHeaderStructure.h"
#include "HttpHeaderElement.h"
#include "BString.h"
#include "InsensitiveEqual.h"
#include "StringUtils.h"
#include <list>

ResponseCacheControl::ResponseCacheControl()
:	max_age(0),
	s_maxage(0),
	max_age_set(false),
	s_maxage_set(false),
	publ(false),
	priv(false),
	no_cache(false),
	no_store(false),
	no_transform(false),
	must_revalidate(false),
	proxy_revalidate(false)
{
}

ResponseCacheControl::ResponseCacheControl(HttpResponseMetadata const& request)
:	max_age(0),
	s_maxage(0),
	max_age_set(false),
	s_maxage_set(false),
	publ(false),
	priv(false),
	no_cache(false),
	no_store(false),
	no_transform(false),
	must_revalidate(false),
	proxy_revalidate(false)
{
	processCacheControl(request.headers().getHeaderPtr(BString("Cache-Control")));
}

void
ResponseCacheControl::processCacheControl(HttpHeader const* cc_hdr)
{
	if (!cc_hdr) {
		return;
	}
	InsensitiveEqual ieq;
	HttpHeaderStructure const structure(*cc_hdr);
	std::list<HttpHeaderElement>::const_iterator it(structure.elements().begin());
	std::list<HttpHeaderElement>::const_iterator const end(structure.elements().end());
	for (; it != end; ++it) {
		BString const& name = it->getName();
		if (!name.empty()) {
			switch (name[0]) {
			case 'n':
			case 'N':
				if (ieq(name, BString("no-cache"))) {
					no_cache = true;
				} else if (ieq(name, BString("no-store"))) {
					no_store = true;
				} else if (ieq(name, BString("no-transform"))) {
					no_transform = true;
				}
				break;
			case 'm':
			case 'M':
				if (ieq(name, BString("must-revalidate"))) {
					must_revalidate = true;
				} else if (ieq(name, BString("max-age"))) {
					max_age_set = true;
					max_age = parseInt(it->getValue());
				}
				break;
			case 'p':
			case 'P':
				if (ieq(name, BString("proxy-revalidate"))) {
					proxy_revalidate = true;
				} else if (ieq(name, BString("public"))) {
					publ = true;
				} else if (ieq(name, BString("private"))) {
					priv = true;
				}
				break;
			case 's':
			case 'S':
				if (ieq(name, BString("s-maxage"))) {
					s_maxage_set = true;
					s_maxage = parseInt(it->getValue());
				}
			}
		}
	}
}

int
ResponseCacheControl::parseInt(BString const& val)
{
	char const* end = val.end();
	int res = StringUtils::parseSigned<int>(val.begin(), end);
	if (end != val.end()) {
		res = 0;
	}
	return res;
}
