/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "RequestCacheControl.h"
#include "HttpRequestMetadata.h"
#include "HttpHeadersCollection.h"
#include "HttpHeader.h"
#include "HttpHeaderStructure.h"
#include "HttpHeaderElement.h"
#include "BString.h"
#include "InsensitiveEqual.h"
#include "StringUtils.h"
#include <list>

RequestCacheControl::RequestCacheControl()
:	max_age(0),
	max_stale(0),
	min_fresh(0),
	max_age_set(false),
	max_stale_set(false),
	max_stale_has_value(false),
	min_fresh_set(false),
	no_cache(false),
	no_store(false),
	no_transform(false),
	only_if_cached(false)
{
}

RequestCacheControl::RequestCacheControl(HttpRequestMetadata const& request)
:	max_age(0),
	max_stale(0),
	min_fresh(0),
	max_age_set(false),
	max_stale_set(false),
	max_stale_has_value(false),
	min_fresh_set(false),
	no_cache(false),
	no_store(false),
	no_transform(false),
	only_if_cached(false)
{
	processPragma(request.headers().getHeaderPtr(BString("Pragma")));
	processCacheControl(request.headers().getHeaderPtr(BString("Cache-Control")));
}

void
RequestCacheControl::processPragma(HttpHeader const* pragma_hdr)
{
	if (!pragma_hdr) {
		return;
	}
	HttpHeaderStructure const structure(*pragma_hdr);
	if (structure.hasElement(BString("no-cache"))) {
		no_cache = true;
	}
}

void
RequestCacheControl::processCacheControl(HttpHeader const* cc_hdr)
{
	if (!cc_hdr) {
		return;
	}
	InsensitiveEqual ieq;
	HttpHeaderStructure const structure(*cc_hdr);
	std::list<HttpHeaderElement>::const_iterator it(structure.elements().begin());
	std::list<HttpHeaderElement>::const_iterator const end(structure.elements().end());
	for (; it != end; ++it) {
		BString const& name = it->getName();
		if (!name.empty()) {
			switch (name[0]) {
			case 'n':
			case 'N':
				if (ieq(name, BString("no-cache"))) {
					no_cache = true;
				} else if (ieq(name, BString("no-store"))) {
					no_store = true;
				} else if (ieq(name, BString("no-transform"))) {
					no_transform = true;
				}
				break;
			case 'm':
			case 'M':
				if (ieq(name, BString("max-age"))) {
					max_age_set = true;
					max_age = parseInt(it->getValue());
				} else if (ieq(name, BString("max-stale"))) {
					max_stale_set = true;
					BString const& val = it->getValue();
					if (!val.empty()) {
						max_stale_has_value = true;
						max_stale = parseInt(val);
					}
				} else if (ieq(name, BString("min-fresh"))) {
					min_fresh_set = true;
					min_fresh = parseInt(it->getValue());
				}
				break;
			case 'o':
			case 'O':
				if (ieq(name, BString("only-if-cached"))) {
					only_if_cached = true;
				}
			}
		}
	}
}

int
RequestCacheControl::parseInt(BString const& val)
{
	char const* end = val.end();
	int res = StringUtils::parseSigned<int>(val.begin(), end);
	if (end != val.end()) {
		res = 0;
	}
	return res;
}
