/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ETag.h"
#include "BString.h"
#include "HttpHeader.h"
#include "DataChunk.h"
#include <memory>
#include <algorithm>
#include <string.h>
#include <stddef.h>
#include <assert.h>

ETag::ETag()
:	m_flags(ETAG_NULL)
{
}

ETag::ETag(HttpHeader const& header)
:	m_flags(0)
{
	BString const& value = header.getValue();
	parse(value.begin(), value.end());
}

ETag::ETag(BString const& etag)
:	m_flags(0)
{
	parse(etag.begin(), etag.end());
}

ETag::ETag(std::string const& etag)
:	m_flags(0)
{
	parse(etag.c_str(), etag.c_str() + etag.size());
}

ETag::ETag(char const* str, size_t len)
:	m_flags(0)
{
	parse(str, str + len);
}

ETag::~ETag()
{
}

void
ETag::swap(ETag& other)
{
	m_value.swap(other.m_value);
	std::swap(m_flags, other.m_flags);
}

BString
ETag::toBString() const
{
	if (!isValid()) {
		return BString();
	}
	
	std::auto_ptr<DataChunk> chunk(DataChunk::create(m_value.size() + 2));
	char* p = chunk->getDataAddr();
	size_t skip_back = 0;
	if (isWeak()) {
		*p++ = 'W';
		*p++ = '/';
	} else {
		skip_back = 2;
	}
	memcpy(p, m_value.c_str(), m_value.length());
	
	return BString(chunk, 0, skip_back);
}

bool
ETag::strongIdentity(ETag const& other) const
{
	if ((m_flags | other.m_flags) & ETAG_WEAK) {
		return false;
	}
	if (!(m_flags & other.m_flags & ETAG_VALID)) {
		return false;
	}
	return m_value == other.m_value;
}

bool
ETag::weakIdentity(ETag const& other) const
{
	if (!(m_flags & other.m_flags & ETAG_VALID)) {
		return false;
	}
	return m_value == other.m_value;
}

void
ETag::parse(char const* begin, char const* end)
{
	assert(m_flags == 0 && m_value.empty());
	
	if (begin == end) {
		m_flags = ETAG_NULL;
		return;
	}
	
	if (end - begin < 2) {
		return;
	}
	
	if (end[-1] != '"') {
		return;
	}
	
	if (*begin == 'W' || *begin == 'w') {
		++begin;
		assert(begin != end); // because end - begin < 2
		if (*begin == '/') {
			++begin;
			m_flags |= ETAG_WEAK;
		} else {
			return;
		}
	}
	
	if (*begin == '"') {
		m_value.assign(begin, end);
		m_flags |= ETAG_VALID;
	}
}
