/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "DnsResolver.h"
#include "DnsCache.h"
#include "InetAddr.h"
#include "SymbolicInetAddr.h"
#include <ace/config-lite.h>
#include <ace/OS_NS_sys_socket.h>
#include <ace/OS_NS_netdb.h>
#include <ace/OS_NS_arpa_inet.h>
#include <cstdlib>
#include <stddef.h>

using namespace std;

#ifdef ACE_HAS_IPV6
class DnsResolver::AddrInfoHolder
{
public:
	AddrInfoHolder(struct addrinfo* ai) : m_pAddrInfo(ai) {}
	
	~AddrInfoHolder();
	
	struct addrinfo* get() const { return m_pAddrInfo; }
private:
	struct addrinfo* m_pAddrInfo;
};

DnsResolver::AddrInfoHolder::~AddrInfoHolder()
{
	if (m_pAddrInfo) {
		freeaddrinfo(m_pAddrInfo);
	}
}
#endif

std::vector<InetAddr>
DnsResolver::resolve(SymbolicInetAddr const& symbolic_addr, ErrorCode* err)
{
	if (symbolic_addr.getHost().empty()) {
		InetAddr addr;
		addr.set(symbolic_addr.getPort());
		return vector<InetAddr>(1, addr);
	}
	
	std::string const& host = symbolic_addr.getHost();
	if (host.c_str()[0] == '[' && host.c_str()[host.size()-1] == ']') {
		// IPv6 literal address, as described in RFC 2732
		std::string new_host(host, 1, host.size()-2);
		SymbolicInetAddr new_addr(new_host, symbolic_addr.getPort());
		return resolveWithCache(new_addr, err);
	}
	return resolveWithCache(symbolic_addr, err);
}

std::vector<InetAddr>
DnsResolver::resolveWithCache(SymbolicInetAddr const& symbolic_addr, ErrorCode* err)
{
	vector<InetAddr> res = DnsCache::instance()->get(symbolic_addr);
	if (!res.empty()) {
		if (err) {
			*err = SUCCESS;
		}
		return res;
	}
	
	res = realResolve(symbolic_addr, err);
	if (!res.empty()) {
		DnsCache::instance()->put(symbolic_addr, res);
	}
	
	return res;
}

std::vector<InetAddr>
DnsResolver::realResolve(SymbolicInetAddr const& symbolic_addr, ErrorCode* err)
{
	char const* const host = symbolic_addr.getHost().c_str();
	u_short const port = (unsigned)symbolic_addr.getPort();
	std::vector<InetAddr> vec;

#ifdef ACE_HAS_IPV6
	struct addrinfo hints;
	memset(&hints, 0, sizeof(hints));
	hints.ai_socktype = SOCK_STREAM;
	/*
	Without hints, getaddrinfo() would return a separate address
	for each socktype it knows about. Since socktype is not part
	of sockaddr (nor ACE_INET_Addr, which is a wrapper around sockaddr),
	specifying a SOCK_STREAM hint doesn't actually limit us to
	stream sockets.
	*/
	
	struct addrinfo* ai = 0;
	int eai = getaddrinfo(host, 0, &hints, &ai);
	AddrInfoHolder ai_holder(ai);
	if (eai != 0) {
		if (err) {
			*err = translateEAI(eai);
		}
		return vec;
	}
	
	size_t num_addrs = 0;
	for (; ai; ai = ai->ai_next) {
		++num_addrs;
	}
	vec.reserve(num_addrs);
	
	for (ai = ai_holder.get(); ai; ai = ai->ai_next) {
		InetAddr addr;
		struct sockaddr_in* sa_in = reinterpret_cast<sockaddr_in*>(ai->ai_addr);
		// This actually works also for IPv6.
		if (addr.set(sa_in, ai->ai_addrlen) == 0) {
			addr.set_port_number(port);
			vec.push_back(addr);
		}
	}
	
#else // ACE_HAS_IPV6
	
	struct in_addr ipv4_addr;
	if (ACE_OS::inet_aton(host, &ipv4_addr)) {
		struct sockaddr_in sa_in;
		memset(&sa_in, 0, sizeof(sa_in));
		sa_in.sin_family = AF_INET;
		sa_in.sin_port = htons(port);
		sa_in.sin_addr.s_addr = ipv4_addr.s_addr;
		InetAddr addr;
		addr.set(&sa_in, sizeof(sa_in));
		vec.push_back(addr);
	} else {
		hostent hentry;
		ACE_HOSTENT_DATA buf;
		int h_err = 0;
		hostent* hp = ACE_OS::gethostbyname_r(host, &hentry, buf, &h_err);
		if (!hp) {
			if (err) {
				*err = translateHERR(h_err);
			}
			return vec;
		} else if (hp->h_addrtype != AF_INET) {
			if (err) {
				*err = NO_ADDRESSES;
			}
			return vec;
		}
		
		size_t num_addrs = 0;
		for (char** p = hp->h_addr_list; *p; ++p) {
			++num_addrs;
		}
		vec.reserve(num_addrs);
		
		for (char** p = hp->h_addr_list; *p; ++p) {
			InetAddr addr;
			struct sockaddr_in sa_in;
			memset(&sa_in, 0, sizeof(sa_in));
			sa_in.sin_family = AF_INET;
			sa_in.sin_port = htons(port);
			sa_in.sin_addr = *(struct in_addr*)*p;
			if (addr.set(&sa_in, sizeof(sa_in)) == 0) {
				vec.push_back(addr);
			}
		}
	}
#endif
	
	if (err) {
		if (vec.empty()) {
			*err = NO_ADDRESSES;
		} else {
			*err = SUCCESS;
		}
	}
	return vec;
}

#ifdef ACE_HAS_IPV6
DnsResolver::ErrorCode
DnsResolver::translateEAI(int err)
{
	if (err == 0) {
		return SUCCESS;
	} else if (err == EAI_NONAME) {
		return NO_ADDRESSES;
#if defined(EAI_NODATA)
	} else if (err == EAI_NODATA) {
		// EAI_NODATA is deprecated and is usually defined to EAI_NONAME
		return NO_ADDRESSES;
#endif
	} else if (err == EAI_AGAIN) {
		return TEMPORARY_ERROR;
	} else {
		return MISC_ERROR;
	}
}
#endif

DnsResolver::ErrorCode
DnsResolver::translateHERR(int err)
{
	if (err == 0) {
		return SUCCESS;
	} else if (err == HOST_NOT_FOUND) {
		return NO_ADDRESSES;
	} else if (err == NO_DATA) {
		return NO_ADDRESSES;
	} else if (err == TRY_AGAIN) {
		return TEMPORARY_ERROR;
	} else {
		return MISC_ERROR;
	}
}
