/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "DeflateCompressor.h"
#include "SplittableBuffer.h"
#include "DataChunk.h"
#include <memory>
#include <algorithm>
#include <cassert>

using namespace std;

struct DeflateCompressor::OperationContext
{
	std::auto_ptr<DataChunk> chunk;
	size_t chunk_data_size;
	size_t input_consumed;
	size_t output_produced;
	size_t const output_limit;
	
	OperationContext(size_t const limit)
	: chunk(), chunk_data_size(0), input_consumed(0),
	  output_produced(0), output_limit(limit) {}
};


DeflateCompressor::DeflateCompressor(Flavor flavor)
:	m_flavor(flavor),
	m_state(ST_COMPRESSING),
	m_outputLatency(0),
	m_flushParam(Z_NO_FLUSH),
	m_isEOF(false)
{
	initStream();
}

DeflateCompressor::~DeflateCompressor()
{
	deflateEnd(&m_strm);
}

void
DeflateCompressor::reset()
{
	deflateReset(&m_strm);
	m_state = ST_COMPRESSING;
	m_bufferedInput.clear();
	m_outputLatency = 0;
	m_flushParam = Z_NO_FLUSH;
	m_isEOF = false;
}

bool
DeflateCompressor::isError() const
{
	return (m_state == ST_ERROR);
}

void
DeflateCompressor::consume(SplittableBuffer& data, bool eof)
{
	if (m_state == ST_ERROR) {
		data.clear();
		return;
	}
	if (m_isEOF) {
		assert(data.empty() && eof);
		return;
	}
	m_bufferedInput.appendDestructive(data);
	m_isEOF = eof;
}

size_t
DeflateCompressor::retrieveOutput(SplittableBuffer& buf, size_t limit)
{
	OperationContext ctx(limit);
	SplittableBuffer::SpanIterator iter = m_bufferedInput.spanBegin();
	SplittableBuffer::SpanIterator const iter_end = m_bufferedInput.spanEnd();
	bool const empty_input = (iter == iter_end);
	for (; m_state == ST_COMPRESSING && iter != iter_end; ++iter) {
		uint8_t const* begin = (uint8_t const*)iter->begin();
		uint8_t const* const end = (uint8_t const*)iter->end();
		
		if (m_isEOF && iter == m_bufferedInput.spanRBegin()) {
			m_flushParam = Z_FINISH;
		}
		
		compress(begin, end, buf, ctx);
		if (ctx.output_produced == limit) {
			break;
		}
		if (begin != end) {
			break;
		}
	}
	
	if (m_state == ST_COMPRESSING && m_isEOF && empty_input) {
		m_flushParam = Z_FINISH;
		uint8_t const tmp = '\0';
		uint8_t const* zptr = &tmp;
		compress(zptr, zptr, buf, ctx);
	}
	
	if (ctx.chunk_data_size > 0) {
		buf.append(DataChunk::resize(ctx.chunk, ctx.chunk_data_size));
	}
	
	m_bufferedInput.trimFront(ctx.input_consumed);
	
	return ctx.output_produced;
}

void
DeflateCompressor::initStream()
{
	m_strm.zalloc = Z_NULL;
	m_strm.zfree = Z_NULL;
	m_strm.opaque = Z_NULL;
	m_strm.next_in = Z_NULL;
	if (m_flavor == ZLIB) {
		deflateInit(&m_strm, Z_DEFAULT_COMPRESSION);
	} else { // m_flavor == RAW
		deflateInit2(
			&m_strm, Z_DEFAULT_COMPRESSION,
			Z_DEFLATED, -MAX_WBITS, 8, Z_DEFAULT_STRATEGY
		);
	}
}

void
DeflateCompressor::compress(
	uint8_t const*& begin, uint8_t const* end,
	SplittableBuffer& out, OperationContext& ctx)
{
	while (true) {
		if (!ctx.chunk.get()) {
			size_t avail = ctx.output_limit - ctx.output_produced;
			if (avail == 0) {
				return;
			}
			ctx.chunk = DataChunk::create(
				std::min<size_t>(MAX_OUTBUF_SIZE, avail)
			);
		}
		m_strm.next_in = (Bytef*)begin;
		m_strm.avail_in = end - begin;
		uint8_t* const out_begin = (uint8_t*)ctx.chunk->getDataAddr()
			+ ctx.chunk_data_size;
		m_strm.next_out = out_begin;
		m_strm.avail_out = ctx.chunk->getDataSize() - ctx.chunk_data_size;
		int res = deflate(&m_strm, m_flushParam);
		size_t consumed = m_strm.next_in - begin;
		ctx.input_consumed += consumed;
		m_outputLatency += consumed;
		if (m_outputLatency > MAX_OUTPUT_LATENCY && m_flushParam == Z_NO_FLUSH) {
			m_flushParam = Z_SYNC_FLUSH;
		}
		begin = m_strm.next_in;
		if (out_begin != m_strm.next_out) {
			m_outputLatency = 0;
			if (m_flushParam == Z_SYNC_FLUSH) {
				m_flushParam = Z_NO_FLUSH;
			}
			size_t produced = m_strm.next_out - out_begin;
			ctx.chunk_data_size += produced;
			ctx.output_produced += produced;
			if (ctx.chunk_data_size == ctx.chunk->getDataSize()) {
				out.append(ctx.chunk);
				ctx.chunk_data_size = 0;
			}
		}
		
		if (res == Z_OK && m_strm.avail_out == 0) {
			continue;
		}
		if (res == Z_STREAM_END) {
			m_state = ST_FINISHED;
		} else if (res != Z_OK && res != Z_BUF_ERROR) {
			m_state = ST_ERROR;
		}
		break;
	} 
}
