/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef INTRUSIVEPTR_H_
#define INTRUSIVEPTR_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

template<typename T>
class IntrusivePtr
{
public:
	IntrusivePtr() : m_pObj(0) {}
	
	explicit
	IntrusivePtr(T* obj);
	
	IntrusivePtr(IntrusivePtr const& other);
	
	template<typename OT>
	IntrusivePtr(IntrusivePtr<OT> const& other);
	
	~IntrusivePtr();
	
	IntrusivePtr& operator=(IntrusivePtr const& rhs);
	
	template<typename OT>
	IntrusivePtr& operator=(IntrusivePtr<OT> const& rhs);
	
	T& operator*() const { return *m_pObj; }
	
	T* operator->() const { return m_pObj; }
	
	operator T*() const { return m_pObj; }
	
	T* get() const { return m_pObj; }
	
	void reset(T* obj);
	
	void swap(IntrusivePtr& other);
private:
	T* m_pObj;
};


/**
 * \brief Default implementation of intrusive referencing.
 *
 * May be specialized or overloaded.
 */
template<typename T>
inline void intrusive_ref(T& obj)
{
	obj.ref();
}


/**
 * \brief Default implementation of intrusive unreferencing.
 *
 * May be specialized or overloaded.
 */
template<typename T>
inline void intrusive_unref(T& obj)
{
	obj.unref();
}


template<typename T>
inline
IntrusivePtr<T>::IntrusivePtr(T* obj)
:	m_pObj(obj)
{
	if (obj)
		intrusive_ref(*obj);
}

template<typename T>
inline
IntrusivePtr<T>::IntrusivePtr(IntrusivePtr const& other)
:	m_pObj(other.m_pObj)
{
	if (m_pObj)
		intrusive_ref(*m_pObj);
}

template<typename T>
template<typename OT>
inline
IntrusivePtr<T>::IntrusivePtr(IntrusivePtr<OT> const& other)
:	m_pObj(other.get())
{
	if (m_pObj)
		intrusive_ref(*m_pObj);
}

template<typename T>
inline
IntrusivePtr<T>::~IntrusivePtr()
{
	if (m_pObj)
		intrusive_unref(*m_pObj);
}

template<typename T>
inline IntrusivePtr<T>&
IntrusivePtr<T>::operator=(IntrusivePtr const& rhs)
{
	IntrusivePtr(rhs).swap(*this);
	return *this;
}

template<typename T>
template<typename OT>
inline IntrusivePtr<T>&
IntrusivePtr<T>::operator=(IntrusivePtr<OT> const& rhs)
{
	IntrusivePtr(rhs).swap(*this);
	return *this;
}

template<typename T>
inline void
IntrusivePtr<T>::reset(T* obj)
{
	IntrusivePtr(obj).swap(*this);
}

template<typename T>
inline void
IntrusivePtr<T>::swap(IntrusivePtr& other)
{
	T* obj = other.m_pObj;
	other.m_pObj = m_pObj;
	m_pObj = obj;
}

template<typename T>
inline void swap(IntrusivePtr<T>& o1, IntrusivePtr<T>& o2)
{
	o1.swap(o2);
}

#endif
