/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "NetworkPrefs.h"
#include "CFUtils.h"
#include "StringLess.h"
#include "OutputFunction.h"
#include <CoreFoundation/CoreFoundation.h>
#include <Security/Authorization.h>
#include <Security/AuthorizationTags.h>
#include <mach-o/dyld.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>
#include <dirent.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <errno.h>
#include <string>
#include <utility>
#include <map>

using namespace std;

static OSStatus runAsRoot(char const* path, char* const* args);
static OSStatus performUninstall();
static bool checkStartupItem();
static void stopService();
static bool recursiveDelete(char const* path);
static void removeNetworkProfiles();


int main(int argc, char** argv)
{
	chdir("/");
	
	if (argc == 2 && strcmp(argv[1], "root") == 0) {
		if (geteuid() != 0) {
			printf("Failed to get root permissions\n");
			return 1;
		}
		setuid(0); // so that both uid and euid are 0
		return performUninstall();
	} else {
		char path[PATH_MAX];
		uint32_t path_len = sizeof(path);
		if (_NSGetExecutablePath(path, &path_len) != 0) {
			printf("_NSGetExecutablePath() failed\n");
		}
		char *args[] = { "root", NULL };
		return runAsRoot(path, args);
	}
}


OSStatus runAsRoot(char const* path, char* const* args)
{
	AuthorizationRef auth_ref;
	OSStatus status = AuthorizationCreate(
		NULL, kAuthorizationEmptyEnvironment,
		kAuthorizationFlagDefaults, &auth_ref
	);
	if (status != errAuthorizationSuccess) {
		return status;
	}
 
	do {
		{
			AuthorizationItem items = {kAuthorizationRightExecute, 0, NULL, 0};
			AuthorizationRights rights = {1, &items};
			
			AuthorizationFlags flags = kAuthorizationFlagDefaults |
				kAuthorizationFlagInteractionAllowed |
				kAuthorizationFlagPreAuthorize |
				kAuthorizationFlagExtendRights;
			status = AuthorizationCopyRights(
				auth_ref, &rights, NULL, flags, NULL
			);
		}
		
		if (status != errAuthorizationSuccess) {
			break;
		}
			
		{
			FILE *comm_file = NULL;
			char buf[128];
			
			status = AuthorizationExecuteWithPrivileges(
				auth_ref, path, kAuthorizationFlagDefaults, args, &comm_file
			);
			
			if (status != errAuthorizationSuccess) {
				break;
			}
			
			for(;;) {
				ssize_t len = read(fileno(comm_file), buf, sizeof(buf));
				if (len <= 0) {
					break;
				}
					
				fwrite(buf, 1, len, stdout);
			}
			
			fclose(comm_file);
		}
	} while (false);
 
	AuthorizationFree(auth_ref, kAuthorizationFlagDefaults);
	
	return status;
}


OSStatus performUninstall()
{
	printf("Checking startup record ... ");
	fflush(stdout);
	if (!checkStartupItem()) {
		printf("missing or corrupted\n");
	} else {
		printf("OK\n");
		printf("Stopping background service\n");
		fflush(stdout);
		stopService();
	}
	
	printf("Deleting files\n");
	recursiveDelete("/Library/StartupItems/BFilter");
	recursiveDelete("/Library/Application Support/BFilter");
	recursiveDelete("/Library/Receipts/BFilter.pkg");
	recursiveDelete("/var/run/BFilterDaemon.pid");
	
	printf("Removing BFilter network profiles\n");
	removeNetworkProfiles();
	
	printf("Done\n");
	printf("IMPORTANT: Press Command+Q to quit the Terminal application.\n");
	printf("Until you do so, you won't be able to unmount the disk image.\n");
	
	return 0;
}


bool checkStartupItem()
{
	struct stat st;
	
	if (stat("/Library/StartupItems/BFilter", &st) == -1) {
		return false;
	}
	if (!(st.st_mode & S_IFDIR)) {
		// not directory
		return false;
	}
	if (st.st_uid != 0) {
		// owner is not root
		return false;
	}
	if (st.st_gid != 0 && (st.st_mode & S_IWGRP)) {
		// group is not wheel and has write permission
		return false;
	}
	if (st.st_mode & S_IWOTH) {
		// other has write permission
		return false;
	}
	
	if (stat("/Library/StartupItems/BFilter/BFilterDaemon", &st) == -1) {
		return false;
	}
	if (!(st.st_mode & S_IFREG)) {
		// not a regular file
		return false;
	}
	if (st.st_uid != 0) {
		// owner is not root
		return false;
	}
	if (st.st_gid != 0 && (st.st_mode & S_IWGRP)) {
		// group is not wheel and has write permission
		return false;
	}
	if (st.st_mode & S_IWOTH) {
		// other has write permission
		return false;
	}
	
	return true;
}


void stopService()
{
	pid_t pid = fork();
	if (pid < 0) {
		// error
		return;
	} else if (pid > 0) {
		// parent process
		int status = 0;
		while (waitpid(pid, &status, 0) == -1 && errno == EINTR) {}
		return;
	}
	
	// child process
	char const* daemon_path = "/Library/StartupItems/BFilter/BFilterDaemon";
	char const* daemon_pid = "/var/run/BFilterDaemon.pid";
	execl(daemon_path, daemon_path, "-k", "-p", daemon_pid);
}

bool recursiveDelete(char const* path)
{
	struct stat st;
	if (lstat(path, &st) == -1) {
		return (errno == ENOTDIR || errno == ENOENT);
	}
	
	if (!(st.st_mode & S_IFDIR)) {
		// not directory
		if (unlink(path) == -1) {
			return (errno == ENOENT);
		}
		return true;
	}
	
	// directory
	std::string child_path(path);
	child_path += '/';
	size_t base_len = child_path.size();
	
	DIR* dir = opendir(path);
	if (!dir) {
		return false;
	}
	
	bool ok = true;
	struct dirent* ent = 0;
	while (ok && (ent = readdir(dir))) {
		if (!strcmp(ent->d_name, ".") || !strcmp(ent->d_name, "..")) {
			continue;
		}
		child_path += ent->d_name;
		ok = recursiveDelete(child_path.c_str());
		child_path.resize(base_len);
	}
	
	closedir(dir);
	
	if (!ok) {
		return false;
	}
	
	if (rmdir(path) == -1) {
		return (errno == ENOENT);
	}
	
	return true;
}


void removeNetworkProfiles()
try {
	StringRef const bf_prefix(CFSTR("(BF) "), RETAIN);
	
	NetworkPrefs prefs(CFSTR("BFilter Uninstaller"));
	if (!prefs.lock()) {
		printf("Unable to lock network preferences\n");
		return;
	}
	
	StringRef cur_set_id = prefs.getCurrentSetId();
	StringRef cur_set_name = prefs.getSetName(cur_set_id);
	
	typedef multimap<StringRef, StringRef, StringLess> Sets;
	typedef pair<StringRef, StringRef> StringPair;
	Sets sets; // name => id
	
	DECLARE_INSERTER(
		SetInserter, StringPair, Sets,
		c.insert(std::make_pair(val.second, val.first))
	);
	SetInserter set_inserter(sets);
	prefs.enumSets(set_inserter);
	if (sets.size() <= 1) {
		// we are not going to remove the only location
		return;
	}
	
	StringRef new_set_name = CFUtils::removeStringPrefix(cur_set_name, bf_prefix);
	if (new_set_name != cur_set_name) { // (BF) location
		Sets::iterator it = sets.find(new_set_name);
		if (it != sets.end()) {
			prefs.setCurrentSetId(it->second);
		}
	}
	
	Sets::iterator it = sets.lower_bound(bf_prefix);
	Sets::iterator const end = sets.end();
	for (; it != end; ++it) {
		if (!CFStringHasPrefix(it->first, bf_prefix)) {
			break;
		}
		prefs.deleteSet(it->second);
	}
	
	prefs.deleteUnreferencedServices();
	
	if (!prefs.applyChanges()) {
		printf("Could not apply changes\n");
		return;
	}
} catch (std::exception& e) {
	printf("Error: %s\n", e.what());
	return;
}
