/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Application.h"
#include "OperationLog.h"
#include "TrayIcon.h"
#include "TrayMenu.h"
#include "AboutDialog.h"
#include "LogDialog.h"
#include "BasicConfigDialog.h"
#include "AdvancedConfigWindow.h"
#include "FilterConfigWindow.h"
#include "RequestLogWindow.h"
#include "Conf.h"
#include "ConfigErrorHandler.h"
#include "RegexFilterDescriptor.h"
#include "GlobalState.h"
#include "UrlPatterns.h"
#include "FilterGroupTag.h"
#include "InetAddr.h"
#include "SymbolicInetAddr.h"
#include "DnsResolver.h"
#include "IntrusivePtr.h"
#include "RefCountableSAP.h"
#include "AutoClosingSAP.h"
#include "WorkerThreadPool.h"
#include "ScopedIncDec.h"
#include "EffectiveFileTimestamps.h"
#ifdef DEBUG
#include "DebugWindow.h"
#endif
#include "binreloc/prefix.h"
#include <ace/config-lite.h>
#include <ace/Dirent.h>
#include <ace/FILE_Addr.h>
#include <ace/FILE_IO.h>
#include <ace/FILE_Connector.h>
#include <ace/SOCK_Acceptor.h>
#include <ace/OS_NS_sys_stat.h>
#include <ace/OS_NS_stdio.h>
#include <ace/OS_NS_unistd.h>
#include <boost/tokenizer.hpp>
#include <glibmm/miscutils.h>
#include <glibmm/fileutils.h>
#include <glibmm/convert.h>
#include <gdk/gdk.h> // for gdk_notify_startup_complete()
#include <sstream>
#include <fstream>
#include <string>
#include <list>
#include <deque>
#include <vector>
#include <set>
#include <stddef.h>
#include <time.h>

using namespace std;

struct Application::ContentFilterFnameComparator
{
	bool operator()(
		IntrusivePtr<ContentFilterGroup> const& lhs,
		IntrusivePtr<ContentFilterGroup> const& rhs) {
		return lhs->fileName() < rhs->fileName();
	}
};


Application* Application::m_spInstance = 0;

Application::Application(int argc, char* argv[])
:	m_isExiting(false),
	m_commandNestLevel(0),
	m_gtkMain(argc, argv),
	m_commandQueue(COMMAND_QUEUE_CAPACITY, *this),
	m_networkActivityHandler(m_commandQueue),
	m_filterJsLogHandler(m_commandQueue),
	m_requestLogHandler(m_commandQueue),
#ifdef DEBUG
	m_debugAgent(m_commandQueue),
#endif
	m_isAcceptingConnections(false)
{
	m_dispatcher.connect(
		sigc::mem_fun(*this, &Application::processCommands)
	);
	m_spInstance = this;
	m_userConfDir = Glib::build_filename(Glib::get_home_dir(), ".bfilter");
	m_globalConfDir = string(SYSCONFDIR) + "/bfilter";
	m_ptrTrayIcon.reset(new TrayIcon);
	m_ptrTrayMenu.reset(new TrayMenu);
#ifdef DEBUG
	m_ptrDebugWindow.reset(new DebugWindow);
#endif
	m_ptrWorkerPool.reset(new WorkerThreadPool);
}

Application::~Application()
{
	m_destroySignal.emit();
}

void
Application::run()
{
	bool errors = false;
	errors |= !createMissingUserConfigs();
	errors |= !loadConfig();
	loadStandardUrlPatterns();
	loadLocalUrlPatterns();
	loadContentFilters();
	if (errors) {
		showLogDialog();
	}
	
	m_ptrWorkerPool->start();
	m_gtkMain.run();
	
	// This will finish the worker threads.
	m_ptrWorkerPool.reset(0);
}

void
Application::entryPoint(int argc, char* argv[])
{
	//Glib::thread_init();
	// we don't really need it since we've moved from
	// Glib::signal_idle to Glib::Dispatcher.
	
	Application app(argc, argv);
	gdk_notify_startup_complete();
	app.run();
}

void
Application::showTrayMenu(guint button, guint32 activate_time)
{
	m_ptrTrayMenu->popup(button, activate_time);
}

void
Application::showAboutDialog()
{
	AboutDialog::showWindow();
}

void
Application::showLogDialog()
{
	LogDialog::showWindow();
}

void
Application::showBasicConfigDialog()
{
	BasicConfigDialog::showWindow();
}

void
Application::showAdvancedConfigWindow()
{
	AdvancedConfigWindow::showWindow();
}

void
Application::showFilterConfigWindow()
{
	FilterConfigWindow::showWindow();
}

void
Application::showRequestLogWindow()
{
	RequestLogWindow::showWindow();
}

void
Application::setFilteringEnabled(bool val)
{
	GlobalState::WriteAccessor()->setFilteringEnabled(val);
}

#ifdef DEBUG
void
Application::showDebugWindow()
{
	m_ptrDebugWindow->present();
}
#endif

void
Application::requestAppExit()
{
	// If we just call m_gtkMain.quit(), we may get a deadlock
	// while waiting for worker threads to finish
	// (in AcceptorThread DTOR). This happens if any worker thread
	// is blocked while sending a command through m_commandQueue.
	if (m_commandQueue.close() == 0) {
		// command queue is empty
		m_gtkMain.quit();
	} else {
		// Wait for the pending commands to be processed.
		// The actual quit will be done in notifyQueueEmpty().
		m_isExiting = true;
	}
}

Glib::ustring
Application::localeToUtf8(std::string const& text)
{
	try {
		return Glib::locale_to_utf8(text);
	} catch (Glib::ConvertError& e) {
		string tmp(text);
		string::iterator it = tmp.begin();
		string::iterator const end = tmp.end();
		for (; it != end; ++it) {
			if ((unsigned char)*it > 127) {
				*it = '?';
			}
		}
		return tmp;
	}
}

std::string
Application::localeFromUtf8(Glib::ustring const& text)
{
	try {
		return Glib::locale_from_utf8(text);
	} catch (Glib::ConvertError& e) {
		string charset;
		Glib::get_charset(charset);
		return Glib::convert_with_fallback(text, charset, "utf8");
	}
}

Glib::ustring
Application::filenameToUtf8(std::string const& fname)
{
	// glib 2.6 has g_filename_display_name() that does what we want,
	// but we don't want to depend on glib 2.6.
	try {
		return Glib::filename_to_utf8(fname);
	} catch (Glib::ConvertError& e) {
		string tmp(fname);
		string::iterator it = tmp.begin();
		string::iterator const end = tmp.end();
		for (; it != end; ++it) {
			if ((unsigned char)*it > 127) {
				*it = '?';
			}
		}
		return tmp;
	}
}

bool
Application::fileToStream(ACE_FILE_IO& file, std::ostream& strm)
{
	char buf[4096];
	ssize_t read = 0;
	while ((read = file.recv(buf, sizeof(buf))) > 0) {
		strm.write(buf, read);
	}
	return (read == 0);
}

bool
Application::readFile(
	std::string const& abs_fname, std::string& target,
	time_t* mtime, bool log_errors)
{
	AutoClosingSAP<ACE_FILE_IO> file;
	ACE_FILE_Addr addr(abs_fname.c_str());
	ACE_FILE_Connector connector;
	
	if (connector.connect(file, addr, 0, ACE_Addr::sap_any, 0, O_RDONLY) == -1) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				"Could not open "+filenameToUtf8(abs_fname),
				log->getErrorStyle()
			);
		}
		return false;
	}
	
	ostringstream strm;
	
	if (!fileToStream(file, strm)) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				"Error reading "+filenameToUtf8(abs_fname),
				log->getErrorStyle()
			);
		}
		return false;
	}
	
	target = strm.str();
	
	if (mtime) {
		struct stat st;
		ACE_OS::fstat(file.get_handle(), &st);
		*mtime = st.st_mtime;
	}
	
	return true;
}

bool
Application::writeFile(
	std::string const& abs_fname, std::string const& data,
	time_t* mtime, bool log_errors)
{
	AutoClosingSAP<ACE_FILE_IO> file;
	ACE_FILE_Addr addr(abs_fname.c_str());
	ACE_FILE_Connector connector;
	
	int const flags = O_WRONLY|O_CREAT|O_TRUNC;
	if (connector.connect(file, addr, 0, ACE_Addr::sap_any, 0, flags) == -1) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				"Could not open "+filenameToUtf8(abs_fname),
				log->getErrorStyle()
			);
		}
		return false;
	}
	
	if (file.send_n(data.c_str(), data.size()) == -1) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				"Error writing "+filenameToUtf8(abs_fname),
				log->getErrorStyle()
			);
		}
		return false;
	}
	
	if (mtime) {
		struct stat st;
		ACE_OS::fstat(file.get_handle(), &st);
		*mtime = st.st_mtime;
	}
	
	return true;
}

bool
Application::renameFile(
	std::string const& from, std::string const& to, bool log_errors)
{
	if (ACE_OS::rename(from.c_str(), to.c_str()) == -1) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				"Error renaming "+filenameToUtf8(from)+
				" to "+filenameToUtf8(to),
				log->getErrorStyle()
			);
		}
		return false;
	}
	return true;
}

bool
Application::deleteFile(std::string const& fname, bool log_errors)
{
	if (ACE_OS::unlink(fname.c_str()) == -1 && errno != ENOENT) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				"Error deleting "+filenameToUtf8(fname),
				log->getErrorStyle()
			);
		}
		return false;
	}
	return true;
}

bool
Application::processConfig(
	string const& text, Config& target, ConfigFile& file_structure)
{
	ConfigErrorHandler eh("config");
	file_structure.load(text, target, eh);
	return eh.numErrors() == 0;
}

bool
Application::applyConfig(Config const& config)
{
	typedef RefCountableSAP<ACE_SOCK_Acceptor> Acceptor;
	typedef IntrusivePtr<Acceptor> AcceptorPtr;
	
	Log* log = OperationLog::instance();

	GlobalState::WriteAccessor()->config() = config;
	// We apply the config independently of the result of the
	// bind operation, to make the Basic Config dialog work as expected.
	
	m_ptrWorkerPool->removeAllAcceptors();
	setAcceptingConnections(false);
	
	bool ok = true;
	deque<AcceptorPtr> acceptors;
	list<SymbolicInetAddr> addrs = config.getListenAddresses();
	for (; !addrs.empty(); addrs.pop_front()) {
		SymbolicInetAddr const& addr = addrs.front();
		vector<InetAddr> resolved_addrs = DnsResolver::resolve(addr);
		if (resolved_addrs.empty()) {
			ostringstream err;
			err << "Could not resolve listen address \"" << addr << '"';
			log->appendRecord(err.str().c_str(), log->getErrorStyle());
			ok = false;
			continue;
		}
		AcceptorPtr acceptor(new Acceptor);
		if (acceptor->open(resolved_addrs[0], true) == -1) {
			ostringstream err;
			err << "Could not bind to \"" << addr << '"';
			log->appendRecord(err.str().c_str(), log->getErrorStyle());
			ok = false;
			continue;
		}
		acceptors.push_back(acceptor);
	}
	if (!ok) {
		return false;
	}
	if (acceptors.empty()) {
		log->appendRecord(
			"No addresses to listen on!",
			log->getErrorStyle()
		);
		return false;
	}
	
	for (; !acceptors.empty(); acceptors.pop_front()) {
		m_ptrWorkerPool->addAcceptor(acceptors.front());
	}
	setAcceptingConnections(true);
	
	return true;
}

void
Application::applyBrokenConfig(Config const& broken_config)
{
	GlobalState::WriteAccessor()->config() = broken_config;
	// We apply it anyway, to make the Basic Config dialog work as expected.
	
	m_ptrWorkerPool->removeAllAcceptors();
	setAcceptingConnections(false);
}

bool
Application::loadConfig()
{
	Log* log = OperationLog::instance();
	log->appendRecord("Loading config ... ");
	size_t num_records = log->getNumRecords();
	
	string const abs_fname = Glib::build_filename(m_userConfDir, "config");
	string text;
	time_t mtime = 0;
	if (!readFile(abs_fname, text, &mtime)) {
		return false;
	}
	EffectiveFileTimestamps::config = mtime;
	
	Config config;
	if (!processConfig(text, config, m_configFile)) {
		applyBrokenConfig(config);
		return false;
	}
	
	if (!applyConfig(config)) {
		return false;
	}
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord("done", log->getSuccessStyle());
	}
	
	return true;
}

bool
Application::processUrlPatterns(
	std::string const& text, std::string const& fname,
	UrlPatterns& target, UrlPatternsFile& file_structure)
{
	ConfigErrorHandler eh(filenameToUtf8(fname));
	file_structure.load(text, target, eh);
	return eh.numErrors() == 0;
}

void
Application::applyStandardUrlPatterns(UrlPatterns const& patterns)
{
	GlobalState::WriteAccessor()->urlPatterns().standardPatterns() = patterns;
}

void
Application::applyLocalUrlPatterns(UrlPatterns const& patterns)
{
	GlobalState::WriteAccessor()->urlPatterns().localPatterns() = patterns;
}

void
Application::loadStandardUrlPatterns()
{
	Log* log = OperationLog::instance();
	log->appendRecord("Loading standard url patterns ... ");
	size_t num_records = log->getNumRecords();

	string const fname("urls");
	string const abs_fname = Glib::build_filename(m_globalConfDir, fname);
	string text;
	readFile(abs_fname, text);
	
	UrlPatterns patterns;
	processUrlPatterns(text, fname, patterns, m_standardUrlPatternsFile);
	applyStandardUrlPatterns(patterns);
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord("done", log->getSuccessStyle());
	}
}

void
Application::loadLocalUrlPatterns()
{
	Log* log = OperationLog::instance();
	log->appendRecord("Loading local url patterns ... ");
	size_t num_records = log->getNumRecords();

	string const fname("urls.local");
	string const abs_fname = Glib::build_filename(m_userConfDir, fname);
	string text;
	time_t mtime = 0;
	readFile(abs_fname, text, &mtime);
	
	UrlPatterns patterns;
	processUrlPatterns(text, fname, patterns, m_localUrlPatternsFile);
	applyLocalUrlPatterns(patterns);
	
	EffectiveFileTimestamps::urls_local = mtime;
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord("done", log->getSuccessStyle());
	}
}

void
Application::loadContentFilters()
{
	Log* log = OperationLog::instance();
	log->appendRecord("Loading content filters ... ");
	size_t num_records = log->getNumRecords();
	
	string global_dir = getGlobalFiltersDir();
	string user_dir = getUserFiltersDir();
	deque<string> global_fnames;
	deque<string> user_fnames;
	
	loadFilterFnames(global_dir, global_fnames);
	loadFilterFnames(user_dir, user_fnames);
	
	std::sort(global_fnames.begin(), global_fnames.end());
	std::sort(user_fnames.begin(), user_fnames.end());
	
	m_contentFilters.clear();
	
	while (true) {
		ContentFilterGroup::FileLocation location;
		string dirname;
		string fname;
		string const* global_fname = global_fnames.empty() ? 0 : &global_fnames.front();
		string const* user_fname = user_fnames.empty() ? 0 : &user_fnames.front();
		if (user_fname && !(global_fname && *global_fname < *user_fname)) {
			location = ContentFilterGroup::USER_DIR;
			dirname = user_dir;
			fname = *user_fname;
			user_fnames.pop_front();
			if (global_fname && *global_fname == fname) {
				global_fnames.pop_front();
			}
		} else if (global_fname) {
			location = ContentFilterGroup::GLOBAL_DIR;
			dirname = global_dir;
			fname = global_fnames.front();
			global_fnames.pop_front();
		} else {
			break;
		}
		
		string text;
		string abs_fname = Glib::build_filename(dirname, fname);
		if (!readFile(abs_fname, text)) {
			continue;
		}
		
		IntrusivePtr<ContentFilterGroup> group(
			new ContentFilterGroup(
				FilterGroupTag(), fname, location
			)
		);
		
		ConfigErrorHandler eh(filenameToUtf8(fname));
		group->fileStructure().load(
			text, group->filters(), eh, group->getTag()
		);
		if (eh.numErrors() == 0) {
			loadEnabledFilterList(*group, user_dir);
		}
		
		m_contentFilters.push_back(group);
	}
	
	updateGlobalContentFilters();
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord("done", log->getSuccessStyle());
	}
}

bool
Application::loadFilterFnames(
	std::string const& dirname, std::deque<std::string>& fnames)
{
	Log* log = OperationLog::instance();
	
	ACE_Dirent dir;
	if (dir.open(dirname.c_str()) == -1) {
		log->appendRecord(
			"Could not open directory " + dirname,
			log->getErrorStyle()
		);
		return false;
	}
	
	for (dirent* ent; (ent = dir.read()); ) {
		string const fname(ent->d_name);
		if (fname.find('.') != string::npos) {
			// skip files with extensions
			continue;
		}
		fnames.push_back(fname);
	}
	
	return true;
}

void
Application::loadEnabledFilterList(
	ContentFilterGroup& group, std::string const& dirname)
{
	string fname = Glib::build_filename(dirname, group.fileName() + ".enabled");
	if (!Glib::file_test(fname, Glib::FILE_TEST_EXISTS)) {
		return;
	}
	
	string data;
	if (!readFile(fname, data)) {
		return;
	}
	
	typedef boost::tokenizer<boost::char_separator<char> > Tokenizer;
	boost::char_separator<char> sep("\r\n");
	Tokenizer tokens(data, sep);
	set<string> enabled_set(tokens.begin(), tokens.end());
	bool const all_enabled = (enabled_set.find("*") != enabled_set.end());
	
	typedef ContentFilters::FilterList FilterList;
	FilterList& list = group.filters().filters();
	FilterList::iterator it = list.begin();
	FilterList::iterator const end = list.end();
	for (; it != end; ++it) {
		RegexFilterDescriptor& filter = **it;
		bool const enabled = all_enabled ||
			(enabled_set.find(filter.name()) != enabled_set.end());
		filter.setEnabled(enabled);
	}
}

void
Application::appendContentFilterGroup(
	IntrusivePtr<ContentFilterGroup> const& group)
{
	m_contentFilters.push_back(group);
}

bool
Application::removeContentFilterGroup(
	IntrusivePtr<ContentFilterGroup> const& group)
{
	ContentFilterList::iterator it = std::find(
		m_contentFilters.begin(), m_contentFilters.end(), group
	);
	if (it == m_contentFilters.end()) {
		return false;
	}
	m_contentFilters.erase(it);
	return true;
}

void
Application::sortContentFiltersByFileName()
{
	std::stable_sort(
		m_contentFilters.begin(),
		m_contentFilters.end(),
		ContentFilterFnameComparator()
	);
}

void
Application::updateGlobalContentFilters()
{
	ContentFilters composition;
	
	ContentFilterList::iterator it = m_contentFilters.begin();
	ContentFilterList::iterator const end = m_contentFilters.end();
	for (; it != end; ++it) { 
		typedef ContentFilters::FilterList FilterList;
		FilterList& filters = (*it)->filters().filters();
		FilterList::iterator it2 = filters.begin();
		FilterList::iterator const end2 = filters.end();
		for (; it2 != end2; ++it2) {
			if ((*it2)->isEnabled()) {
				IntrusivePtr<RegexFilterDescriptor> new_filter(
					new RegexFilterDescriptor(**it2)
				);
				// Q: Why do we make a copy?
				// A: Because filters in the global filter list are going
				// to be accessed without locks.
				composition.filters().push_back(new_filter);
			}
		}
	}
	
	composition.sortByOrder();
	GlobalState::WriteAccessor gstate;
	gstate->contentFilters().filters().swap(composition.filters());
}

void
Application::handleNetworkActivity()
{
	m_ptrTrayIcon->displayNetworkActivity();
}

void
Application::notifyCommandsPending()
{
	m_dispatcher.emit();
}

// Note: this function is called from the main thread only.
void
Application::notifyQueueEmpty()
{
	if (m_isExiting) {
		m_gtkMain.quit();
	}
}

void
Application::processCommands()
{
	if (m_commandNestLevel) {
		// If a command does things that invoke a recursive main loop,
		// we can end up here.
		return;
	}
	
	ScopedIncDec<int> nest_level_manager(m_commandNestLevel);
	
	unsigned max_commands = m_commandQueue.getCapacity();
	/*
	We limit the number of iterations to give the GUI thread
	a chance to draw something if other threads keep posting commands.
	*/
	for (; max_commands > 0; --max_commands) {
		InterthreadCommandQueue::CommandPtr command = m_commandQueue.pop();
		if (!command) {
			return;
		}
		(*command)();
	}
	m_dispatcher.emit(); // there could be more commands in the queue
}

bool
Application::createMissingUserConfigs()
{
	std::string config_fname = Glib::build_filename(m_userConfDir, "config");
	std::string urls_local_fname = Glib::build_filename(m_userConfDir, "urls.local");
	std::string filters_dirname = Glib::build_filename(m_userConfDir, "filters");
	bool confdir_exists = Glib::file_test(m_userConfDir, Glib::FILE_TEST_EXISTS);
	bool config_exists = Glib::file_test(config_fname, Glib::FILE_TEST_EXISTS);
	bool urls_local_exists = Glib::file_test(urls_local_fname, Glib::FILE_TEST_EXISTS);
	bool filters_dir_exists = Glib::file_test(filters_dirname, Glib::FILE_TEST_EXISTS);
	if (confdir_exists && config_exists && urls_local_exists && filters_dir_exists) {
		return true;
	}
	
	Log* log = OperationLog::instance();
	log->appendRecord("Creating user configuration ... ");
	size_t num_records = log->getNumRecords();
	bool errors = false;
	
	if (!confdir_exists) {
		errors |= !createDir(m_userConfDir);
	}
	if (!config_exists) {
		string source = Glib::build_filename(m_globalConfDir, "config");
		errors |= !copyFile(source, config_fname);
	}
	if (!urls_local_exists) {
		string source = Glib::build_filename(m_globalConfDir, "urls.local");
		errors |= !copyFile(source, urls_local_fname);
	}
	if (!filters_dir_exists) {
		errors |= !createDir(filters_dirname);
	}
	
	if (!errors && num_records == log->getNumRecords()) {
		log->appendToLastRecord("done", log->getSuccessStyle());
	}
	return !errors;
}

bool
Application::copyFile(std::string const& source, std::string const& target)
{
	Log* log = OperationLog::instance();
	ifstream in(source.c_str());
	if (!in.good()) {
		log->appendRecord(
			"Could not open " + source,
			log->getErrorStyle()
		);
		return false;
	}
	
	ofstream out(target.c_str());
	if (!out.good()) {
		log->appendRecord(
			"Could not create " + target,
			log->getErrorStyle()
		);
		return false;
	}
	
	out << in.rdbuf();
	if (in.bad()) {
		log->appendRecord(
			"Error reading from " + source,
			log->getErrorStyle()
		);
		return false;
	}
	if (out.bad()) {
		log->appendRecord(
			"Error writing to " + target,
			log->getErrorStyle()
		);
		return false;
	}
	
	return true;
}

bool
Application::createDir(std::string const& dirname)
{
	Log* log = OperationLog::instance();
	if (ACE_OS::mkdir(dirname.c_str()) == -1) {
		log->appendRecord(
			"Could not create directory " + dirname,
			log->getErrorStyle()
		);
		return false;
	}
	return true;
}

std::string
Application::getConfigFileName() const
{
	return Glib::build_filename(m_userConfDir, "config");
}

std::string
Application::getConfigDefaultFileName() const
{
	return Glib::build_filename(m_globalConfDir, "config.default");
}

std::string
Application::getStandardUrlPatternsFileName() const
{
	return Glib::build_filename(m_globalConfDir, "urls");
}

std::string
Application::getLocalUrlPatternsFileName() const
{
	return Glib::build_filename(m_userConfDir, "urls.local");
}

std::string
Application::getGlobalFiltersDir() const
{
	return Glib::build_filename(m_globalConfDir, "filters");
}

std::string
Application::getUserFiltersDir() const
{
	return Glib::build_filename(m_userConfDir, "filters");
}

void
Application::setAcceptingConnections(bool val)
{
	if (val != m_isAcceptingConnections) {
		m_isAcceptingConnections = val;
		m_acceptingConnectionsSignal.emit(val);
	}
}
