/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "InterthreadCommandQueue.h"

using namespace std;

class InterthreadCommandQueue::SemaphoreGuard
{
public:
	SemaphoreGuard(ACE_Thread_Semaphore& sem);
	
	~SemaphoreGuard();
	
	void dontRelease() { m_toBeReleased = false; }
private:
	ACE_Thread_Semaphore& m_rSemaphore;
	bool m_toBeReleased;
};


InterthreadCommandQueue::InterthreadCommandQueue(
	unsigned capacity, Notifier& notifier)
:	m_capacity(capacity),
	m_rNotifier(notifier),
	m_capacitySemaphore(capacity),
	m_isClosed(false)
{
}

InterthreadCommandQueue::~InterthreadCommandQueue()
{
}

bool
InterthreadCommandQueue::push(CommandPtr const& cmd)
{
	bool was_empty = false;
	SemaphoreGuard sem_guard(m_capacitySemaphore);
	
	{
		ACE_Guard<ACE_Thread_Mutex> guard(m_queueAccessMutex);
		if (m_isClosed) {
			return false;
		}
		was_empty = m_commandQueue.empty();
		m_commandQueue.push_back(cmd);
		sem_guard.dontRelease();
	}
	
	if (was_empty) {
		m_rNotifier.notifyCommandsPending();
	}
	
	return true;
}

InterthreadCommandQueue::CommandPtr
InterthreadCommandQueue::pop()
{
	CommandPtr command;
	
	{
		ACE_Guard<ACE_Thread_Mutex> guard(m_queueAccessMutex);
		
		if (!m_commandQueue.empty()) {
			command = m_commandQueue.front();
			m_commandQueue.pop_front();
			m_capacitySemaphore.release();
			if (m_commandQueue.empty()) {
				m_rNotifier.notifyQueueEmpty();
			}
		}
	}
	
	return command;
}

size_t
InterthreadCommandQueue::close()
{
	ACE_Guard<ACE_Thread_Mutex> guard(m_queueAccessMutex);
	size_t const num_commands = m_commandQueue.size();
	if (!m_isClosed) {
		m_isClosed = true;
		m_capacitySemaphore.release();
		// If we wake up one blocked thread, we wake up them all.
		// The waked-up thread will see m_isClosed is set and release
		// the semaphore waking another thread.
	}
	return num_commands;
}


/*=============== InterthreadCommandQueue::SemaphoreGuard ================*/

InterthreadCommandQueue::SemaphoreGuard::SemaphoreGuard(
	ACE_Thread_Semaphore& sem)
:	m_rSemaphore(sem),
	m_toBeReleased(true)
{
	m_rSemaphore.acquire();
}

InterthreadCommandQueue::SemaphoreGuard::~SemaphoreGuard()
{
	if (m_toBeReleased) {
		m_rSemaphore.release();
	}
}
