/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "HttpHeaderStructure.h"
#include "HttpHeaderElement.h"
#include "HttpHeader.h"
#include "HeaderParser.h"
#include "HeaderParserListener.h"
#include "InsensitiveEqual.h"
#include "SplittableBuffer.h"
#include "SBOutStream.h"
#include "Debug.h"
#include <ostream>
#include <cstring>
#include <list>

using namespace std;

class HttpHeaderStructure::Loader : public HeaderParserListener
{
public:
	Loader(HttpHeaderStructure& owner, HttpHeader const* header)
	: m_rOwner(owner), m_pHeader(header) {}
	
	virtual void processError(size_t position);
	
	virtual void processElementName(Iterator const& begin, Iterator const& end);
	
	virtual void processElementValue(Iterator const& begin, Iterator const& end);
	
	virtual void processElementNullValue();
	
	virtual void processParamName(Iterator const& begin, Iterator const& end);
	
	virtual void processParamValue(Iterator const& begin, Iterator const& end);
	
	virtual void processParamNullValue();
private:
	HttpHeaderStructure& m_rOwner;
	HttpHeader const* m_pHeader;
	BString m_name;
};


HttpHeaderStructure::HttpHeaderStructure(HttpHeader const& header)
{
	Loader loader(*this, &header);
	SplittableBuffer buf;
	HeaderParser parser;
	list<BString>::const_iterator it = header.getValues().begin();
	list<BString>::const_iterator const end = header.getValues().end();
	for (; it != end; ++it) {
		buf.append(*it);
		parser.consume(buf, &loader);
	}
}

bool
HttpHeaderStructure::hasElement(BString const& name) const
{
	ElementList::const_iterator it = m_elements.begin();
	ElementList::const_iterator const end = m_elements.end();
	InsensitiveEqual comp;
	for (; it != end; ++it) {
		if (comp(it->getName(), name)) {
			return true;
		}
	}
	return false;
}

void
HttpHeaderStructure::removeElementsByName(BString const& name)
{
	ElementList::iterator it = m_elements.begin();
	ElementList::iterator const end = m_elements.end();
	InsensitiveEqual comp;
	while (it != end) {
		if (comp(it->getName(), name)) {
			m_elements.erase(it++);
		} else {
			++it;
		}
	}
}

void
HttpHeaderStructure::toStream(std::ostream& strm) const
{
	ElementList::const_iterator e_it = m_elements.begin();
	ElementList::const_iterator const e_end = m_elements.end();
	InsensitiveEqual comp;
	for (bool first = true; e_it != e_end; ++e_it, first = false) {
		if (!first) {
			strm << ", ";
		}
		HttpHeaderElement const& el = *e_it;
		strm << el.getName();
		if (!el.isNullValue()) {
			strm << '=';
			printValue(strm, el.getValue());
		}
		ParamList::const_iterator p_it = el.parameters().begin();
		ParamList::const_iterator const p_end = el.parameters().end();
		for (; p_it != p_end; ++p_it) {
			HttpHeaderElement::Param const& par = *p_it;
			strm << ';' << par.name;
			if (!par.isNullValue) {
				strm << '=';
				printValue(strm, par.value);
			}
		}
	}
}

void
HttpHeaderStructure::printValue(std::ostream& strm, BString const& str)
{
	if (isQuotingRequired(str)) {
		printQuoted(strm, str);
	} else {
		strm << str;
	}
}

bool
HttpHeaderStructure::isQuotingRequired(BString const& str)
{
	static char const unsafe[] = "\a\b\f\n\r\t\v=,;\" ";
	char const* p = str.begin();
	char const* const end = str.end();
	for (; p != end; ++p) {
		char const* found = static_cast<char const*>(memchr(unsafe, *p, sizeof(unsafe)-1));
		if (found) {
			return true;
		}
	}
	return false;
}

void
HttpHeaderStructure::printQuoted(std::ostream& strm, BString const& str)
{
	static const char in[]  = {'a', 'b', 'f', 'n', 'r', 't', 'v', '\\','\"'}; 
	static const char out[] = {'\a','\b','\f','\n','\r','\t','\v','\\','\"'};
	strm << '\"';
	char const* p = str.begin();
	char const* const end = str.end();
	for (; p != end; ++p) {
		char const* found = static_cast<char const*>(memchr(in, *p, sizeof(in)));
		if (found) {
			strm << '\\' << out[found - in];
		} else {
			strm << *p;
		}
	}
	strm << '\"';
}

void
HttpHeaderStructure::commitChanges(HttpHeader& hdr)
{
	SBOutStream strm(50 /* chunk size */);
	toStream(strm);
	hdr.setValue(strm.data().toBString());
}


/*========================= HttpHeaderStructure::Loader ====================*/

void
HttpHeaderStructure::Loader::processError(size_t position)
{
	DEBUGLOG("Error parsing header: " << m_pHeader->getName()
		<< ": " << m_pHeader->getValue());
}

void
HttpHeaderStructure::Loader::processElementName(
	Iterator const& begin, Iterator const& end)
{
	m_name = SplittableBuffer::toBString(begin, end);
}

void
HttpHeaderStructure::Loader::processElementValue(
	Iterator const& begin, Iterator const& end)
{
	BString value = SplittableBuffer::toBString(begin, end);
	m_rOwner.appendElement(HttpHeaderElement(m_name, value));
}

void
HttpHeaderStructure::Loader::processElementNullValue()
{
	m_rOwner.appendElement(HttpHeaderElement(m_name));
}

void
HttpHeaderStructure::Loader::processParamName(
	Iterator const& begin, Iterator const& end)
{
	m_name = SplittableBuffer::toBString(begin, end);
}

void
HttpHeaderStructure::Loader::processParamValue(
	Iterator const& begin, Iterator const& end)
{
	BString value = SplittableBuffer::toBString(begin, end);
	m_rOwner.elements().back().parameters().push_back(
		HttpHeaderElement::Param(m_name, value)
	);
}

void
HttpHeaderStructure::Loader::processParamNullValue()
{
	m_rOwner.elements().back().parameters().push_back(
		HttpHeaderElement::Param(m_name)
	);
}
