/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "FlashResponseFilter.h"
#include "SplittableBuffer.h"
#include "GlobalState.h"
#include "HttpRequestMetadata.h"
#include "HttpResponseMetadata.h"
#include "HttpRequestLine.h"
#include "HttpHeadersCollection.h"
#include "HeuristicScore.h"
#include "CraftedResponse.h"
#include "ReplacementFlash.h"
#include "DownloadProgress.h"
#include "ErrorDescriptor.h"
#include "ErrorCodes.h"
#include "IntrusivePtr.h"
#include "RequestTag.h"
#include "BString.h"
#include "URI.h"
#include <string>
#include <stddef.h>

using namespace std;

FlashResponseFilter::FlashResponseFilter(ResponseFilterChain& chain, bool ignore_size)
:	ResponseFilterBase(chain),
	m_state(INSPECTING),
	m_ignoreSize(ignore_size)
{
}

FlashResponseFilter::~FlashResponseFilter()
{
}

void
FlashResponseFilter::processMetadata(auto_ptr<HttpResponseMetadata> metadata)
{
	m_ptrMetadata = metadata;
}

void
FlashResponseFilter::processBodyData(SplittableBuffer& data, bool eof)
{
	switch (m_state) {
		case FORWARDING: {
			outputBodyData(data, eof);
			break;
		}
		case SUBSTITUTING: {
			data.clear();
			break;
		}
		case INSPECTING: {
			m_inspector.consumeDataChunk(data, eof);
			switch (m_inspector.getStatus()) {
				case FlashInspector::COMPLETE: {
					if (isAd()) {
						m_state = SUBSTITUTING;
 						outputAdSubstitution();
						considerRequestCancel();
						break;
					}
					// fall through
				}
				case FlashInspector::FAILED: {
					m_state = FORWARDING;
					outputMetadata(m_ptrMetadata);
					outputBodyData(m_inspector.accumulatedData(), eof);
					break;
				}
				default: break;
			}
			break;
		}
	}
}

bool
FlashResponseFilter::isAd()
{
	return m_inspector.isClickable() || (
		!m_ignoreSize &&
		HeuristicScore::isCommonAdSize(
			m_inspector.getWidth(),
			m_inspector.getHeight()
		)
	);
}

void
FlashResponseFilter::outputAdSubstitution()
{
	auto_ptr<CraftedResponse> response = ReplacementFlash::createHttpResponse(
		false, m_inspector.getWidth(), m_inspector.getHeight(),
		getRequest().requestLine().getURI().toString(),
		m_inspector.getFrameRate()
	);
	
	getRequestTag()->flags().set(RequestTag::RESPONSE_CRAFTED);
	response->metadata().setClientRequestId(m_ptrMetadata->getClientRequestId());
	response->metadata().headers().setHeader(
		BString("X-BFilter-Content"), BString("generated")
	);
	outputMetadata(response->retrieveMetadata());
	outputBodyData(response->body(), true);
}

void
FlashResponseFilter::considerRequestCancel()
{
	DownloadProgress const& dp = getFilterChain().getDownloadProgress();
	size_t threshold = GlobalState::ReadAccessor()->config().getSaveTrafficThreshold();
	if (dp.total() - dp.received() > threshold*1024) { // or if dp.total() is undefined
		raiseError(IntrusivePtr<ErrorDescriptor>(
			new ErrorDescriptor(
				ErrorCodes::REQUEST_CANCEL_TO_SAVE_TRAFFIC,
				"request cancelled to save traffic"
			)
		));
	}
}
