/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef ASYNCCONNECTOR_H_
#define ASYNCCONNECTOR_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "NonCopyable.h"
#include "EventHandler.h"
#include "AsyncConnectorListener.h"
#include "Reactor.h"
#include "Observer.h"
#include "ServerReachability.h"
#include <vector>

class SymbolicInetAddr;
class TimeStamp;
class TimeDelta;

class AsyncConnector : private EventHandlerBase
{
	DECLARE_NON_COPYABLE(AsyncConnector)
public:
	typedef AsyncConnectorListener Listener;
	
	AsyncConnector();
	
	virtual ~AsyncConnector();
	
	bool isInProgress() const { return m_isInProgress; }
	
	void initiate(Listener& listener, Reactor& reactor,
		SymbolicInetAddr const& addr, TimeDelta const* timeout = 0);
	
	void abort();
private:
	class IpRecord;
	class IpRecordQualityCompare;
	
	enum { NEXT_ATTEMPT_TIMEOUT = 5000 }; // msec
	enum { PAUSE_BEFORE_RETRY   = 1200 }; // msec
	
	struct Error
	{
		typedef AsyncConnectorListener::FailureCode Code;
		
		Code code;
		int errnum;
		
		Error() { clear(); }
		
		void clear() { code = AsyncConnectorListener::FAIL_OTHER; errnum = -1; }
		
		void set(Code code, int errnum) {
			this->code = code; this->errnum = errnum;
		}
		
		bool isSet() const {
			return !(code == AsyncConnectorListener::FAIL_OTHER && errnum == -1);
		}
	};
	
	virtual void handleRead(ACE_HANDLE handle);
	
	virtual void handleWrite(ACE_HANDLE handle);
	
	virtual void handleExcept(ACE_HANDLE handle);
	
	virtual void handleTimeout(ReactorTimerId const& timer_id);
	
	virtual void ref();
	
	virtual void unref();
	
	void nextConnectionAttempt();
	
	bool initiateConnection(IpRecord& ip_rec);
	
	void abortConnection(IpRecord& ip_rec, Error::Code err_code, int err_num);
	
	IpRecord* findCandidateForConnection(int& potential_candidates);
	
	IpRecord* findIpRecordByHandle(ACE_HANDLE handle);
	
	IpRecord* findIpRecordByTimerId(ReactorTimerId const& timer_id);
	
	void clearIpRecords();
	
	void clearMainTimeout();
	
	void handleCompletion(ACE_HANDLE handle);
	
	void updateReachabilityInfo(IpRecord& ip_rec,
		ServerReachability::ConnectResult res);
	
	void updateReachabilityInfo(IpRecord& ip_rec,
		ServerReachability::ConnectResult res,
		TimeStamp const& timestamp);
	
	void markInProgressConnectionsAsUndefined();
	
	void onConnEstablished(IpRecord& ip_rec);
	
	void handleConnFailure(Listener::FailureCode code, int errnum);
	
	SingleObserverLink<Listener> m_observerLink;
	Error m_lastError;
	bool m_isInProgress;
	std::vector<IpRecord> m_ipRecords;
	Reactor* m_pReactor;
	ReactorTimerId m_timeoutTimerId;
	int m_refCounter;
};

#endif
