/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef TIMESTAMP_H_
#define TIMESTAMP_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "TimeDelta.h"
#include "types.h"
#include <ace/os_include/os_time.h>
#include <ace/Time_Value.h> // older versions of ACE define struct timespec here
#include <limits>

class TimeStamp
{
public:
	TimeStamp() : m_usec(0) {}
	
	static TimeStamp fromSec(int64_t sec);
	
	static TimeStamp fromMsec(int64_t msec);
	
	static TimeStamp fromUsec(int64_t usec);
	
	static TimeStamp fromTimeval(struct timeval const& tv);
	
	static TimeStamp fromTimespec(struct timespec const& ts);
	
	int64_t toSec() const;
	
	int64_t toMsec() const;
	
	int64_t toUsec() const;
	
	struct timeval toTimeval() const;
	
	struct timespec toTimespec() const;
	
	TimeStamp& operator+=(TimeDelta const& delta);
	
	TimeStamp& operator-=(TimeDelta const& delta);
	
	static TimeStamp min();
	
	static TimeStamp max();
	
	static TimeStamp zero();
private:
	TimeStamp(int64_t usec) : m_usec(usec) {}
	
	int64_t m_usec;
};

TimeStamp operator+(TimeStamp const& lhs, TimeDelta const& rhs);

TimeStamp operator+(TimeDelta const& lhs, TimeStamp const& rhs);

TimeStamp operator-(TimeStamp const& lhs, TimeDelta const& rhs);

TimeDelta operator-(TimeStamp const& lhs, TimeStamp const& rhs);

bool operator==(TimeDelta const& lhs, TimeDelta const& rhs);

bool operator!=(TimeDelta const& lhs, TimeDelta const& rhs);

bool operator<(TimeDelta const& lhs, TimeDelta const& rhs);

bool operator>(TimeDelta const& lhs, TimeDelta const& rhs);

bool operator<=(TimeDelta const& lhs, TimeDelta const& rhs);

bool operator>=(TimeDelta const& lhs, TimeDelta const& rhs);



inline TimeStamp
TimeStamp::fromSec(int64_t sec)
{
	return TimeStamp(sec * 1000000L);
}

inline TimeStamp
TimeStamp::fromMsec(int64_t msec)
{
	return TimeStamp(msec * 1000L);
}

inline TimeStamp
TimeStamp::fromUsec(int64_t usec)
{
	return TimeStamp(usec);
}

inline TimeStamp
TimeStamp::fromTimeval(struct timeval const& tv)
{
	int64_t usec = int64_t(tv.tv_sec) * 1000000L;
	usec += tv.tv_usec;
	return TimeStamp(usec);
}

inline TimeStamp
TimeStamp::fromTimespec(struct timespec const& ts)
{
	int64_t usec = int64_t(ts.tv_sec) * 1000000L;
	usec += ts.tv_nsec / 1000L;
	return TimeStamp(usec);
}

inline int64_t
TimeStamp::toSec() const
{
	return m_usec / 1000000L;
}

inline int64_t
TimeStamp::toMsec() const
{
	return m_usec / 1000L;
}

inline int64_t
TimeStamp::toUsec() const
{
	return m_usec;
}

inline struct timeval
TimeStamp::toTimeval() const
{
	struct timeval tv;
	tv.tv_sec = m_usec / 1000000L;
	tv.tv_usec = m_usec % 1000000L;
	return tv;
}

inline struct timespec
TimeStamp::toTimespec() const
{
	struct timespec ts;
	ts.tv_sec = m_usec / 1000000L;
	ts.tv_nsec = (m_usec % 1000000L) * 1000L;
	return ts;
}

inline TimeStamp&
TimeStamp::operator+=(TimeDelta const& delta)
{
	m_usec += delta.toUsec();
	return *this;
}

inline TimeStamp&
TimeStamp::operator-=(TimeDelta const& delta)
{
	m_usec -= delta.toUsec();
	return *this;
}

inline TimeStamp
TimeStamp::min()
{
	return TimeStamp(std::numeric_limits<int64_t>::min());
}

inline TimeStamp
TimeStamp::max()
{
	return TimeStamp(std::numeric_limits<int64_t>::max());
}

inline TimeStamp
TimeStamp::zero()
{
	return TimeStamp();
}

inline TimeStamp operator+(TimeStamp const& lhs, TimeDelta const& rhs)
{
	return TimeStamp::fromUsec(lhs.toUsec() + rhs.toUsec());
}

inline TimeStamp operator+(TimeDelta const& lhs, TimeStamp const& rhs)
{
	return TimeStamp::fromUsec(lhs.toUsec() + rhs.toUsec());
}

inline TimeStamp operator-(TimeStamp const& lhs, TimeDelta const& rhs)
{
	return TimeStamp::fromUsec(lhs.toUsec() - rhs.toUsec());
}

inline TimeDelta operator-(TimeStamp const& lhs, TimeStamp const& rhs)
{
	return TimeDelta::fromUsec(lhs.toUsec() - rhs.toUsec());
}

inline bool operator==(TimeStamp const& lhs, TimeStamp const& rhs)
{
	return lhs.toUsec() == rhs.toUsec();
}

inline bool operator!=(TimeStamp const& lhs, TimeStamp const& rhs)
{
	return lhs.toUsec() != rhs.toUsec();
}

inline bool operator<(TimeStamp const& lhs, TimeStamp const& rhs)
{
	return lhs.toUsec() < rhs.toUsec();
}

inline bool operator>(TimeStamp const& lhs, TimeStamp const& rhs)
{
	return lhs.toUsec() > rhs.toUsec();
}

inline bool operator<=(TimeStamp const& lhs, TimeStamp const& rhs)
{
	return lhs.toUsec() <= rhs.toUsec();
}

inline bool operator>=(TimeStamp const& lhs, TimeStamp const& rhs)
{
	return lhs.toUsec() >= rhs.toUsec();
}

#endif
