// resource.cc for bbppp - a ppp connection manager for Blackbox
//
//  Copyright (c) 1998-2000 John Kennis, jkennis@chello.nl
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// (See the included file COPYING / GPL-2.0)
//

#include "resource.hh"
#include "blackboxstyle.hh"

Resource::Resource(ToolWindow *toolwindow):
		BaseResource(toolwindow)
{
	defaultFont[0] = 0;
	Load();
}

Resource::~Resource()
{
	Clean();
}

void Resource::Clean()
{
	if (label.font)
		XFreeFont(bbtool->getXDisplay(), label.font);
	if (frame.font)
		XFreeFont(bbtool->getXDisplay(), frame.font);

	delete [] report.startCommand;
	delete [] report.stopCommand;
	delete [] report.resumeCommand;
}


// virtual function called by BaseResource::Load() during construction
void Resource::LoadBBToolResource(void)
{
	XrmValue value;
	char *value_type;

	if (XrmGetResource(resource_db, "bbppp.autoConfig",
	                   "Bbppp.Autoconfig", &value_type, &value)) {
		if (! strncasecmp("true", value.addr, value.size)) {
			style.auto_config = True;
		} else
			style.auto_config = False;
	} else
		style.auto_config = False;

	ButtonStyle();
	SizeAndPosition();
	Frame();
	Show();
	Label();
	Button();
	GetActions();
}

// ButtonStyle determines whether switch/rx/tx buttons should be compact or normal
void Resource::ButtonStyle()
{
	XrmValue value;
	char *value_type;

	if (XrmGetResource(resource_db, "bbppp.buttonstyle","Bbppp.Buttonstyle", &value_type, &value)) {
		if (! strncasecmp("compact", value.addr, value.size))
			compact = true;
		else
			compact = false;
	} else
		compact = false;

	SetDefaultFont();
}

void Resource::SetDefaultFont()
{
	if (compact)
		strcpy(defaultFont, "-*-helvetica-medium-r-*-*-*-100-*-*-*-*-*-*");
	else
		strcpy(defaultFont, "-*-helvetica-medium-r-*-*-*-120-*-*-*-*-*-*");
}


void Resource::Frame()
{
	XrmValue value;
	char *value_type;

	readTexture("bbppp.frame","Bbppp.Frame",BB_FRAME,"Toolbar",
	            "slategrey","darkslategrey","Raised Gradient Vertical Bevel1",
	            &frame.texture);

	if (XrmGetResource(resource_db, "bbppp.bevelWidth","Bbppp.BevelWidth",
	                   &value_type, &value)) {
		if (sscanf(value.addr, "%u", &frame.bevelWidth) != 1)
			frame.bevelWidth = 4;
		else if (frame.bevelWidth == 0)
			frame.bevelWidth = 4;
		else if (XrmGetResource(resource_db, BB_BEVELWIDTH,"BevelWidth", &value_type,
		                        &value)) {
			if (sscanf(value.addr, "%u", &frame.bevelWidth) != 1)
				frame.bevelWidth = 4;
			else if (frame.bevelWidth == 0)
				frame.bevelWidth = 4;
		} else
			frame.bevelWidth = 4;
	} else
		frame.bevelWidth = 4;


	if (XrmGetResource(resource_db, "bbppp.withdrawn.bevelWidth",
	                   "Bbppp.Withdrawn.BevelWidth",
	                   &value_type, &value)) {
		if (sscanf(value.addr, "%u", &frame.withdrawn_bevelWidth) != 1)
			frame.withdrawn_bevelWidth = 0;
	} else
		frame.withdrawn_bevelWidth = 0;

}


void Resource::SizeAndPosition()
{
	XrmValue value;
	char *value_type;
	unsigned int w,h;
	char positionstring[11];

	if (!(bbtool->getOptions()->withdrawn)) {
		if (XrmGetResource(resource_db, "bbppp.withdrawn",
		                   "Bbppp.Withdrawn", &value_type, &value)) {
			if (! strncasecmp("true", value.addr, value.size))
				bbtool->getOptions()->withdrawn = True;
			else
				bbtool->getOptions()->withdrawn = False;
		} else
			bbtool->getOptions()->withdrawn = False;
	}

	if (!(bbtool->getOptions()->shape)) {
		if (XrmGetResource(resource_db, "bbmail.shape",
		                   "Bbmail.Shape", &value_type, &value)) {
			if (! strncasecmp("true", value.addr, value.size))
				bbtool->getOptions()->shape = True;
			else
				bbtool->getOptions()->shape = False;
		} else
			bbtool->getOptions()->shape = bbtool->getOptions()->withdrawn;
	}


	if (!(bbtool->getOptions()->geometry)) {
		if (!(XrmGetResource(resource_db, "bbppp.position","Bbppp.Position",
		                     &value_type, &value)))
			strncpy(positionstring, "-0-0", 5);
		else
			strncpy(positionstring, value.addr, strlen(value.addr)+1);
	} else
		strncpy(positionstring, bbtool->getOptions()->geometry, strlen(bbtool->getOptions()->geometry)+1);


	position.mask=XParseGeometry(positionstring, &position.x, &position.y, &w, &h);
	if (!(position.mask & XValue))
		position.x=0;
	if (!(position.mask & YValue))
		position.y=0;

	if (frame.font) {
		XFreeFont(bbtool->getXDisplay(), frame.font);
		frame.font = 0;
	}

	if (XrmGetResource(resource_db, "bbppp.heightBy.font","Bbppp.heightBy.Font",
	                   &value_type, &value)) {
		if ((frame.font = XLoadQueryFont(bbtool->getXDisplay(), value.addr)) == NULL) {
			fprintf(stderr, " blackbox: couldn't load font '%s'\n"
			        " ...  reverting to default font.", value.addr);
			if ((frame.font = XLoadQueryFont(bbtool->getXDisplay(), defaultFont)) == NULL) {
				fprintf(stderr,"blackbox: couldn't load default font.  please check to\n"
				        "make sure the necessary font is installed '%s'\n",
				        defaultFont);
				exit(2);
			}
		}
	} else if (XrmGetResource(resource_db, BB_FONT,"TitleFont", &value_type, &value)) {
		if ((frame.font = XLoadQueryFont(bbtool->getXDisplay(), value.addr)) == NULL) {
			fprintf(stderr, " blackbox: couldn't load font '%s'\n"
			        " ...  reverting to default font.", value.addr);
			if ((frame.font = XLoadQueryFont(bbtool->getXDisplay(), defaultFont)) == NULL) {
				fprintf(stderr,
				        "blackbox: couldn't load default font.  please check to\n"
				        "make sure the necessary font is installed '%s'\n",
				        defaultFont);
				exit(2);
			}
		}
	} else {
		if ((frame.font = XLoadQueryFont(bbtool->getXDisplay(), defaultFont)) == NULL) {
			fprintf(stderr,
			        "blackbox: couldn't load default font.  please check to\n"
			        "make sure the necessary font is installed '%s'\n", defaultFont);
			exit(2);
		}
	}
}

void Resource::Label(void)
{
	XrmValue value;
	char *value_type;

	/* text-label resources */
	if (XrmGetResource(resource_db, "bbppp.label.transparent",
	                   "Bbppp.label.Transparent", &value_type, &value)) {
		if (! strncasecmp("true", value.addr, value.size))
			label.transparent = True;
		else
			label.transparent = False;
	} else
		label.transparent = False;

	readTexture("bbppp.label", "Bbppp.Label",BB_LABEL,"Toolbar.Label",
	            "slategrey","darkslategrey","Sunken Gradient Vertical Bevel1",
	            &label.texture);

	readColor("bbppp.textColor","Bbppp.TextColor",
	          BB_LABEL_TEXTCOLOR, "Toolbar.TextColor",
	          "LightGrey",&label.textColor);

	if (label.font) {
		XFreeFont(bbtool->getXDisplay(), label.font);
		label.font = 0;
	}

	if (XrmGetResource(resource_db, "bbppp.label.font", "Bbppp.Label.Font",	&value_type, &value)) {
		if ((label.font = XLoadQueryFont(bbtool->getXDisplay(), value.addr)) == NULL) {
			fprintf(stderr, " blackbox: couldn't load font '%s'\n"
			        " ...  reverting to default font.", value.addr);
			if ((label.font = XLoadQueryFont(bbtool->getXDisplay(), defaultFont)) == NULL) {
				fprintf(stderr, "blackbox: couldn't load default font.  please check to\n"
				        "make sure the necessary font is installed '%s'\n",	defaultFont);
				exit(2);
			}
		}
	} else if (XrmGetResource(resource_db, BB_FONT, "TitleFont", &value_type, &value)) {
		if ((label.font = XLoadQueryFont(bbtool->getXDisplay(), value.addr)) == NULL) {
			fprintf(stderr, " blackbox: couldn't load font '%s'\n"
			        " ...  reverting to default font.", value.addr);
			if ((label.font = XLoadQueryFont(bbtool->getXDisplay(), defaultFont)) == NULL) {
				fprintf(stderr,	"blackbox: couldn't load default font.  please check to\n"
				        "make sure the necessary font is installed '%s'\n",	defaultFont);
				exit(2);
			}
		}
	} else if ((label.font = XLoadQueryFont(bbtool->getXDisplay(), defaultFont)) == NULL) {
		fprintf(stderr,"blackbox: couldn't load default font.  please check to\n"
		        "make sure the necessary font is installed '%s'\n", defaultFont);
		exit(2);
	}
}


void Resource::Show()
{
	XrmValue value;
	char *value_type;

	// this resource should really be named "show.uptime",
	// but we dont want to affect local scripts unnecessarily
	if (XrmGetResource(resource_db, "bbppp.show.label",
	                   "Bbppp.show.Label", &value_type, &value)) {
		if (! strncasecmp("false", value.addr, value.size))
			show.upTimeLabel = False;
		else
			show.upTimeLabel = True;
	} else
		show.upTimeLabel = True;

	if (XrmGetResource(resource_db, "bbppp.show.thruput",
	                 "Bbppp.show.thruput", &value_type, &value)) {
		if (! strncasecmp("false", value.addr, value.size))
			show.thruputLabel = False;
		else
			show.thruputLabel = True;
	} else
		show.thruputLabel = False;
	
	if (XrmGetResource(resource_db, "bbppp.show.secs",
	                   "Bbppp.show.secs", &value_type, &value)) {
		if (! strncasecmp("false", value.addr, value.size))
			show.upTimeLabel = False;
		else
			show.upTimeLabel = True;
	} else
		show.upTimeLabel = True;


}

void Resource::Button()
{
	XrmValue value;
	char *value_type;

	if (XrmGetResource(resource_db, "bbppp.checkDelay",
	                   "Bbppp.checkDelay",
	                   &value_type, &value)) {
		if (sscanf(value.addr, "%u", &report.check_delay) != 1)
			report.check_delay = 1;
		else if (report.check_delay == 0)
			report.check_delay = 1;
	} else
		report.check_delay = 1;


	if (XrmGetResource(resource_db, "bbppp.button.bevelWidth",
	                   "Bbppp.Button.BevelWidth",
	                   &value_type, &value)) {
		if (sscanf(value.addr, "%u", &button.bevelWidth) != 1)
			button.bevelWidth = 4;
		else if (button.bevelWidth == 0)
			button.bevelWidth = 4;
	} else
		button.bevelWidth = 6;

	readTexture("bbppp.button", "Bbppp.Button",BB_BUTTON,"Toolbar.Button",
	            "slategrey","darkslategrey","Raised Gradient Diagonal Bevel1",
	            &button.texture);

	readTexture("bbppp.button.pressed", "Bbppp.Button.Pressed",
	            BB_BUTTON_PRESSED,"Toolbar.Button.Pressed",
	            "slategrey","darkslategrey","Sunken Gradient Diagonal Bevel1",
	            &button.texture_pressed);


	readColor("bbppp.switchColor.on","Bbppp.SwitchColor.On",0,0,
	          "green",&button.switch_on_color);

	readColor("bbppp.switchColor.off","Bbppp.SwitchColor.Off",0,0,
	          "black",&button.switch_off_color);

	readColor("bbppp.switchColor.switching","Bbppp.SwitchColor.Switching",
	          0,0,"yellow",&button.switch_switching_color);

	readColor("bbppp.txColor.on","Bbppp.TxColor.On",0,0,
	          "green1",&button.tx_on_color);

	readColor("bbppp.txColor.off","Bbppp.TxColor.Off",0,0,
	          "black",&button.tx_off_color);

	readColor("bbppp.rxColor.on","Bbppp.RxColor.On",0,0,
	          "green1",&button.rx_on_color);

	readColor("bbppp.rxColor.off","Bbppp.RxColor.Off",0,0,
	          "black",&button.rx_off_color);
}

void Resource::GetActions()
{
	XrmValue value;
	char *value_type;

	if (XrmGetResource(resource_db,"bbppp.startCommand",
	                   "Bbppp.StartCommand",&value_type, &value)) {
		int len = strlen(value.addr);
		delete [] report.startCommand;
		report.startCommand = new char[len + 1];
		memset(report.startCommand, 0, len + 1);
		strncpy(report.startCommand, value.addr, len);
	} else {
		delete [] report.startCommand;
		report.startCommand = new char[7];
		memset(report.startCommand, 0, 7);
		strncpy(report.startCommand, "ppp-on", 6);
	}

	if (XrmGetResource(resource_db,"bbppp.stopCommand",
	                   "Bbppp.StopCommand",&value_type, &value)) {
		int len = strlen(value.addr);
		delete [] report.stopCommand;
		report.stopCommand = new char[len + 1];
		memset(report.stopCommand, 0, len + 1);
		strncpy(report.stopCommand, value.addr, len);
	} else {
		delete [] report.stopCommand;
		report.stopCommand = new char[8];
		memset(report.stopCommand, 0, 8);
		strncpy(report.stopCommand, "ppp-off", 7);
	}

	if (XrmGetResource(resource_db,"bbppp.resumeCommand",
	                   "Bbppp.ResumeCommand",&value_type, &value)) {
		int len = strlen(value.addr);
		delete [] report.resumeCommand;
		report.resumeCommand = new char[len + 1];
		memset(report.resumeCommand, 0, len + 1);
		strncpy(report.resumeCommand, value.addr, len);
	} else
		report.resumeCommand=NULL;
}
