/* unit-pfs.c
 *
 ****************************************************************
 * Copyright (C) 2005, Canonical Limited
 *               Author: James Blackwell
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */

#include "hackerlab/bugs/panic.h"
#include "hackerlab/char/str.h"
#include "hackerlab/vu/safe.h"
#include "libarch/pfs.h"
#include "libarch/authinfo.h"


static void dav_parse_authlist_correct (void);
static void dav_parse_authlist_incorrect (void);

int
main (int argc, char * argv[])
{

  dav_parse_authlist_correct();
  dav_parse_authlist_incorrect();

  return 0;
}

void
dav_parse_authlist_correct ()
{
  rel_table parsed_file;

/* -------------------------------------------
 * All one one line
 */
  parsed_file = arch_auth_parse_auth_file (
"machine santasworkshop.north.pl login santaclause password elf");
  if (1 != rel_n_records(parsed_file))
    panic ("simple authlist failure\n");
  rel_free_table (parsed_file);

/* -------------------------------------------
 * Lets have some leading spaces in this one
 */
  parsed_file = arch_auth_parse_auth_file (
"  \t  machine santasworkshop.north.pl login santaclause password elf");
  if (1 != rel_n_records(parsed_file))
    panic ("simple authlist failure\n");
  rel_free_table (parsed_file);


/*  ------------------------------------------
 *  How about space crazy? 
 */
  parsed_file = arch_auth_parse_auth_file (
"machine \"santasworkshop.north.pl\" login \"santaclause\" password \"elf\"");
  if (1 != rel_n_records(parsed_file))
    panic ("simple authlist failure\n");
  rel_free_table (parsed_file);



/* -------------------------------------------
 * This one has spaces and tabs
 */
  parsed_file = arch_auth_parse_auth_file (
"machine santasworkshop.north.pl\tlogin santaclause\tpassword elf\n");
  if (1 != rel_n_records(parsed_file))
    panic ("simple authlist with tabs failure\n");

  rel_free_table (parsed_file);

/* -------------------------------------------
 * Spaces and newlines
 */
  parsed_file = arch_auth_parse_auth_file (
"machine santasworkshop.north.pl\nlogin santaclause password\nelf");
  if (1 != rel_n_records(parsed_file))
    panic ("simple authlist with spaces and newlines failed\n");
  rel_free_table (parsed_file);

/* -------------------------------------------
 * With " "" "" """"" """ "" ! 
 */
  parsed_file = arch_auth_parse_auth_file (
"machine santasworkshop.north.pl\nlogin \"santa clause\" password\nelf");
  if (1 != rel_n_records(parsed_file))
    panic ("simple authlist with spaces and newlines failed\n");
  rel_free_table (parsed_file);



/* -------------------------------------------
 * This one gets three entries with all the above
 */
  parsed_file = arch_auth_parse_auth_file (
"machine santasworkshop.north.pl\nlogin santaclause\tpassword\telf\n"
"machine mafia.org\nlogin\tgotti password\ndontdropthesoap "
"machine movebits.org login garry_mcyol password\tigotfired");
  if ( 3 != rel_n_records(parsed_file))
    panic ("authlist failure for three entries\n");

  rel_free_table (parsed_file);
  
}


void
dav_parse_authlist_incorrect ()
{
  rel_table parsed_file;

/* -------------------------------------------
 * All one one line
 */
  parsed_file = arch_auth_parse_auth_file (
"machine santasworkshop.south.pl machine santasworkshop.north.pl login santaclause password elf");
  if (1 != rel_n_records(parsed_file))
    panic ("extra field not ignored\n");
  if (0 != str_cmp(parsed_file[0][1], "santaclause"))
    panic ("Wrong login (all in one)\n");
  rel_free_table (parsed_file);

/* -------------------------------------------
 * No password
 */
  parsed_file = arch_auth_parse_auth_file (
"machine santasworkshop.south.pl login santaclause\nmachine otherhost login me password mypass");
  if (0 != str_cmp(parsed_file[0][1], "santaclause"))
    panic ("Wrong login (no pass)\n");
  if (0 != str_cmp(parsed_file[0][2], ""))
    panic ("Empty password not parsed correctly");
  if (0 != str_cmp(parsed_file[1][2], "mypass"))
    panic ("Second password not parsed correctly");
  rel_free_table (parsed_file);

/* -------------------------------------------
 * This one has spaces and tabs
 */
  parsed_file = arch_auth_parse_auth_file (
"macchine santasworkshop.north.pl\tlogin santaclause\tpassword elf\n");
  if (0 != rel_n_records(parsed_file))
    panic ("misspelled description not caught\n");

  rel_free_table (parsed_file);

/* -------------------------------------------
 * Duplicated fields
 */
  parsed_file = arch_auth_parse_auth_file (
"machine santasworkshop.north.pl\nmachine santaclause password\nelf");
  if (0 != rel_n_records(parsed_file))
    panic ("duplicated field failed\n");
  rel_free_table (parsed_file);

/* -------------------------------------------
 * Quotes
 */
  parsed_file = arch_auth_parse_auth_file (
"machine santasworkshop.north.pl login \"foo bar\" password \"my secret pass\"\n");
  if (0 != str_cmp(parsed_file[0][1], "foo bar"))
    panic ("Wrong login (quotes)\n");
  if (0 != str_cmp(parsed_file[0][2], "my secret pass"))
    panic ("Wrong password (quotes)\n");
  rel_free_table (parsed_file);

}

/* tag: James Blackwell Mon Apr 11 22:42:21 EDT 2005 (unit-pfs-dav.c) */

