/* init-tree.c
 *
 * vim:smartindent ts=8:sts=2:sta:et:ai:shiftwidth=2
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/cmd/main.h"
#include "libarch/namespace.h"
#include "libarch/project-tree.h"
#include "libarch/patch-logs.h"
#include "commands/cmd.h"
#include "commands/init-tree.h"
#include "commands/version.h"



static t_uchar * usage = N_("[options] [[archive]/version]");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit.")) \
  OP (opt_archive, "A", "archive", 1, \
      N_("Override `my-default-archive'")) \
  OP (opt_dir, "d", "dir DIR", 1, \
      N_("cd to DIR first")) \
  OP (opt_nested, 0, "nested", 0, \
      N_("initialize a nested project tree."))


t_uchar arch_cmd_init_tree_help[] = N_("initialize a new project tree\n"
                                     "Initialize DIR as a new project tree.\n"
                                     "\n"
                                     "An empty patch-log for VERSION will be created, and VERSION\n"
                                     "will be made the default location for check-ins.\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_init_tree (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * default_archive = 0;
  char * dir = 0;
  int nested = 0;

  dir = str_save (0, ".");
  nested = 0;

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_init_tree_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

        case opt_archive:
          {
            default_archive = str_save (0, option->arg_string);
            break;
          }

        case opt_dir:
          {
            dir = str_save (0, option->arg_string);
            break;
          }

        case opt_nested:
          {
            nested = 1;
            break;
          }
        }
    }

  if (argc > 2)
    goto usage_error;

  {
    t_uchar * version_spec;
    t_uchar * tree_root = 0;
    t_uchar * archive = 0;
    t_uchar * version = 0; /* should not be inited */
    t_uchar * fqversion = 0; /* but gcc gets it wrong */

    if (argc == 2)
      {
        version_spec = argv[1];
      }
    else
      {
        version_spec = 0;
      }

    if (default_archive)
      {
        if (!arch_valid_archive_name (default_archive))
          {
            safe_printfmt (2, "%s: invalid archive name (%s)\n",
                           argv[0], default_archive);
            exit (1);
          }
      }

    if (version_spec)
      {
        if (!arch_valid_package_name (version_spec, arch_maybe_archive, arch_req_version, 0))
          {
            safe_printfmt (2, "%s: invalid version name (%s)\n",
                           argv[0], version_spec);
            exit (1);
          }
        archive = arch_parse_package_name (arch_ret_archive, default_archive, version_spec);
        version = arch_parse_package_name (arch_ret_package_version, 0, version_spec);
        fqversion = arch_fully_qualify (archive, version);
      }

    tree_root = arch_tree_root (0, dir, 0);

    if (!nested && tree_root)
      {
        safe_printfmt (2, "%s: dir is already in project tree (%s)\n",
                       argv[0], dir);
        exit (1);
      }

    tree_root = str_save (0, dir);
    arch_init_tree (tree_root);
    if (version)
      {
        arch_patch_id * patch_id = arch_patch_id_new (fqversion);;
        arch_add_log_version (tree_root, archive, version);
        arch_set_tree_version (tree_root, patch_id);
        talloc_free (patch_id);
      }

    lim_free (0, tree_root);
    lim_free (0, archive);
    lim_free (0, version);
    lim_free (0, fqversion);
  }

  lim_free (0, default_archive);
  lim_free (0, dir);

  return 0;
}




/* tag: Tom Lord Tue May 13 08:33:06 2003 (init-tree.c)
 */
