/* cmd-whereis-archive.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/cmd/main.h"
#include "libfsutils/ensure-dir.h"
#include "libarch/my.h"
#include "libarch/namespace.h"
#include "libarch/archives.h"
#include "commands/whereis-archive.h"
#include "commands/version.h"



static t_uchar * usage = N_("[options] archive");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit."))

t_uchar arch_cmd_whereis_archive_help[] = N_("print an archive location registration\n"
                                           "Print the registered location of an archive.\n"
                                           "\n"
                                           "Usually the archive must have been previously registered with\n"
                                           "\"tla register-archive\".\n"
                                           "\n"
                                           "As a special exception, the the archive is not registered, but\n"
                                           "is the name of the archive rooted at the location given with\n"
                                           "the option -R (--root) or in the environment variable ARCHROOT\n"
                                           "then print that root directory.\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_whereis_archive (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * which_archive;
  t_uchar * found_archive;

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_whereis_archive_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;
        }
    }

  if (argc != 2)
    goto usage_error;

  which_archive = argv[1];

  if (!arch_valid_archive_name (which_archive))
    {
      safe_printfmt (2, "%s: invalid archive name (%s)\n",
                     argv[0], which_archive);
      exit (2);
    }

  found_archive = arch_archive_location (which_archive, 1);

  if (!found_archive)
      exit (1);
  
  safe_printfmt (1, "%s\n", found_archive);

  return 0;
}




/* tag: Tom Lord Sun May 18 21:37:02 2003 (whereis-archive.c)
 */
