/***************************************************************************
 *   Copyright (C) 2003 by Sbastien Laot                                 *
 *   sebastien.laout@tuxfamily.org                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <kmimetype.h>
#include <qdir.h>
#include <klocale.h>

#include "basketfactory.h"
#include "global.h"
#include "container.h"
#include "basket.h"

/** BasketFactory */

QString BasketFactory::createBasketClips(const QString &name, const QString &folderName)
{
	QString realName   = newBasketName(Clips, name);
	QString properties = QString(
		" <name>%1</name>\n" )
			.arg(escapeXML(realName));
	return makeBasketDir(properties, folderName);
}

QString BasketFactory::createBasketChecks(const QString &name, const QString &folderName)
{
	QString realName   = newBasketName(Checklist, name);
	QString properties = QString(
		"  <name>%1</name>\n"
		"  <icon>todo</icon>\n"
		"  <showCheckBoxes>true</showCheckBoxes>\n" )
			.arg(escapeXML(realName));
	return makeBasketDir(properties, folderName);
}

QString BasketFactory::createBasketStack(bool takeOnSameSide, int afterDrag, const QString &name, const QString &folderName)
{
	QString realName   = newBasketName(Stack, name);
	QString properties = QString(
		"  <name>%1</name>\n"
		"  <icon>db</icon>\n"
		"  <stack activated=\"true\" takeOnSameSide=\"%2\" afterDrag=\"%3\" />\n"
		"  <addItemPolicy insertAtCursorPos=\"false\" />\n" )
			.arg(escapeXML(realName))
			.arg(takeOnSameSide ? "true" : "false")
			.arg(QString::number(afterDrag));
	return makeBasketDir(properties, folderName);
}

QString BasketFactory::createBasketMirror(bool onlyNewFiles, bool reload, const QString &name, const QString &folderName)
{
	QString realName     = newBasketName(Mirror, name, folderName);
	if ( ! QDir(folderName).exists(".basket") ) // If the user want to reload an unexisting basket file
		reload = false;
	if ( !reload ) {
		QDir dir(folderName);
		dir.remove(".basket"); // If it was existing
		QString properties   = QString(
			"  <name>%1</name>\n"
			"  <icon>%2</icon>\n"
			"  <mirror onlyNewFiles=\"%3\" />\n" )
				.arg(escapeXML(realName))
				.arg(KMimeType::iconForURL( KURL(folderName) ))
				.arg(onlyNewFiles ? "true" : "false");
		return makeBasketDir(properties, folderName);
	} else {
		// It alread exists: Do nothing and reload it
		return folderName;
	}
}

QString BasketFactory::createBasketClipboard(const QString &name, const QString &folderName)
{
	QString realName   = newBasketName(Clipboard, name);
	QString properties = QString(
		"  <name>%1</name>\n"
		"  <icon>klipper</icon>\n"
		"  <clipboard activated=\"true\" />\n"
		"  <stack activated=\"true\" afterDrag=\"0\" />\n" )
			.arg(escapeXML(realName));
	return makeBasketDir(properties, folderName);
}

void BasketFactory::reloadAMirrorBasket(const QString &folderName)
{
	// TODO: Take care if it doesn't exists or is corrupted
	Global::mainContainer->loadBasket(folderName);
}

QString BasketFactory::newBasketName(BasketType basketType, const QString &wantedName, const QString &folderName)
{
	QString nameBase;
	QString nameTemplate;

	switch (basketType) {
		case Clips:
			return newBasketName( i18n("Basket"),    i18n("Basket %1"),    wantedName );
		case Checklist:
			return newBasketName( i18n("Checklist"), i18n("Checklist %1"), wantedName );
		case Stack:
			return newBasketName( i18n("Stack"),     i18n("Stack %1"),     wantedName );
		case Mirror:
			nameBase     = i18n("like \"folderFoo mirror\"",   "%1 mirror")   .arg( QDir(folderName).dirName() );
			nameTemplate = i18n("like \"folderFoo mirror 2\"", "%1 mirror %2").arg( QDir(folderName).dirName() );
			return newBasketName( nameBase,          nameTemplate,         wantedName );
		case Launcher:
			return newBasketName( i18n("Launcher"),  i18n("Launcher %1"),  wantedName );
		case Clipboard:
			return newBasketName( i18n("Clipboard manager"), i18n("Clipboard manager %1"),  wantedName );
		default:
			return "";
	}
}

QString BasketFactory::newFolderName()
{
	QString folder;
	QString fullPath;
	QDir    dir;

	for (int i = 1; ; ++i) {
		folder   = "basket" + QString::number(i) + "/";
		fullPath = Global::basketsFolder() + folder;
		dir      = QDir(fullPath);
		if ( ! dir.exists() ) // OK : The folder do not yet exists :
			break;            //  We've found one !
	}

	return folder;
}

QString newBasketName(  int basketType );

QString BasketFactory::escapeXML(QString text)
{
	return text.replace("&", "&amp;").replace("<", "&lt;").replace(">", "&gt;").replace("\"", "&quot;");
}

QString BasketFactory::newBasketName(const QString &baseName, const QString &templateName, const QString &wantedName)
{
	// If the user specified a name, take it
	if (wantedName != "")
		return wantedName;

	// If not, try to find a unique one: First try the base name, more beatifull:
	if ( ! isAlreadyNamed(baseName) )
		return baseName;

	// Else, we should add a number to the base name:
	for (int i = 2; ; ++i)
		if ( ! isAlreadyNamed(templateName.arg( QString::number(i) )) )
			return templateName.arg( QString::number(i) );
	// FIXME: If exists "Stack 1" and "Stack 3",
	//         return "Stack 4" instead of "Stack 2" !!!
	//        Or use (Container::count() + 1)
}

bool BasketFactory::isAlreadyAClipboard()
{
	QPtrList<Basket> baskets = Global::mainContainer->listBaskets();

	for (uint i = 0; i < baskets.count(); ++i )
		if (baskets.at(i)->isAClipboard())
			return true;

	return false;
}

bool BasketFactory::isAlreadyMirrored(const QString &folderName)
{
	QPtrList<Basket> baskets = Global::mainContainer->listBaskets();

	for (uint i = 0; i < baskets.count(); ++i )
		if ( KURL(baskets.at(i)->folderName()) == KURL(folderName) ) // KURL to normalize them
			return true;

	return false;
}

bool BasketFactory::isAlreadyNamed(const QString &basketName)
{
	QPtrList<Basket> baskets = Global::mainContainer->listBaskets();

	for (uint i = 0; i < baskets.count(); ++i )
		if ( baskets.at(i)->name() == basketName )
			return true;

	return false;
}

QString BasketFactory::makeBasketDir(const QString &properties, const QString &folderName)
{
	// Choose and create the folder if it doesn't exists :
	QString realFolderName = ( folderName.isEmpty() ? newFolderName() : folderName );
	QString fullPath = Basket::fullPathForFolderName(realFolderName);
	QDir dir("");
	dir.mkdir(fullPath);

	QFile file(fullPath + "/.basket");
	if ( file.open(IO_WriteOnly) ) {
		QTextStream stream(&file);
		stream.setEncoding(QTextStream::UnicodeUTF8);
		stream << QString( "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
		                   "<!DOCTYPE basket>\n"
		                   "<basket>\n"
		                   " <properties>\n" )
		       <<              properties
		       << QString( " </properties>\n"
		                   "</basket>\n"     );
		file.close();
	}

	return realFolderName;
}
