/** *********************************************************************
 * Copyright (C) 2003 Catalyst IT                                       *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to:                           *
 *   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    *
 *   Boston, MA  02111-1307  USA                                        *
 ************************************************************************/
package nz.net.catalyst.lucene.server;

import java.io.File;
import java.io.IOException;

import nz.net.catalyst.ELog;
import nz.net.catalyst.Log;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.Term; 

/**
 * This class implements a LuceneServer UnIndex request.
 *<p>
 * The format of an unindex request is as follows:
 *<pre>UNINDEX
 *Application: your_app_name_here
 *Id: id_of_document_to_delete
 *END</pre>
 *<p>
 *There is also a development only purge feature.  
 *<b>DO NOT DEPEND ON THIS TO REMAIN IN THE LUCENE SERVER.</b>
 *It will remove all documents for a given application using the application's
 *domain.  <i>There is no support for an alternative domain at this time.</i>
 *<p>
 *To use it use the following syntax:
 *<pre>UNINDEX
 *Application: your_app_name_here
 *Purge: true
 *END</pre>
 */

public class UnIndex implements IPackage, Constants {
  private final Transmission input;

  private Application application;

  /**
   * Create an UnIndex object with the transmission data set to
   * the passed transmission
   * @param transmission The transmission data to initialise the UnIndex
   * object with.
   */
  UnIndex(Transmission transmission) {
    input = transmission;
  }

  /**
   * UnIndex (delete) a document
   */
  Transmission execute() {
    boolean purge         = false;
    String id             = null;
    Transmission response = new Transmission(ECommand.UNINDEX_RESPONSE);
    response.setSerial(input.getSerial()); //set the serial if client is using serial numbers to identify requests.
    int count 			  = -1; //no of documents deleted.
    //process the input.  Either purge or delete one document.  If neither then fail.
    String purgeHeader = input.get("Purge", NO_APP);
    if (purgeHeader != null) {
        purge = true;
    } else {
        id = input.get(ID, NO_APP);
        if (id == null)
          return error("Mandatory '" + ID + "' header is missing");
    }
    
    String appName = input.get(APPLICATION, NO_APP);
    try {
      application = Application.getAppOrDefault(appName);
    } catch (ApplicationMissingException e) {
      return error(e.getMessage());
    }

    input.setApplication(application);
    String domain = input.get(DOMAIN);

    if (domain == null)
      return error("'" + DOMAIN +
                   "' header is missing and is not in Application defaults");
    
    File luceneStoreDir = Application.getIndexDirectory(application);
    
    long beforeOpen       = System.currentTimeMillis();
    long afterReaderOpen  = 0L;
    long afterReaderClose = 0L;
    long afterDelete      = 0L;

    // Delete the current instance of the document from the index.

    IndexReader reader = null;

    try {
      reader = IndexReader.open(luceneStoreDir);
      afterReaderOpen = System.currentTimeMillis();
      Term keyField;
      if (purge) {
          Log.log(ELog.NOTICE, "About to PURGE All documents from domain: " + domain);
          keyField = new Term(DOMAIN, domain);
      } else {
          keyField = new Term(DOCUMENT_KEY, domain + "\u0000" + id);
      }
      count = reader.deleteDocuments(keyField);
      afterDelete = System.currentTimeMillis();
      Log.debug("Deleted " + count + " documents (this should always be one unless purging.)");
    } catch (java.io.FileNotFoundException e) {
      // Maybe the index is brand new and contains no files!
      Log.log(ELog.ERROR, "Possible Cause: Index is brandnew and contains no files yet.\n" +
                          "You need to index something before deleting it!");
      Log.log(ELog.ERROR, e.toString());
      return error("Attempt to delete on non-existing index. Index either missing or empty.");
    } catch (IOException e) {
      String message = "Error while deleting previous document " + DOMAIN +
        '=' + domain + ", " + ID +'=' + id + ": " + e.toString();
      Log.log(ELog.ERROR, message);
      return error(message);
    } finally {
      if (reader != null) {
        try {
          // We must always close the IndexReader!
          reader.close();
        } catch (Throwable e) {
          String message = "Error while closing IndexReader after delete: " + e.toString();
          Log.log(ELog.ERROR, message);
          return error(message);
        }
      }
    }
   
    afterReaderClose = System.currentTimeMillis();
 	
 	if (count == 1 ) {
 		response.add("Status", "Successfully deleted document with id: " + id);
 	} else if (count > 1) {
 		response.add("Status", "Purge Successful.  Deleted " + count + " documents");
 	} else {
 		response.add("Status", "No documents deleted");
 	}
    
    Log.debug("Open"   + String.valueOf(afterReaderOpen - beforeOpen));
    Log.debug("Delete" + String.valueOf(afterDelete - afterReaderOpen));
    Log.debug("Close"  +  String.valueOf(afterReaderClose - afterDelete));
    return response;
  }


  /**
   * Build an error response for sending back to the client.
   *
   * @param message The text of the error message
   * @return An UNINDEX-RESPONSE Transmission
   */
  private Transmission error(String message)  {
    Transmission response = new Transmission(ECommand.UNINDEX_RESPONSE);
    response.setSerial(input.getSerial());
    response.add(ERROR, message);
    return response;
  }
}