/** *********************************************************************
 * Copyright (C) 2003 Catalyst IT                                       *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to:                           *
 *   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    *
 *   Boston, MA  02111-1307  USA                                        *
 ************************************************************************/
package nz.net.catalyst.lucene.server;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;

import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.io.OutputFormat;
import org.dom4j.io.XMLWriter;

import nz.net.catalyst.Log;
import nz.net.catalyst.StringPair;
import nz.net.catalyst.Util;


/**
 * Read, store and write a Transmission.  A transmission consists of a
 * Command word and an ordered list of headers.  The headers consist
 * of a keyword and value.  The header list may contain duplicate
 * keywords.  Only one keyword is treated specially by this class:
 * Content-Length.  This is used in case a body is requested.
 */

public class Transmission implements IPackage, Constants {
  /**
   * A value which represents an invalid Content-Length value.
   */
  private static final Long INVALID_CONTENT_LENGTH = new Long(Long.MIN_VALUE);

  /**
   * The command associated with this header.
   */
  private final ECommand command;

  /**
   * The list of headers, in order.  Each header is made up of a StringPair
   */
  private final List headers = new LinkedList();

  /**
   * An unmodifiable view of the headers that can be given away safely.
   */
  private final List headersView = Collections.unmodifiableList(headers);

  /**
   * Identifies the header key that will be written as a body.  If no
   * such header key exists, no body is written.  If more than one
   * exists, the last is written.  This will override any
   * Content-Length header that may exist in the header list.
   */
  private String body = null;

  /**
   * An optional Application object which will augment the values
   * associated with this transmission.  In effect, the fields in the
   * Application default or add to (depending on context) the fields
   * in the Transmission object
   */
  private Application application = null;

  /**
   * A string containing the result (if any) of the transmission.
   * This is primarily intended as a replacement for the out.print 
   * mechanism that was previously used.
   */
  private StringBuffer result = new StringBuffer();
  
  /**
   * Flag to deetermine whether transmission is xml based or not.
   * CURRENTLY THIS IS SET MANUALLY!!
   */
  private boolean XML = false;
  
  public Transmission(ECommand command)
  {
    this.command = command;
  }

  public ECommand getCommand()
  {
    return command;
  }

  /**
   * Reads the buffered input (Reader) and makes a transmission based on the 
   * lines contained within.<p>
   * It will loop through the input line by line looking for a command header
   * eg INDEX, QUERY, CONTROL, UNINDEX etc.
   * Once it has found a suitable header it will create a new transmission
   * based on that header.
   */
  public static Transmission create(BufferedReader in) throws IOException
  {
    String line;
    boolean skipping = false; 

    while (null != (line = in.readLine()))
    {
      ECommand command = ECommand.getByName(line.trim());
      if (command != null)
      {
        Transmission result = new Transmission(command);
        result.read(in);
        return result;
      }

      if (!skipping)
      {
        skipping = true;
        Log.error("Skipping crap starting with: \"" + line + '"');
      }
    }
    return null;
  }
  
  /**
   * Add a header to the list of headers.
   */
  public void add(String header, String value)
  {
    headers.add(new StringPair(header.intern(), value));
  }

  /**
   * Set a default application to augment the fields in the
   * Transmission object.  Set to null to clear it out.
   */
  public void setApplication(Application application)
  {
    this.application = application;
  }

  public Application getApplication()
  {
    return application;
  }

  /**
   * Return the value of the last header with the requested key.  If
   * not specified, then get the value from the default Application
   *
   * @param key The header key to be retrieved
   *
   * @return The value of the last header with the key, or null if
   * neither the headers nor the Application contain the key.
   */
  public String get(String key)
  {
    return get(key, USE_APP);
  }

  /**
   * Return the value of the last header with the requested key.  If
   * not specified, then optionally get the value from the default
   * Application
   *
   * @param key The header key to be retrieved
   * @param useApp Whether to check the default Application if
   *               the key is not found in this Transmission
   *
   * @return The value of the last header with the key, or null if
   * neither the headers nor the Application contain the key.
   */
  public String get(String key, EUseApp useApp)
  {
    String[] result = get(key, useApp, NO_SPLIT);
    return result.length >= 1 ? result[0] : null;
  }

  /**
   * General getter.  Gets values in last header associated with a key
   * subject to a couple of constraints.
   *
   * @param key The key for which values are required.
   * @param useApp If a default application has been provided,
   *               then look there if the Transmission does
   *               not contain the value.
   * @param split If every individual header value is to be
   *              split into separate white-space separated
   *              values
   *
   * @return A List of String objects retrieved.  The return value is
   *         never null, but may be zero-length.  If splitValues is
   *         false, the List length is never greater than one.
   */

  public String[] get(String key, EUseApp useApp, ESplit split)
  {
    String value = null;
    
    for (ListIterator it = headers.listIterator(headers.size()); it.hasPrevious(); )
    {
      StringPair entry = (StringPair)it.previous();
      String entryKey = entry.getKeyString();
      
      if (entryKey.equals(key))
      {
        value = entry.getValueString();
        break;
      }
    }

    if (value == null && useApp == USE_APP && application != null)
      value = application.getProperty(key);

    if (value == null)
      return new String[0];

    if (split == SPLIT)
      return Util.split(value);

    return new String[]{value};
  }

  /**
   * Assign a value to the body.  On writing the Transmission to the
   * stream, if the body is non-null, a Content-Length header will be
   * appended as the last header and the value of this body will be
   * written out.  It is up to the caller to ensure that the "Body" header
   * specifies a field which will receive the body at the receiver.
   *
   * @param body The data to be sent through as a meassge body, or
   * null if no message body is to be sent.
   */
  public void setBody(String body)
  {
    this.body = body;
  }

  /**
   * Return an unmodifiable view over the List of headers.  This view
   * is backed by the actual header list so it will change as
   * additional headers are added.
   *
   * @return a List containing StringPair objects.
   */
  public List getHeadersView()
  {
    return headersView;
  }


  /**
   * Remove all headers with the specified key, and return them as a list.
   * Strings as there were headers matching the given key (possibly
   * zero) in the same order as they were specified.
   *
   * @param header The header key to be removed
   *
   * @return a List of String objects, possibly zero-length, but not null.
   */
  public List removeAll(String header)
  {
    List result = new ArrayList(headers.size());
    for (Iterator it = headers.iterator(); it.hasNext(); )
    {
      StringPair entry = (StringPair)it.next();
      if (entry.getKeyString().equalsIgnoreCase(header))
      {
        it.remove();
        result.add(entry.getValueString());
      }
    }
    return result;
  }

  /**
   * Remove all headers.
   */
  public void clear()
  {
    headers.clear();
  }
  
  /** 
   * Outputs the transmission to the passed print writer.
   * This is used to build responses to commands.
   */
  public void write(PrintWriter out)
  {
    if (this.isXML()) {
    	
    	Document document = this.writeXML();
	   //quick hack by hamish
/****************************************	   	   
    	//Document document = DocumentHelper.createDocument();

    	// **********************************************************************************************************
    	if (command.getName().equals("QUERY-RESPONSE")) {
			this.writeLuceneQueryResponse(document);
   		// **********************************************************************************************************
	   } else {	
    		Element root = document.addElement("LuceneResponse");

    		String _Serial = get("Serial");
    		if (_Serial != null) {
    			 removeAll("Serial"); //so it won't show up as a field
    			 root.addElement("Serial").addText(_Serial);
    		}
			
			root.addAttribute("type", command.getName());

			for (Iterator it = headers.iterator(); it.hasNext(); ) {
				StringPair header = (StringPair)it.next();
				root.addElement(header.getKeyString()).addCDATA(header.getValueString());
		   	}
	   }
//****************************************/
    	OutputFormat of  = OutputFormat.createPrettyPrint();
    		try {
    	//XMLWriter writerLog    = new XMLWriter(new java.io.PrintWriter(new java.io.BufferedOutputStream(System.out)), of);
    	XMLWriter writerSocket = new XMLWriter(out, of);
    	//writerLog.write(document);
    	writerSocket.write(document);
    		} catch (Exception e) {
    			//this shouldn't happen.
    			Log.error("Error outputting xml.");
    			Log.error(e.getMessage());
    		}

    	Log.debug("DOM4J Output:\n" + document.asXML());
    	//out.print(document.asXML());

	   
		   
  	} else {	
        out.println(command.getName());
        Log.debug("> " + command.getName());

        for (Iterator it = headers.iterator(); it.hasNext(); )
        {
          StringPair header = (StringPair)it.next();
          if (Log.willDebug())
            Log.debug("> " +
                      header.getKeyString() + ": " + header.getValueString());
          out.print(Util.textEncode(header.getKeyString(), " :"));
          out.print(": ");
          out.println(Util.textEncode(header.getValueString()));
        }
        if (body != null)
        {
          if (Log.willDebug())
            Log.debug("> " + CONTENT_LENGTH + ": " + body.length());
          out.print(CONTENT_LENGTH);
          out.print(": ");
          out.println(body.length());
          out.println();
          out.write(body);
          if (Log.willDebug())
            Log.debug("> <body>");
        }
        else
        {
          out.println(END);
          if (Log.willDebug())
            Log.debug("> END");
        }

        out.flush();
    }
  }

	 /**
	  * Reads the input and parses it into key/value pairs.
	  */
	public void read(BufferedReader in) throws IOException {
		String line;
		boolean body = false;
		String bodyField = null;
		Long contentLength = null;
		long startTime = System.currentTimeMillis();
	
		while (null != (line = in.readLine())) {
			if (Log.willDebug())
				Log.debug("< " + Util.clean(line));
	
			if (line.length() == 0) {
				body = true;
				break;
			}
	
			line = line.trim();
			if (line.equalsIgnoreCase(END))
				break;
	
			int separator = line.indexOf(':');
	
			if (separator == -1)
				throw new TransmissionException(
					"Missing ':' in header line: \"" + line + '"');
			String header = line.substring(0, separator).trim();
			if (header.length() == 0)
				throw new TransmissionException(
					"Blank header illegal in header line: \"" + line + '"');
	
			header = Util.textDecode(header);
			String value = Util.textDecode(line.substring(separator + 1).trim());
			if (Log.willDebug())
				Log.debug("==> header: \""
						+ Util.clean(header)
						+ "\", Value: \""
						+ Util.clean(value)
						+ "\"");
	
			add(header, value);
			if (header.equalsIgnoreCase(BODY))
				bodyField = value;
	
			if (header.equalsIgnoreCase(CONTENT_LENGTH)) {
				try {
					contentLength = new Long(value.trim());
				} catch (NumberFormatException e) {
					contentLength = INVALID_CONTENT_LENGTH;
				}
			}
		}
	
		long beforeBody = System.currentTimeMillis();
	
		if (body) {
			if (contentLength == INVALID_CONTENT_LENGTH) {
				Log.error("Invalid \""
						+ CONTENT_LENGTH
						+ "\" header.  Rest of input swallowed.");
				contentLength = null;
			}
	
			if (contentLength == null) {
				Log.debug("reading body to EOF");
			} else {
				Log.debug("reading body for "
						+ contentLength
						+ " bytes");
			}
	
			Reader bodyReader =
				contentLength == null
					? (Reader) in
					: (Reader) new FiniteReader(in, contentLength.longValue());
	
			char[] buffer = new char[65536];
			StringBuffer sb = new StringBuffer();
			int length;
	
			while (-1 != (length = bodyReader.read(buffer)))
				sb.append(buffer, 0, length);
	
			if (bodyReader != in)
				bodyReader.close();
	
			String bodyText = sb.toString();
	
			if (bodyField == null) {
				if (Log.willDebug())
					Log.debug("Ignoring body: " + Util.clean(bodyText));
			} else {
				add(bodyField, bodyText);
				if (Log.willDebug())
					Log.debug("==> header: \""
							+ Util.clean(bodyField)
							+ "\", Value: \""
							+ Util.clean(bodyText));
			}
		}
	
		long endTime = System.currentTimeMillis();
		Log.debug("Time to read "
				+ command.getName()
				+ ": "
				+ (endTime - startTime)
				+ " ms. Body took "
				+ (endTime - beforeBody)
				+ " ms");
	
	}

  /**
   * Return a List of RangeDef range definitions.
   *
   * @return a non-null List containing zero or more RangeDef objects.
   */

  public List getRanges()
  {
    List ranges = new LinkedList();

    RangeDef range = null;

    for (Iterator hdr = headers.iterator(); hdr.hasNext(); )
    {
      StringPair header = (StringPair)hdr.next();
      String key = header.getKeyString().trim();
      String value = header.getValueString().trim();

      if (key.equals(RANGE_FIELD))
      {
        range = new RangeDef(value);
        ranges.add(range);
      }
      else if (range != null)
      {
        // We have a range name: We can process attributes.

        if (key.equals(RANGE_FROM))
          range.from = value;
        else if (key.equals(RANGE_TO))
          range.to = value;
      }
    }
    return ranges;
  }

  /**
   * Return a List of FieldDef field definitions.  This is the list
   * from the Application default merged with any new ones defined in
   * the Transmission.
   *
   * @return a non-null List containing zero or more FieldDef objects.
   */

  public List getFields()
  {
    List fields = new LinkedList();
    if (application != null)
      fields.addAll(application.getAllFieldDefs());

    FieldDef field = null;

    for (Iterator hdr = headers.iterator(); hdr.hasNext(); ) {
      StringPair header = (StringPair)hdr.next();
      String key = header.getKeyString().trim();
      String value = header.getValueString().trim();

      if (key.equals(FIELD_NAME)) {
        if (value.length() == 0)
          fields.clear();  // Blank field name means clear out defaults.

        else {
          // The transmission is defining a field.  Is it new or an override?

          field = new FieldDef(value);
          int position = fields.indexOf(field);

          if (position == -1)
            fields.add(field);  // This is a new field.
          else                  // This is an override of a default field.
            field = (FieldDef) fields.get(position);
        }
      } else if (field != null) {
        // We have a field name: We can process attributes.

        if (key.equals(FIELD_TYPE)) {
          field.token = value.equalsIgnoreCase(TEXT); //will return either true or false
          field.date  = value.equalsIgnoreCase(DATE);
        } else if (key.equals(FIELD_INDEXED)) {
          field.index = value.equalsIgnoreCase(YES);
        } else if (key.equals(FIELD_STORED))  {
          field.store = value.equalsIgnoreCase(YES);
        }
      }
    } //end for each header.
    return fields;
  }



  /**
   * Get the Serial associated with this Transmission.
   *
   * @return the value associated with the first "Serial" header (if
   * any).
   */

  public String getSerial()
  {
    return get(SERIAL, NO_APP);
  }

  /**
   * Set the Serial associated with this transmission.
   *
   * @param serial The single value to replace all other "Serial"
   * headers.  If null, all Serial headers are deleted.
   */

  public void setSerial(String serial)
  {
    removeAll(SERIAL);
    if (serial != null)
      add(SERIAL, serial);
  }
  
	/**
	 * Is this transmission using XML as the protocol?
	 *
	 * @return whether XML is used or not.
	 */

	public boolean isXML()
	{
	  return this.XML;
	}

	/**
	 * Set the XML usage status of this transmission
	 *
	 * @param XML usage
	 */

	public void setXML(boolean isXML)
	{
	  this.XML = isXML;
	}
	
  /**
   * Return the result associated with this transmission.
   */
  public String getResult() {
      return this.result.toString();
  }




  private Document writeXML() {
  	Document document = DocumentHelper.createDocument();

  	 if (command.getName().equals("QUERY-RESPONSE")) {
	  	Element root = document.addElement("LuceneQueryResponse");
	
	  	String _Serial = get("Serial");
	  	if (_Serial != null) {
	  		 removeAll("Serial"); //so it won't show up as a field
	  		 //root.addElement("Serial").addCDATA(_Serial);
	  		root.addElement("Serial").addText(_Serial);
	  	}
	
	  	String _error = get("Error");
	  	if (_error != null) {
	  		root.addElement("Error").addCDATA(_error);
	  	} else {
	
	  		String _Count = get("Count");
	  		if (_Count != null) {
	  			 removeAll("Count");
	  			 root.addElement("Count").addText(_Count);
	  		}
	
	  		Iterator it = headers.iterator();
	  		if (it.hasNext()) {
	  			Element results = root.addElement("Results");
	
	  			StringPair header = (StringPair)it.next();
	  			do {
	  				Element result = null;
	  				String key   = header.getKeyString();
	  				String value = header.getValueString();
	
	  				do {
	  					//Log.debug("Key:" + key + ": \tValue:" + value);
	
	  					if (key.equalsIgnoreCase("I"))  {
	  						result = results.addElement("Result");
	  						result.addAttribute("counter", value);
	  					} else if (key.equalsIgnoreCase("RANK"))  {
	  						result.addAttribute("rank", value);
	  					} else {
	  						result.addElement("Field").addAttribute("name", key).addCDATA(value);
	  					}
	  					if (it.hasNext()) {
	  						header = (StringPair)it.next();
	  						key   = header.getKeyString();
	  						value = header.getValueString();
	  					} else {
	  						break;
	  					}
	
	  				} while (header != null && !key.equalsIgnoreCase("I"));
	
	  			} while ( it.hasNext() );
	
	
	  		} //end if has results
	  	} //end if no error.
  	 } else if (command.getName().equals("INDEX-RESPONSE")) {
 	   Element root = document.addElement("LuceneIndexResponse");

 	   String _Serial = get("Serial");
 	   if (_Serial != null) {
 			removeAll("Serial"); //so it won't show up as a field
 			root.addElement("Serial").addCDATA(_Serial);
 	   }

 	   String _error = get("Error");
 	   if (_error != null) {
 		   root.addElement("Error").addCDATA(_error);
 	   } else {

 		   Iterator it = headers.iterator();
 		   if (it.hasNext()) {
 			   StringPair header;
 			   do {
 			   	   header = (StringPair)it.next();
 				   String key   = header.getKeyString();
 				   String value = header.getValueString();

				   root.addElement(key).addCDATA(value);

 			   } while ( it.hasNext() );

 		   } //end of headers
 	   } //end if no error.

 	} else { //do default xml output
 		//do header to xml element mapping for simple responses.
 		String name = command.getName();
 		String element = null;
  	 	if (name.equals("ERROR")) 				element = "LuceneErrorResponse";
		if (name.equals("UNINDEX-RESPONSE")) 	element = "LuceneUnIndexResponse";
		if (name.equals("COMMAND")) 			element = "LuceneUtilityResponse";
		
		Element root;  	 	 
 		if (element != null) {
 			 root = document.addElement(element);
 		} else {
 			root = document.addElement("LuceneResponse");
 			root.addAttribute("type", command.getName());
 		}

 		 for (Iterator it = headers.iterator(); it.hasNext(); ) {
 			 StringPair header = (StringPair)it.next();
 			 root.addElement(header.getKeyString()).addCDATA(header.getValueString());
 		 }
 	}
 	return document;
  	
  }  	
  
}