/** *********************************************************************
 * Copyright (C) 2003 Catalyst IT                                       *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to:                           *
 *   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    *
 *   Boston, MA  02111-1307  USA                                        *
 ************************************************************************/
package nz.net.catalyst.lucene.server;

import java.io.File;
import java.io.IOException;

import nz.net.catalyst.Log;
import nz.net.catalyst.ELog;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;


/**
 * Backup a Lucene Index directory.
 * 
 * More information can be found in the Lucene Documentation.
 * 
 * This method will copy an application's index into another directory.
 * 
 */

public class Backup implements IPackage, Constants {
  private final Transmission input;
  private Application application;

  Backup(Transmission transmission) {
    input = transmission;
  }

  /**
   * 
   * Summary:
   * 
   * Create new directory from "Target" parameter.
   * 
   * Merge specified application's index into above directory. 
   * 
   * @return
   */

  Transmission execute() {
    long queryStart = System.currentTimeMillis();
    Transmission response = new Transmission(ECommand.CONTROL_RESPONSE);
    response.setSerial(input.getSerial());

    String appName = input.get(APPLICATION, NO_APP);

	Log.info("About to Backup Index");

    try {
      application = Application.getAppOrDefault(appName);
    } catch (ApplicationMissingException e) {
      return error(e.getMessage());
    }

    input.setApplication(application);
    Analyzer analyzer   = Application.getAnalyzer(input); // Don't care which one
    File luceneStoreDir = Application.getIndexDirectory(application);

	if (luceneStoreDir == null) {
		return error("Application's Index Directory could not be retreived.");
	}
    Log.info(" Source Directory: " + luceneStoreDir );
    
    Directory luceneDir;
	  try {
	    luceneDir = FSDirectory.getDirectory(luceneStoreDir, false);
	  } catch (IOException e) {
		return error(e.getMessage());
	  }

	//get target directory.
	
	//get from request
	String newDir = input.get("BackUpTarget", USE_APP);

	if (newDir != null) {
		Log.debug(" using target from request");
	} else {
		//get from application settings
		newDir = input.get("Lucene-Backup-Directory", USE_APP);
		if (newDir != null) {
			Log.debug(" using target from application properties");
		} else {
			//get from system
			newDir = System.getProperty(PACKAGE + "Lucene-Backup-Directory");
			
			if (newDir != null) {
				Log.debug(" using target from System.config");
			} else {
				//default to indexpath + "_backup"
				Log.debug(" No target Directory supplied. Using default");
				newDir = luceneStoreDir + "_backup";
			}
		}
	}

	Log.info(" Target Directory:" + newDir);
	
	File newFile = new File(newDir);

	//log some status.	
	Log.info(" Target directory exists:      " + newFile.exists());
	Log.info(" Target directory writable:    " + newFile.canWrite());
	Log.info(" Target directory isDirectory: " + newFile.isDirectory());

	//some informational messages
	if (! newFile.exists()) {
		Log.info(" Attempting to create target directory.");
		boolean success = newFile.mkdirs();
		Log.info(" Successfull creation of directory: " + success);
	} else {
		Log.warn("Index Backup (for application: " 
											+ appName 
											+ ") will overwrite target directory: " 
											+ newDir);
	}

	if (!newFile.isDirectory()) {
		return error("Backup target is NOT a directory: " + newDir);
	}
	if (!newFile.canWrite()) {
		return error("Application cannot write to target directory: " + newDir);
	}
	
	long beforeOpen = System.currentTimeMillis();
	long afterWriterOpen;
	long afterBackup;
	boolean succeed = false;

	IndexWriter newIndex;  
	
	try {
		newIndex  = new IndexWriter(newDir, analyzer, true);
		Log.debug(" Created new index in target directory");
	} catch (IOException e) {
	  return error(e.getMessage());
	}
	
	Log.debug("About to backup index in: " + luceneStoreDir);
	afterWriterOpen = System.currentTimeMillis();

	//array of directories
	Directory[] dirs = {luceneDir};
	
    try {
	  //Log.debug("Length of dirs: " + dirs.length);
	  Log.debug("about to addIndexes");
	  newIndex.addIndexes(dirs);
	  Log.info("Successfully Backed Up Index");

	  afterBackup = System.currentTimeMillis();
      succeed = true;
	} catch (IOException e) {
	  String message = "Error while backing up " + luceneStoreDir +
		": " +e.toString();
	  Log.log(ELog.ERROR, message);
	  return error(message);
    } catch (ArrayIndexOutOfBoundsException e) {
      String message = "Error while backing up (probable index corruption) " 
      					+ luceneStoreDir 
      					+ ": " 
      					+ e.toString();
      Log.log(ELog.ERROR, message);
      return error(message);
    } finally {
      if (newIndex != null) {
        try
        {
          // We must always free the IndexWriter for other users.
          newIndex.close();
        }
        catch (Throwable e)
        {
          String message = "Error while closing IndexWriter: " + e.toString();
          Log.log(ELog.ERROR, message);
          if (succeed)
            return error(message);
        }
      }
    }
    long afterWriterClose = System.currentTimeMillis();
    response.add("Open",  String.valueOf(afterWriterOpen - beforeOpen));
    response.add("Backup", String.valueOf(afterBackup - afterWriterOpen));
    response.add("Close",  String.valueOf(afterWriterClose - afterBackup));
    return response;
}

  /**
   * Build an error response for sending back to the client.
   *
   * @param message The text of the error message
   * @return An INDEX-RESPONSE Transmission
   */

  private Transmission error(String message)
  {
    Transmission response = new Transmission(ECommand.CONTROL_RESPONSE);
    response.setSerial(input.getSerial());
    response.add(ERROR, message);
    Log.error("ERROR during BACKUP: " + message);
    return response;
  }
}
