# Arch Perl library, Copyright (C) 2005 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package AXP::Command::triggers::list;
use base 'AXP::Command::triggers';

sub optusage {
	"[options] [limit ...]"
}

sub infoline {
	"list defined triggers"
}

sub options {
	(
		show_vars   => { sh => 'v', desc => "show associated user variables" },
		show_state  => { sh => 's', desc => "show associated state" },
		show_action => { sh => 'a', desc => "show associated action" },
		all         => { sh => 'A', desc => "show all associated data" },
		no_nums     => { sh => 'N', desc => "do not show numbers" },
		no_comments => { sh => 'C', desc => "do not show empty and comment lines" },
		__PACKAGE__->limits_option,
	)
}

sub helptext {
	q{
		Print all defined (or chosen) triggers and associated data.

		By default only the limit (i.e. arch name) is shown.  Optionally,
		for every trigger also show its state (last processed revisions),
		user variables (pairs KEY=VALUE), and action script.
	}
}

sub _tab_pad ($;$) {
	my $str = shift || "(none)";
	my $hide_comments = shift || 0;

	$str = [ map { "$_=$str->{$_}" } sort keys %$str ] if ref($str) eq 'HASH';
	$str = join("", map { "$_\n" } @$str) if ref($str) eq 'ARRAY';
	$str =~ s/^(#.*|$)\n//mg if $hide_comments;
	$str =~ s/^/\t/mg;
	$str =~ s/\s+$//s;

	return "$str\n\n";
}

sub execute {
	my $self = shift;
	my %opt = %{$self->{options}};

	my $show_nums = !$opt{no_nums} && -t STDOUT;
	$opt{show_vars} = $opt{show_state} = $opt{show_action} = 1 if $opt{all};

	my @names = @ARGV;
	@names = @{$self->all_trigger_names} unless $opt{limits} || @names;

	foreach my $name (@names) {
		my $trigger = $self->load_trigger($name);
		$name = $trigger->{name};

		my $slave_suffix = $trigger->{is_slave}? " (slave)": "";
		print "$trigger->{num}. " if $show_nums;
		print "$name$slave_suffix\n";
		print _tab_pad($trigger->{state}) if $opt{show_state};
		print _tab_pad($trigger->{vars})  if $opt{show_vars};
		print _tab_pad($trigger->{action}, $opt{no_comments}) if $opt{show_action};
	}
}

1;
