/*
 * operator for MIDI device
 *
 * Virtual Tiny Keyboard
 *
 * Copyright (c) 1997-1999 by Takashi Iwai
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "vkb.h"
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#ifdef __FreeBSD__
#  include <machine/soundcard.h>
#elif defined(linux)
#  include <linux/soundcard.h>
#endif


/*
 * functions
 */
static int seq_open(Tcl_Interp *ip, void **private_return);
static void seq_close(void *private);
static void note_on(void *private, int note, int vel);
static void note_off(void *private, int note, int vel);
static void program(void *private, int bank, int preset);
static void control(void *private, int type, int val);
static void bender(void *private, int bend);
#define midiputc(byte) SEQ_MIDIOUT(my_dev, byte)


/*
 * definition of device information
 */

static vkb_oper_t midi_oper = {
	seq_open,
	seq_close,
	program,
	note_on,
	note_off,
	control,
	bender,
	NULL, /* chorus_mode */
	NULL, /* reverb_mode */
};

vkb_devinfo_t midi_devinfo = {
	"midi",
	"MIDI device",
	&midi_oper,
};


/*
 * for OSS interface
 */

SEQ_USE_EXTBUF();
extern int seqfd;
static int my_dev = 0;
static int chan_no = 0;

/*
 */

static int
seq_open(Tcl_Interp *ip, void **private_return)
{
	int nrmidis;
	char *var;

	if (vkb_open_oss(ip) < 0)
		return 0;

	if (ioctl(seqfd, SNDCTL_SEQ_NRMIDIS, &nrmidis) < 0 || nrmidis <= 0) {
		vkb_error(ip, "there is no soundcard installed");
		close(seqfd);
		return 0;
	}

	if ((var = Tcl_GetVar(ip, "mididev", 0)) != NULL) {
		my_dev = atoi(var);
	}

	if (my_dev < 0 || my_dev >= nrmidis) {
		vkb_error(ip, "MIDI device number is out of range %d: using device #0\n", my_dev);
		my_dev = 0;
	}

	ioctl(seqfd, SNDCTL_SEQ_TESTMIDI, &my_dev);

	return 1;
}

static void
seq_close(void *private)
{
	close(seqfd);
}

/*
 */

static void
note_on(void *private, int note, int vel)
{
	midiputc(MIDI_NOTEON | chan_no);
	midiputc(note);
	midiputc(vel);
	SEQ_DUMPBUF();
}

static void
note_off(void *private, int note, int vel)
{
	midiputc(MIDI_NOTEOFF | chan_no);
	midiputc(note);
	midiputc(vel);
	SEQ_DUMPBUF();
}

static void
program(void *private, int bank, int preset)
{
	if (bank == 128)
		chan_no = 9;
	else {
		chan_no = 0;
		midiputc(MIDI_CTL_CHANGE | chan_no);
		midiputc(CTL_BANK_SELECT);
		midiputc(bank);
	}
	midiputc(MIDI_PGM_CHANGE | chan_no);
	midiputc(preset);
	// no dumpbuf here
}

static void
control(void *private, int type, int val)
{
	midiputc(MIDI_CTL_CHANGE | chan_no);
	midiputc(type);
	midiputc(val);
	SEQ_DUMPBUF();
}

static void
bender(void *private, int bend)
{
	bend += 8192;
	midiputc(MIDI_PITCH_BEND | chan_no);
	midiputc(bend & 0x7f);
	midiputc((bend >> 7) & 0x7f)
	SEQ_DUMPBUF();
}

